/*
 * Parse RedBoot-style Flash Image System (FIS) tables and
 * produce a Linux partition array to match.
 *
 * Copyright © 2001      Red Hat UK Limited
 * Copyright © 2001-2010 David Woodhouse <dwmw2@infradead.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/vmalloc.h>

#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/module.h>
#ifdef MTD_SLPFLASH_NAND
#include "devices/mdrv_nand_partition.h"
#endif

#ifdef CONFIG_TP_TAPO_SPMINIOS
#ifdef MTD_SLPFLASH_NAND	/* nandflash可擦除块为128K */
  #define SECTOR_VERIFY_SIZE    0x20000
#else
  #define SECTOR_VERIFY_SIZE    0x10000
#endif
#endif

/* tp header part defination , by Wu Yin, 27Mar14 */
/* modify for cloud router tp header, by Wu Yin, 28Jul14 */
#include "../../arch/powerpc/boot/redboot.h"

struct fis_image_desc {
    unsigned char name[16];      // Null terminated name
    uint32_t	  flash_base;    // Address within FLASH of image
    uint32_t	  mem_base;      // Address in memory where it executes
    uint32_t	  size;          // Length of image
    uint32_t	  entry_point;   // Execution entry point
    uint32_t	  data_length;   // Length of actual data
    unsigned char _pad[256-(16+7*sizeof(uint32_t))];
    uint32_t	  desc_cksum;    // Checksum over image descriptor
    uint32_t	  file_cksum;    // Checksum over image data
};

/********add by zhangwu 24May07***************/
struct fis_image_desc flash_sectors[] = {
    {
       FACTORY_BOOT_MTD_NAME,
    },
    {
		FACTORY_INFO_MTD_NAME,
	},
	{
		RADIO_MTD_NAME,
	},
	{
		UC_MTD_NAME,
	},
	{
		NORMAL_BOOT_MTD_NAME,
	},
	{
		KERNEL_MTD_NAME,
	},
	{
		ROMFS_MTD_NAME,
	},
#ifndef CONFIG_MTD_ROOTFS_SPLIT
	{
		JFFS2_MTD_NAME,
	},
#endif
#ifdef CONFIG_USER_DATA_PARTITION
	{
		USER_RECORD_MTD_NAME,
	},
#endif
#ifdef CONFIG_TP_TAPO_SPMINIOS
	{
		VERIFY_MTD_NAME,
	},
#endif
#ifdef CONFIG_TAPO_NAND_UPGRADE
	{
		UPBOOT_MTD_NAME,
	},
#endif
	{
		FIRMWARE_MTD_NAME,
	},
	{
       UITRON_MTD_NAME,
    },
    {
       UITRON_EXT_MTD_NAME,
    },
    {
       LD_MTD_NAME,
    },

    {
        ISP_MTD_NAME,
    },
	{
		AF_MTD_NAME,
	},
	{
		0xff,
	}
};
struct fis_image_desc extra_flash_sectors[] = {
	{
		TP_HEADER_MTD_NAME,
	},
	{
		KERNEL_AND_ROMFS_MTD_NAME,
	},
#ifdef CONFIG_MTD_ROOTFS_SPLIT
	{
		JFFS2_MTD_NAME,
	},
#endif

	{
		0xff,
	}
};

struct fis_list {
	struct fis_image_desc *img;
	struct fis_list *next;
};

static int directory = CONFIG_MTD_REDBOOT_DIRECTORY_BLOCK;
module_param(directory, int, 0);

static inline int redboot_checksum(struct fis_image_desc *img)
{
	/* RedBoot doesn't actually write the desc_cksum field yet AFAICT */
	return 1;
}

int nand_mtd_read(struct mtd_info *mtd,
            loff_t from, size_t len, size_t *retlen, u_char *buf);

#ifdef ENCRYPT_ROOTFS_HEADER
#include "rijndael-alg-fst.h"
#include "rijndael-api-fst.h"

#define AES_KEY_LEN		16
#define ENCRYPT_ROOTFS_HEADER_SIZE (512)

int decrypt_rootfs_header_flg = 0;

static unsigned char AES_CFB1_key[] = CONFIG_ENCRYPT_ROOTFS_KEY;

static unsigned char AES_CFB1_iv[] =
{
	0x55, 0xAA, 0xDE, 0xAD, 0xC0, 0xDE, 'L', 'I',
	'N', 'U', 'X', 'E', 'x', 'T', 0xAA, 0x55,
};

long long rootfs_start = 0;
unsigned char decrypt_rootfs_header_buffer[ENCRYPT_ROOTFS_HEADER_SIZE] = {0};

static int _AES_cfb1_decrypt_rijndael(const unsigned char *key, int key_len, const unsigned char *iv, const unsigned char *in, int in_len, unsigned char *out)
{
	keyInstance keyInst;
	cipherInstance cipherInst;

	if(makeKey(&keyInst, DIR_DECRYPT, key_len << 3 /* byte to bit */, key) < 0){
		printk("rijndael decrypt makeKey fail\n");
		return -1;
	}
	if(cipherInit(&cipherInst, MODE_CFB1, iv) < 0){
		printk("rijndael cipherInit makeKey fail\n");
		return -1;
	}
	return blockDecrypt(&cipherInst, &keyInst, in, in_len, out);
}

static int decrypt_rootfs_header(struct mtd_info *master, int offset)
{
	int ret = 0;
	size_t retlen = 0;
	unsigned char *plain = (unsigned char *)kmalloc(ENCRYPT_ROOTFS_HEADER_SIZE, GFP_KERNEL);

	rootfs_start = offset;
	if(!plain){
		printk("decrypt_rootfs_header kmalloc fail\n");
		goto exit;
	}
#ifdef MTD_SLPFLASH_NAND
		nand_mtd_read(master, (loff_t)offset, (size_t)ENCRYPT_ROOTFS_HEADER_SIZE, &retlen, (void *)plain);
#else
		master->_read(master, (loff_t)offset, (size_t)ENCRYPT_ROOTFS_HEADER_SIZE, &retlen, (void *)plain);
#endif

	if(retlen != ENCRYPT_ROOTFS_HEADER_SIZE){
		printk("decrypt_rootfs_header read fail\n");
		goto exit;
	}

	ret = _AES_cfb1_decrypt_rijndael(AES_CFB1_key, AES_KEY_LEN, AES_CFB1_iv, plain, ENCRYPT_ROOTFS_HEADER_SIZE<<3, decrypt_rootfs_header_buffer);
	if(ret != (ENCRYPT_ROOTFS_HEADER_SIZE<<3)){
		printk("decrypt_rootfs_header decrypt fail ret:%d\n",ret);
		goto exit;
	}

	printk("decrypt_rootfs_header done\n");
	decrypt_rootfs_header_flg = 1;
exit:
	if(plain){
		kfree(plain);
	}
	return ret;
}
#endif

static int parse_redboot_partitions(struct mtd_info *master,
				    const struct mtd_partition **pparts,
                             struct mtd_part_parser_data * fis_orig)
{
	int nrparts = 0;
	struct fis_image_desc *buf;
	struct mtd_partition *parts;
	struct fis_list *fl = NULL, *tmp_fl;
	int ret, i;
	size_t retlen;
	char *names;
	char *nullname;
	int namelen = 0;
	int nulllen = 0;
	int numslots;
#ifdef ENCRYPT_ROOTFS_HEADER
	int rootfs_base = 0;
#endif
#ifdef CONFIG_MTD_REDBOOT_PARTS_UNALLOCATED
	static char nullstring[] = "unallocated";
#endif

	unsigned long fis_origin = (unsigned long)fis_orig;

	TP_HEADER hdr;
	/* Read mtd partion from flash hdr, add by lsz 30Nov07 */
	/* Modify for SLP, by wuyin 11Feb14 */
#ifdef MTD_SLPFLASH_NAND
    printk("SLP flash nand read\n");
    ret = nand_mtd_read(master, MTD_REDBOOT_TP_HEADER_ADDRESS, sizeof(hdr), &retlen, (void *)&hdr);
#else
    printk("SLP flash nor read\n");
    ret = master->_read(master, MTD_REDBOOT_TP_HEADER_ADDRESS, sizeof(hdr), &retlen, (void *)&hdr);
#endif
    printk("MTD_REDBOOT_TP_HEADER_ADDRESS:0x%x\n",MTD_REDBOOT_TP_HEADER_ADDRESS);
	for (i = 0; i < sizeof(flash_sectors)/sizeof(struct fis_image_desc); i ++)
	{
        if (strcmp(flash_sectors[i].name, LD_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.ldOffset);
			flash_sectors[i].size = ntohl(hdr.ldLen);
		}
		if (strcmp(flash_sectors[i].name, UITRON_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.uitronOffset);
			flash_sectors[i].size = ntohl(hdr.uitronLen);
		}
		if (strcmp(flash_sectors[i].name, UITRON_EXT_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.uitronExtOffset);
			flash_sectors[i].size = ntohl(hdr.uitronExtLen);
		}
		if (strcmp(flash_sectors[i].name, FACTORY_BOOT_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.factoryBootOffset);
			flash_sectors[i].size = ntohl(hdr.factoryBootLen);
		}
		if (strcmp(flash_sectors[i].name, FACTORY_INFO_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.factoryInfoOffset);
			flash_sectors[i].size = ntohl(hdr.factoryInfoLen);
		}
		if (strcmp(flash_sectors[i].name, NORMAL_BOOT_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.bootloaderOffset);
			flash_sectors[i].size = ntohl(hdr.bootloaderLen);
		}
		if (strcmp(flash_sectors[i].name, KERNEL_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.kernelOffset);
			flash_sectors[i].size = ntohl(hdr.kernelLen);
		}
#ifdef CONFIG_MTD_ROOTFS_SPLIT
		if (strcmp(flash_sectors[i].name, ROMFS_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.romFsOffset);
			flash_sectors[i].size = ntohl(hdr.romFsLen) + ntohl(hdr.jffs2FsLen);
#ifdef ENCRYPT_ROOTFS_HEADER
			rootfs_base = flash_sectors[i].flash_base;
#endif
		}
#else
		if (strcmp(flash_sectors[i].name, ROMFS_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.romFsOffset);
			flash_sectors[i].size = ntohl(hdr.romFsLen);
#ifdef ENCRYPT_ROOTFS_HEADER
			rootfs_base = flash_sectors[i].flash_base;
#endif
		}
		if (strcmp(flash_sectors[i].name, JFFS2_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.jffs2FsOffset);
			flash_sectors[i].size = ntohl(hdr.jffs2FsLen);
		}
#endif
#ifdef CONFIG_USER_DATA_PARTITION
		if (strcmp(flash_sectors[i].name, USER_RECORD_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.jffs2FsOffset) + ntohl(hdr.jffs2FsLen);
			flash_sectors[i].size = MTD_USER_RECORD_PARTITION_LEN;
		}
#endif
#ifdef CONFIG_TP_TAPO_SPMINIOS
		if (strcmp(flash_sectors[i].name, VERIFY_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.jffs2FsOffset) + ntohl(hdr.jffs2FsLen)
#ifdef CONFIG_USER_DATA_PARTITION
				+ MTD_USER_RECORD_PARTITION_LEN
#endif
			;
			flash_sectors[i].size = SECTOR_VERIFY_SIZE;
		}
#endif
#ifdef CONFIG_TAPO_NAND_UPGRADE
		if (strcmp(flash_sectors[i].name, UPBOOT_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.upbootOffset) ;
			flash_sectors[i].size = ntohl(hdr.upbootLen);
		}
#endif
		if (strcmp(flash_sectors[i].name, UC_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.ucOffset);
			flash_sectors[i].size = ntohl(hdr.ucLen);
		}

		if (strcmp(flash_sectors[i].name, AF_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.factoryInfoOffset) + ntohl(hdr.factoryInfoLen);
			flash_sectors[i].size = ntohl(hdr.radioOffset) - flash_sectors[i].flash_base;
		}

		if (strcmp(flash_sectors[i].name, RADIO_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.radioOffset);
			flash_sectors[i].size = ntohl(hdr.radioLen);
		}

        if (strcmp(flash_sectors[i].name, ISP_MTD_NAME) == 0)
		{
			flash_sectors[i].flash_base = ntohl(hdr.ispOffset);
			flash_sectors[i].size = ntohl(hdr.ispLen);
		}
		if (strcmp(flash_sectors[i].name, FIRMWARE_MTD_NAME) == 0)
		{
			/* QSDK AP135 has "firmware" as the last mtd partition(seems for upgrade),
			add it here since basing on QSDK */
			flash_sectors[i].flash_base = ntohl(hdr.tpHeaderOffset);
			flash_sectors[i].size = ntohl(hdr.tpHeaderLen) + ntohl(hdr.kernelLen) + ntohl(hdr.romFsLen) + ntohl(hdr.jffs2FsLen);
		}
	}
		
	for (i = 0; i < sizeof(extra_flash_sectors)/sizeof(struct fis_image_desc); i++)
	{
		if (strcmp(extra_flash_sectors[i].name, TP_HEADER_MTD_NAME) == 0)
		{
			extra_flash_sectors[i].flash_base = ntohl(hdr.tpHeaderOffset);
			extra_flash_sectors[i].size = ntohl(hdr.tpHeaderLen);
		}
		else if (strcmp(extra_flash_sectors[i].name, KERNEL_AND_ROMFS_MTD_NAME) == 0)
		{
			extra_flash_sectors[i].flash_base = ntohl(hdr.tpHeaderOffset);
			extra_flash_sectors[i].size = ntohl(hdr.tpHeaderLen) + ntohl(hdr.kernelLen) + ntohl(hdr.romFsLen);/* include TP header */
		}
#ifdef CONFIG_MTD_ROOTFS_SPLIT
		else if (strcmp(extra_flash_sectors[i].name, JFFS2_MTD_NAME) == 0)
		{
			extra_flash_sectors[i].flash_base = ntohl(hdr.jffs2FsOffset);
			extra_flash_sectors[i].size = ntohl(hdr.jffs2FsLen);
		}
#endif
	}

#ifdef ENCRYPT_ROOTFS_HEADER
	decrypt_rootfs_header(master, rootfs_base);
#endif

	/************add by zhangwu 24May07********************/
	buf = flash_sectors;
	printk(KERN_INFO "Searching for RedBoot partition table\n");
	/******************************************************/


	numslots = (master->erasesize / sizeof(struct fis_image_desc));


	for (i = 0; i < numslots; i++) {
		struct fis_list *new_fl, **prev;

		if (buf[i].name[0] == 0xff) {
            break;
		}
		if (!redboot_checksum(&buf[i]))
			break;

		new_fl = kmalloc(sizeof(struct fis_list), GFP_KERNEL);
		namelen += strlen(buf[i].name)+1;
		if (!new_fl) {
			ret = -ENOMEM;
			goto out;
		}
		new_fl->img = &buf[i];
                if (fis_origin) {
                        buf[i].flash_base -= fis_origin;
                } else {
                        buf[i].flash_base &= master->size-1;
                }

		/* I'm sure the JFFS2 code has done me permanent damage.
		 * I now think the following is _normal_
		 */
		prev = &fl;
		while(*prev)
			prev = &(*prev)->next;
		new_fl->next = NULL;
		*prev = new_fl;

		nrparts++;
	}
#ifdef CONFIG_MTD_REDBOOT_PARTS_UNALLOCATED
	if (fl->img->flash_base) {
		nrparts++;
		nulllen = sizeof(nullstring);
	}

	for (tmp_fl = fl; tmp_fl->next; tmp_fl = tmp_fl->next) {
		if (tmp_fl->img->flash_base + tmp_fl->img->size + master->erasesize <= tmp_fl->next->img->flash_base) {
			nrparts++;
			nulllen = sizeof(nullstring);
		}
	}
#endif
	parts = kzalloc(sizeof(*parts)*nrparts + nulllen + namelen, GFP_KERNEL);

	if (!parts) {
        printk("Malloc parts failed\n");
		ret = -ENOMEM;
		goto out;
	}
    memset(parts, 0, sizeof(*parts) * nrparts + nulllen + namelen);

	nullname = (char *)&parts[nrparts];
#ifdef CONFIG_MTD_REDBOOT_PARTS_UNALLOCATED
	if (nulllen > 0) {
		strcpy(nullname, nullstring);
	}
#endif
	names = nullname + nulllen;

	i=0;

#ifdef CONFIG_MTD_REDBOOT_PARTS_UNALLOCATED
	if (fl->img->flash_base) {
	       parts[0].name = nullname;
	       parts[0].size = fl->img->flash_base;
	       parts[0].offset = 0;
		i++;
	}
#endif
	for ( ; i<nrparts; i++) {
		parts[i].size = fl->img->size;
#ifdef MTD_SLPFLASH_NAND
        extern struct fis_image_desc_nand nand_flash_sectors[];
        if (!strcmp(fl->img->name, ROMFS_MTD_NAME))
        {
            int k = 0;
            for (k = 0; k < get_nand_flash_sectors_num();k++)
            {
                if (!strcmp(nand_flash_sectors[k].name, ROMFS_MTD_NAME))
                {
                    parts[i].size = nand_flash_sectors[k].size;
                    break;
                }
            }
        }
#endif
		parts[i].offset = fl->img->flash_base;
		parts[i].name = names;

		strcpy(names, fl->img->name);
#ifdef CONFIG_MTD_REDBOOT_PARTS_READONLY
		if (!memcmp(names, "RedBoot", 8) ||
				!memcmp(names, "RedBoot config", 15) ||
				!memcmp(names, "FIS directory", 14)) {
			parts[i].mask_flags = MTD_WRITEABLE;
		}

		/* make "boot" and "art" partition readonly, by wuyin, 11Feb14 */
		if (!memcmp(names, "boot", 5) ||
			!memcmp(names, "art", 4))
		{
			parts[i].mask_flags = MTD_WRITEABLE;
		}
#endif
		names += strlen(names)+1;

#ifdef CONFIG_MTD_REDBOOT_PARTS_UNALLOCATED
		if(fl->next && fl->img->flash_base + fl->img->size + master->erasesize <= fl->next->img->flash_base) {
			i++;
			parts[i].offset = parts[i-1].size + parts[i-1].offset;
			parts[i].size = fl->next->img->flash_base - parts[i].offset;
			parts[i].name = nullname;
		}
#endif
		tmp_fl = fl;
		fl = fl->next;
		kfree(tmp_fl);
	}
	ret = nrparts;
	*pparts = parts;
 out:
	while (fl) {
		struct fis_list *old = fl;
		fl = fl->next;
		kfree(old);
	}
	//vfree(buf);
	return ret;
}

static struct mtd_part_parser redboot_parser = {
	.owner = THIS_MODULE,
	.parse_fn = parse_redboot_partitions,
	.name = "RedBoot",
};

/* mtd parsers will request the module by parser name */
MODULE_ALIAS("RedBoot");

static int __init redboot_parser_init(void)
{
	register_mtd_parser(&redboot_parser);
	return 0;
}

static void __exit redboot_parser_exit(void)
{
	deregister_mtd_parser(&redboot_parser);
}

int get_info_from_mtd_partition_name(char* partition_name, int *offset, int *size)
{
	int i = 0;
	for (i = 0; i < sizeof(flash_sectors)/sizeof(struct fis_image_desc); i++)
	{
		if (strcmp(flash_sectors[i].name, partition_name) == 0)
		{
			*offset = flash_sectors[i].flash_base;
			*size = flash_sectors[i].size;
			return 0;
		}
	}
	for (i = 0; i < sizeof(extra_flash_sectors)/sizeof(struct fis_image_desc); i++)
	{
		if (strcmp(extra_flash_sectors[i].name, partition_name) == 0)
		{
			*offset = extra_flash_sectors[i].flash_base;
			*size = extra_flash_sectors[i].size;
			return 0;
		}
	}
	return -1;
}
EXPORT_SYMBOL(get_info_from_mtd_partition_name);

module_init(redboot_parser_init);
module_exit(redboot_parser_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("David Woodhouse <dwmw2@infradead.org>");
MODULE_DESCRIPTION("Parsing code for RedBoot Flash Image System (FIS) tables");
