/*
 * Copyright (c) International Business Machines Corp., 2006
 * Copyright (C) 2008 Nokia Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Generating UBI images.
 *
 * Authors: Oliver Lohmann
 *          Artem Bityutskiy
 */

#define PROGRAM_NAME "libubigen"

#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <string.h>

#include <mtd/ubi-media.h>
#include <mtd_swab.h>
#include <libubigen.h>
#include <crc32.h>
#include <rs_enc.h>
#include <BCH_bench.h>
#include <BCH_mdecoder.h>
#include "common.h"

void ubigen_info_init(struct ubigen_info *ui, struct ubi_args *args)
{
	if (!args->vid_hdr_offs) {
		args->vid_hdr_offs = UBI_EC_HDR_SIZE + args->subpage_size - 1;
		args->vid_hdr_offs /= args->subpage_size;
		args->vid_hdr_offs *= args->subpage_size;
	}

	ui->peb_size = args->peb_size;
	ui->min_io_size = args->min_io_size;
	ui->vid_hdr_offs = args->vid_hdr_offs;
	ui->data_offs = args->vid_hdr_offs + UBI_VID_HDR_SIZE + args->min_io_size - 1;
	ui->data_offs /= args->min_io_size;
	ui->data_offs *= args->min_io_size;
	ui->leb_size = args->peb_size - ui->data_offs;
	ui->ubi_ver = args->ubi_ver;
	ui->image_seq = args->image_seq;
    ui->ecc_format = args->ecc_format;
    ui->ecc_algorithm = args->ecc_algorithm;
	ui->oob_size = args->oob_size;

	ui->max_volumes = ui->leb_size / UBI_VTBL_RECORD_SIZE;
	if (ui->max_volumes > UBI_MAX_VOLUMES)
		ui->max_volumes = UBI_MAX_VOLUMES;
	ui->vtbl_size = ui->max_volumes * UBI_VTBL_RECORD_SIZE;

    switch (ui->min_io_size)
    {
        case 512:
			ui->page_shift = 9;
			break;
		case 2048:
			ui->page_shift = 11;
			break;
		case 4096:
			ui->page_shift = 12;
			break;
		case 8192:
			ui->page_shift = 13;
			break;
		default:
			errmsg("Wrong min I/O unit size: %d", ui->min_io_size);
    }

	switch (args->ecc_algorithm)
	{
	    case BCH_4_512:
			ui->ecc_len = 7;
			ui->main_sector_size = 512;
			break;
		case BCH_8_512:
			ui->ecc_len = 13;
			ui->main_sector_size = 512;
			break;
		case BCH_12_512:
			ui->ecc_len = 20;
			ui->main_sector_size = 512;
			break;
		case BCH_16_512:
			ui->ecc_len = 26;
			ui->main_sector_size = 512;
			break;
		case BCH_20_512:
			ui->ecc_len = 33;
			ui->main_sector_size = 512;
			break;
		case BCH_24_512:
			ui->ecc_len = 39;
			ui->main_sector_size = 512;
			break;
		case BCH_24_1024:
			ui->ecc_len = 42;
			ui->main_sector_size = 1024;
			break;
		case BCH_32_1024:
			ui->ecc_len = 56;
			ui->main_sector_size = 1024;
			break;
		case RS_4_512:
			ui->ecc_len = 10;
			ui->main_sector_size = 512;
			break;
		case BCH_4_512_V2:
			ui->ecc_len = 8;
			ui->main_sector_size = 512;
			break;
		case BCH_8_512_V2:
			ui->ecc_len = 14;
			ui->main_sector_size = 512;
			break;
		case BCH_12_512_V2:
			ui->ecc_len = 20;
			ui->main_sector_size = 512;
			break;
		case BCH_16_512_V2:
			ui->ecc_len = 26;
			ui->main_sector_size = 512;
			break;
		case BCH_20_512_V2:
			ui->ecc_len = 34;
			ui->main_sector_size = 512;
			break;
		case BCH_24_512_V2:
			ui->ecc_len = 40;
			ui->main_sector_size = 512;
			break;
		case BCH_24_1024_V2:
			ui->ecc_len = 42;
			ui->main_sector_size = 1024;
			break;
		case BCH_32_1024_V2:
			ui->ecc_len = 56;
			ui->main_sector_size = 1024;
			break;
		case BCH_40_1024_V2:
			ui->ecc_len = 70;
			ui->main_sector_size = 1024;
			break;
		default:
			errmsg("Wrong ECC algorithm choice: %d", ui->min_io_size);
	}

	ui->sector_num = ui->min_io_size / ui->main_sector_size;
	ui->oob_sector_size = (ui->oob_size / ui->sector_num) & 0xfffffffe;
	if (ui->oob_sector_size < ui->ecc_len + 1)
	{
	    sys_errmsg("Wrong ECC algorithm(%d) choice or wrong oob size(%d bytes), oob sector:%d",
			        ui->ecc_algorithm, ui->oob_size, ui->oob_sector_size);
	}
	ui->sector_size_oob = ui->main_sector_size + ui->oob_sector_size;
	ui->ecc_offs = ui->oob_sector_size - ui->ecc_len;
	ui->oob_remain = ui->oob_size - ui->oob_sector_size * ui->sector_num;
    ui->page_num = ui->peb_size / ui->min_io_size;
	ui->page_size_oob = ui->min_io_size + ui->oob_size;
	ui->peb_size_oob = ui->page_size_oob * ui->page_num;
	ui->leb_size_oob = (ui->leb_size >> ui->page_shift) * ui->page_size_oob;
	ui->vid_offs_oob = (ui->vid_hdr_offs >> ui->page_shift) * ui->page_size_oob;
	ui->data_offs_oob = ui->peb_size_oob - ui->leb_size_oob;
}

struct ubi_vtbl_record *ubigen_create_empty_vtbl(const struct ubigen_info *ui)
{
	struct ubi_vtbl_record *vtbl;
	int i;

	vtbl = calloc(1, ui->vtbl_size);
	if (!vtbl) {
		sys_errmsg("cannot allocate %d bytes of memory", ui->vtbl_size);
		return NULL;
	}

	for (i = 0; i < ui->max_volumes; i++) {
		uint32_t crc = mtd_crc32(UBI_CRC32_INIT, &vtbl[i],
				     UBI_VTBL_RECORD_SIZE_CRC);
		vtbl[i].crc = cpu_to_be32(crc);
	}

	return vtbl;
}

/* empty page check */
static int ubigen_empty_check(const struct ubigen_info *ui, unsigned char* inbuf,
                                      unsigned int len)
{
    unsigned int j = 0;
    
    while (inbuf[j++] == 0xFF)
    {
	    if (j == len)
	    {
		    j = len + 1;
		    break;
	    }
    }
    /* for empty page, no ecc calculation */
    if (j == (len + 1))
    {
	    //memset(pbuf + pagelen * i, 0xFF, pagelen);
	    // empty page
	    return 1;
    }

    // non-empty page
	return 0;
}
/* calculate ecc for one page data */
static int ubigen_ecc_calc(const struct ubigen_info *ui, unsigned char* outbuf,
	                       unsigned char* inbuf, unsigned int page_num)
{
    int rd;
    unsigned int k, ms_size, ss_size, pagelen;

    ms_size = ui->main_sector_size;
    ss_size = ui->oob_sector_size;
    pagelen = ui->page_size_oob;
	
    if (inbuf != NULL)
    {
        if (ubigen_empty_check(ui, (inbuf + ui->min_io_size * page_num), ui->min_io_size))
        {
            return 1;
        }

        if (ui->ecc_format == 1)
        {
            memcpy(outbuf + pagelen * page_num,
                   inbuf + ui->min_io_size * page_num, ui->min_io_size);
        }
    }

    for (k = 0; k < ui->sector_num; k++)
	{
	   if (ui->ecc_format == 1)
	   {
	       rd = ecc_cal(ui->ecc_algorithm, outbuf + pagelen * page_num + ms_size * k,
		   	            outbuf + pagelen * page_num + ui->min_io_size + ss_size * k, ui->ecc_offs,
	                    outbuf + pagelen * page_num + ui->min_io_size + ss_size * k + ui->ecc_offs, 0);
	       #if 0
	       if (rd != NO_ERROR)
	       {
	           sys_errmsg("cannot calculate ecc, error code:%d", rd);
	           goto out_free1;
	       }
	       #endif
	   }
	   else
	   {
	       if (inbuf != NULL)
	       {
	           memcpy(outbuf + pagelen * page_num + ui->sector_size_oob * k,
		              inbuf + ui->min_io_size * page_num + ms_size * k, ms_size);
	       }
	       //Calculate_ECC(outbuf + 528 * k, outbuf + 528 * k + 512);
	       rd = ecc_cal(ui->ecc_algorithm, outbuf + pagelen * page_num + ui->sector_size_oob * k,
	                    outbuf + pagelen * page_num + ui->sector_size_oob * k + ms_size, ui->ecc_offs,
	                    outbuf + pagelen * page_num + ui->sector_size_oob * k + ms_size + ui->ecc_offs, 0);
	       #if 0
	       if (rd != NO_ERROR)
	       {
	           sys_errmsg("cannot calculate ecc, error code:%d", rd);
	           goto out_free1;
	       }
	       #endif
	   }
	}

	return 0;
}

/* calculate ecc for data which size is less than one page */
static void ubigen_ecc_calc_tail(const struct ubigen_info *ui, unsigned char* outbuf,
	                       unsigned char* inbuf, unsigned int len)
{
    int rd, l;
    unsigned int k, page_cnt, pagelen, ms_size, ss_size;

	ms_size = ui->main_sector_size;
    ss_size = ui->oob_sector_size;
    pagelen = ui->page_size_oob;
	page_cnt = len >> ui->page_shift;
	l = len % ui->min_io_size;
	
	if (l)
	{
	  if (!ubigen_empty_check(ui, (inbuf + ui->min_io_size * page_cnt), l))
	  {
		  if (ui->ecc_format == 1)
		  {
			  memcpy(outbuf + pagelen * page_cnt,
							inbuf + ui->min_io_size * page_cnt, l);
		  }
		  for (k = 0; k < ui->sector_num; k++)
		  {
			  if (ui->ecc_format == 1)
			  {
				  rd = ecc_cal(ui->ecc_algorithm, outbuf + pagelen * page_cnt + ms_size * k,
							   outbuf + pagelen * page_cnt + ui->min_io_size + ss_size * k, ui->ecc_offs,
							   outbuf + pagelen * page_cnt + ui->min_io_size + ss_size * k + ui->ecc_offs, 0);
              #if 0
				  if (rd != NO_ERROR)
				  {
					  sys_errmsg("cannot calculate ecc, error code:%d", rd);
					  goto out_free1;
				  }
              #endif
			  }
			  else
			  {
				  if (l > ui->main_sector_size)
				  {
					  l -= ui->main_sector_size;
					  memcpy(outbuf + pagelen * page_cnt + ui->sector_size_oob * k,
                                     inbuf + ui->min_io_size * page_cnt + ui->main_sector_size * k,
                                     ui->main_sector_size);
				  }
				  else if (l >0 && l <= ui->main_sector_size)
				  {
                                     memcpy(outbuf + pagelen * page_cnt + ui->sector_size_oob * k,
                                     inbuf + ui->min_io_size * page_cnt + ui->main_sector_size * k, l);
					  l = 0;
				  }
				  else
				  {
				  }
				  //Calculate_ECC(pbuf + pagelen * page_cnt + 528 * k,
										 //pbuf + pagelen * page_cnt + 528 * k + 512);
				  rd = ecc_cal(ui->ecc_algorithm, outbuf + pagelen * page_cnt + ui->sector_size_oob * k,
							   outbuf + pagelen * page_cnt + ui->sector_size_oob * k + ui->main_sector_size, ui->ecc_offs,
							   outbuf + pagelen * page_cnt + ui->sector_size_oob * k + ui->main_sector_size + ui->ecc_offs, 0);
              #if 0
				  if (rd != NO_ERROR)
				  {
					  sys_errmsg("cannot calculate ecc, error code:%d", rd);
					  goto out_free1;
				  }
              #endif
			}
		  }
	  }
	}
}

/* calculate ecc for miu binary file of BFN */
static int ubigen_ecc_calc_miu(const struct ubigen_info *ui, unsigned char* outbuf,
	                       unsigned char* inbuf, unsigned int page_num)
{
    int rd;
    unsigned int k, ms_size, ss_size;

    ms_size = ui->main_sector_size;
    ss_size = ui->oob_sector_size;

	if (ubigen_empty_check(ui, (inbuf + 512 * page_num), 512))
	{
		return 1;
	}

    memcpy(outbuf + ui->page_size_oob * page_num,
           inbuf + 512 * page_num, 512);

    for (k = 0; k < ui->sector_num; k++)
	{
       rd = ecc_cal(ui->ecc_algorithm, outbuf + ui->page_size_oob * page_num + 528 * k,
                    outbuf + ui->page_size_oob * page_num + 528 * k + 512, ui->ecc_offs,
                    outbuf + ui->page_size_oob * page_num + 528 * k + 512 + ui->ecc_offs, 0);
       #if 0
       if (rd != NO_ERROR)
       {
           sys_errmsg("cannot calculate ecc, error code:%d", rd);
           goto out_free1;
       }
       #endif
	}

	return 0;
}

int ubigen_add_volume(const struct ubigen_info *ui,
		      const struct ubigen_vol_info *vi,
		      struct ubi_vtbl_record *vtbl)
{
	struct ubi_vtbl_record *vtbl_rec = &vtbl[vi->id];
	uint32_t page_cnt;

	if (vi->id >= ui->max_volumes) {
		errmsg("too high volume id %d, max. volumes is %d",
		       vi->id, ui->max_volumes);
		errno = EINVAL;
		return -1;
	}

	if (vi->alignment >= ui->leb_size) {
		errmsg("too large alignment %d, max is %d (LEB size)",
		       vi->alignment, ui->leb_size);
		errno = EINVAL;
		return -1;
	}

	memset(vtbl_rec, 0, sizeof(struct ubi_vtbl_record));
	page_cnt = (vi->bytes + ui->leb_size - 1) / ui->leb_size;
	vtbl_rec->reserved_pebs = cpu_to_be32(page_cnt);
	vtbl_rec->alignment = cpu_to_be32(vi->alignment);
	vtbl_rec->vol_type = vi->type;
	page_cnt = ui->leb_size % vi->alignment;
	vtbl_rec->data_pad = cpu_to_be32(page_cnt);
	vtbl_rec->flags = vi->flags;

	memcpy(vtbl_rec->name, vi->name, vi->name_len);
	vtbl_rec->name[vi->name_len] = '\0';
	vtbl_rec->name_len = cpu_to_be16(vi->name_len);

	page_cnt = mtd_crc32(UBI_CRC32_INIT, vtbl_rec, UBI_VTBL_RECORD_SIZE_CRC);
	vtbl_rec->crc =	 cpu_to_be32(page_cnt);
	return 0;
}

void ubigen_init_ec_hdr(const struct ubigen_info *ui,
		        struct ubi_ec_hdr *hdr, long long ec)
{
	uint32_t crc;

	memset(hdr, 0, sizeof(struct ubi_ec_hdr));

	hdr->magic = cpu_to_be32(UBI_EC_HDR_MAGIC);
	hdr->version = ui->ubi_ver;
	hdr->ec = cpu_to_be64(ec);
	hdr->vid_hdr_offset = cpu_to_be32(ui->vid_hdr_offs);
	hdr->data_offset = cpu_to_be32(ui->data_offs);
	hdr->image_seq = cpu_to_be32(ui->image_seq);

	crc = mtd_crc32(UBI_CRC32_INIT, hdr, UBI_EC_HDR_SIZE_CRC);
	hdr->hdr_crc = cpu_to_be32(crc);
}

void ubigen_init_vid_hdr(const struct ubigen_info *ui,
			 const struct ubigen_vol_info *vi,
			 struct ubi_vid_hdr *hdr, int lnum,
			 const void *data, int data_size)
{
	uint32_t crc;

	memset(hdr, 0, sizeof(struct ubi_vid_hdr));

	hdr->magic = cpu_to_be32(UBI_VID_HDR_MAGIC);
	hdr->version = ui->ubi_ver;
	hdr->vol_type = vi->type;
	hdr->vol_id = cpu_to_be32(vi->id);
	hdr->lnum = cpu_to_be32(lnum);
	hdr->data_pad = cpu_to_be32(vi->data_pad);
	hdr->compat = vi->compat;

	if (vi->type == UBI_VID_STATIC) {
		hdr->data_size = cpu_to_be32(data_size);
		hdr->used_ebs = cpu_to_be32(vi->used_ebs);
		crc = mtd_crc32(UBI_CRC32_INIT, data, data_size);
		hdr->data_crc = cpu_to_be32(crc);
	}

	crc = mtd_crc32(UBI_CRC32_INIT, hdr, UBI_VID_HDR_SIZE_CRC);
	hdr->hdr_crc = cpu_to_be32(crc);
}

int ubigen_write_volume(const struct ubigen_info *ui,
			const struct ubigen_vol_info *vi, long long ec,
			long long bytes, int in, int out)
{
    int rd;
	unsigned int i, page_cnt, lnum = 0, len = vi->usable_leb_size;
	unsigned char *inbuf, *outbuf, *pbuf;

	if (vi->id >= ui->max_volumes) {
		errmsg("too high volume id %d, max. volumes is %d",
		       vi->id, ui->max_volumes);
		errno = EINVAL;
		return -1;
	}

	if (vi->alignment >= ui->leb_size) {
		errmsg("too large alignment %d, max is %d (LEB size)",
		       vi->alignment, ui->leb_size);
		errno = EINVAL;
		return -1;
	}

	inbuf = malloc(ui->leb_size);
	if (!inbuf)
		return sys_errmsg("cannot allocate %d bytes of memory",
				  ui->leb_size);
	outbuf = malloc(ui->peb_size_oob);
	if (!outbuf) {
		sys_errmsg("cannot allocate %d bytes of memory", ui->peb_size_oob);
		goto out_free;
	}

	//memset(outbuf, 0xFF, ui->data_offs);
    pbuf = outbuf + ui->data_offs_oob;
    memset(outbuf, 0xFF, ui->data_offs_oob);

	ubigen_init_ec_hdr(ui, (struct ubi_ec_hdr *)outbuf, ec);
	/* calculate ecc of ec hdr */
    ubigen_ecc_calc(ui, outbuf, NULL, 0);

	while (bytes) {
		int l;
		struct ubi_vid_hdr *vid_hdr;

		if (bytes < len)
			len = bytes;
		bytes -= len;

		l = len;
        page_cnt = len >> ui->page_shift;
        memset(pbuf, 0xFF, ui->leb_size_oob);

		do {
			rd = read(in, inbuf + len - l, l);
			if (rd != l) {
				sys_errmsg("cannot read %d bytes from the input file", l);
				goto out_free1;
			}

			l -= rd;
		} while (l);

		vid_hdr = (struct ubi_vid_hdr *)(&outbuf[ui->vid_offs_oob]);
		ubigen_init_vid_hdr(ui, vi, vid_hdr, lnum, inbuf, len);
        /* calculate ecc of vid hdr */
        ubigen_ecc_calc(ui, outbuf, NULL, 1);

		for (i = 0; i < page_cnt; i++)
		{
		  /* copy data and calculate ecc for every subpage byte */
		  ubigen_ecc_calc(ui, pbuf, inbuf, i);
		}

		ubigen_ecc_calc_tail(ui, pbuf, inbuf, len);
		//memset(outbuf + ui->data_offs + len, 0xFF,
		       //ui->peb_size - ui->data_offs - len);

		if (write(out, outbuf, ui->peb_size_oob) != ui->peb_size_oob) {
			sys_errmsg("cannot write %d bytes to the output file", ui->peb_size_oob);
			goto out_free1;
		}

		lnum += 1;
	}

	free(outbuf);
	free(inbuf);
	return 0;

out_free1:
	free(outbuf);
out_free:
	free(inbuf);
	return -1;
}

int ubigen_write_layout_vol(const struct ubigen_info *ui, int peb1, int peb2,
			    long long ec1, long long ec2,
			    struct ubi_vtbl_record *vtbl, int fd)
{
	int ret;
    unsigned int i, page_cnt, len;
	struct ubigen_vol_info vi;
	unsigned char *outbuf, *pbuf;
	struct ubi_vid_hdr *vid_hdr;
	off_t seek;

	vi.bytes = ui->leb_size * UBI_LAYOUT_VOLUME_EBS;
	vi.id = UBI_LAYOUT_VOLUME_ID;
	vi.alignment = UBI_LAYOUT_VOLUME_ALIGN;
	vi.data_pad = ui->leb_size % UBI_LAYOUT_VOLUME_ALIGN;
	vi.usable_leb_size = ui->leb_size - vi.data_pad;
	vi.data_pad = ui->leb_size - vi.usable_leb_size;
	vi.type = UBI_LAYOUT_VOLUME_TYPE;
	vi.name = UBI_LAYOUT_VOLUME_NAME;
	vi.name_len = strlen(UBI_LAYOUT_VOLUME_NAME);
	vi.compat = UBI_LAYOUT_VOLUME_COMPAT;

    outbuf = malloc(ui->peb_size_oob);
    if (!outbuf)
        return sys_errmsg("failed to allocate %d bytes", ui->peb_size_oob);
    //memset(outbuf, 0xFF, ui->data_offs);
    memset(outbuf, 0xFF, ui->peb_size_oob);

    vid_hdr = (struct ubi_vid_hdr *)(&outbuf[ui->vid_offs_oob]);
    pbuf = outbuf + ui->data_offs_oob;
	page_cnt = ui->vtbl_size >> ui->page_shift;

	for (i = 0; i < page_cnt; i++)
	{
	   ubigen_ecc_calc(ui, pbuf, (unsigned char *)vtbl, i);
	}

	/* write the remainder data which is less than one page */
	ubigen_ecc_calc_tail(ui, pbuf, (unsigned char *)vtbl, ui->vtbl_size);
	//memset(outbuf + ui->data_offs + ui->vtbl_size, 0xFF,
	       //ui->peb_size - ui->data_offs - ui->vtbl_size);

	seek = peb1 * ui->peb_size_oob;
	if (lseek(fd, seek, SEEK_SET) != seek) {
		sys_errmsg("cannot seek output file");
		goto out_free;
	}

	ubigen_init_ec_hdr(ui, (struct ubi_ec_hdr *)outbuf, ec1);
	ubigen_init_vid_hdr(ui, &vi, vid_hdr, 0, NULL, 0);
	/* calculate ecc of ec hdr */
	ubigen_ecc_calc(ui, outbuf, NULL, 0);
	/* calculate ecc of vid hdr */
    ubigen_ecc_calc(ui, outbuf, NULL, 1);
	
	ret = write(fd, outbuf, ui->peb_size_oob);
	if (ret != ui->peb_size_oob) {
		sys_errmsg("cannot write %d bytes", ui->peb_size_oob);
		goto out_free;
	}

	seek = peb2 * ui->peb_size_oob;
	if (lseek(fd, seek, SEEK_SET) != seek) {
		sys_errmsg("cannot seek output file");
		goto out_free;
	}
	ubigen_init_ec_hdr(ui, (struct ubi_ec_hdr *)outbuf, ec2);
	ubigen_init_vid_hdr(ui, &vi, vid_hdr, 1, NULL, 0);
	/* calculate ecc of ec hdr */
	ubigen_ecc_calc(ui, outbuf, NULL, 0);
	/* calculate ecc of vid hdr */
    ubigen_ecc_calc(ui, outbuf, NULL, 1);
	
	ret = write(fd, outbuf, ui->peb_size_oob);
	if (ret != ui->peb_size_oob) {
		sys_errmsg("cannot write %d bytes", ui->peb_size_oob);
		goto out_free;
	}

	free(outbuf);
	return 0;

out_free:
	free(outbuf);
	return -1;
}

/* write out mtd partition without ubi information, only ecc is considered */
int ubigen_write_partition(const struct ubigen_info *ui,
			long long bytes, int in, int out)
{
    int rd;
	unsigned int i, page_cnt, len = ui->peb_size;
	unsigned char *inbuf, *outbuf;
    /*
	if (vi->alignment >= ui->peb_size) {
		errmsg("too large alignment %d, max is %d (LEB size)",
		       vi->alignment, ui->leb_size);
		errno = EINVAL;
		return -1;
	}
    */
    
	inbuf = malloc(ui->peb_size);
	if (!inbuf)
		return sys_errmsg("cannot allocate %d bytes of memory",
				  ui->peb_size);
	outbuf = malloc(ui->peb_size_oob);
	if (!outbuf) {
		sys_errmsg("cannot allocate %d bytes of memory", ui->peb_size_oob);
		goto out_free;
	}

	while (bytes) {
		int l;

		if (bytes < len)
			len = bytes;
		bytes -= len;

		l = len;
        page_cnt = len >> ui->page_shift;
        memset(outbuf, 0xFF, ui->peb_size_oob);

		do {
			rd = read(in, inbuf + len - l, l);
			if (rd != l) {
				sys_errmsg("cannot read %d bytes from the input file", l);
				goto out_free1;
			}

			l -= rd;
		} while (l);

		for (i = 0; i < page_cnt; i++)
		{
		  ubigen_ecc_calc(ui, outbuf, inbuf, i);
		}

		/* write the remainder data which is less than one page */
		ubigen_ecc_calc_tail(ui, outbuf, inbuf, len);
	
		if (write(out, outbuf, ui->peb_size_oob) != ui->peb_size_oob) {
			sys_errmsg("cannot write %d bytes to the output file", ui->peb_size_oob);
			goto out_free1;
		}
	}

	free(outbuf);
	free(inbuf);
	return 0;

out_free1:
	free(outbuf);
out_free:
	free(inbuf);
	return -1;
}

/* write the length of size 0xff to the file pointed by fd, append mode */
int ubigen_write_empty_file(const struct ubigen_info *ui, long long size, int out)
{
    unsigned char *outbuf;

    outbuf = malloc(ui->peb_size_oob);
    if (!outbuf) {
    	sys_errmsg("cannot allocate %d bytes of memory", ui->peb_size_oob);
    	goto out_free;
    }
    memset(outbuf, 0xFF, ui->peb_size_oob);

    while (size >= ui->peb_size_oob)
    {
        if (write(out, outbuf, ui->peb_size_oob) != ui->peb_size_oob) {
            sys_errmsg("cannot write %d bytes to the output file", ui->peb_size_oob);
            goto out_free;
        }
        size = size - ui->peb_size_oob;
    }

    if (size > 0)
    {
        if (write(out, outbuf, size) != size) {
            sys_errmsg("cannot write %d bytes to the output file", size);
            goto out_free;
        }
    }

    free(outbuf);
    return 0;

out_free:
    free(outbuf);
    return -1;
}

/* write out bfn miu partition without ubi information */
int ubigen_write_miu_partition(const struct ubigen_info *ui,
			long long bytes, int in, int out)
{
    int rd;
	unsigned int i, k, page_cnt, pagelen, rbytes, rlen, 
                 first = 1, len = 512 * ui->page_num;
	unsigned char *inbuf, *outbuf, *pbuf;

    pagelen = ui->page_size_oob;
	inbuf = malloc(len);
	if (!inbuf)
		return sys_errmsg("cannot allocate %d bytes of memory",
				  ui->peb_size);
	outbuf = malloc(ui->peb_size_oob);
	if (!outbuf) {
		sys_errmsg("cannot allocate %d bytes of memory", ui->peb_size_oob);
		goto out_free;
	}

    *((unsigned int*)(inbuf+0x00)) = MIUIDTAG; //0x4D495530
    *((unsigned int*)(inbuf+0x04)) = ui->min_io_size;
    *((unsigned int*)(inbuf+0x10)) = bytes >> 9; // number of MIU page (in 512 bytes size)
    *((unsigned int*)(inbuf+0x14)) = ui->page_num;
    rbytes = bytes + 512;

	while (rbytes) {
		int l;

		if (rbytes < len)
			len = rbytes;
		rbytes -= len;

		rlen = l = len;
		pbuf = inbuf;
		if (first)
		{
		    rlen = l = l - 512;
			pbuf = inbuf + 512;
		}
        page_cnt = len >> 9;
        memset(outbuf, 0xFF, ui->peb_size_oob);

		do {
			rd = read(in, pbuf + rlen - l, l);
			if (rd != l) {
				sys_errmsg("cannot read %d bytes from the input file", l);
				goto out_free1;
			}

			l -= rd;
		} while (l);

		for (i = 0; i < page_cnt; i++)
		{
		  ubigen_ecc_calc_miu(ui, outbuf, inbuf, i);
		}

		/* write the remainder data which is less than one page */
		l = len % 512;
		if (l)
		{
		  if (!ubigen_empty_check(ui, (inbuf + 512 * page_cnt), l))
		  {
	          memcpy(outbuf + pagelen * page_cnt,
	                        inbuf + ui->min_io_size * page_cnt, l);

		      for (k = 0; k < ui->sector_num; k++)
		      {
	              rd = ecc_cal(ui->ecc_algorithm, outbuf + pagelen * page_cnt + 528 * k,
	                            outbuf + pagelen * page_cnt + 528 * k + 512, ui->ecc_offs,
	                            outbuf + pagelen * page_cnt + 528 * k + 512 + ui->ecc_offs, 0);
	              #if 0
	              if (rd != NO_ERROR)
	              {
	                  sys_errmsg("cannot calculate ecc, error code:%d", rd);
	                  goto out_free1;
	              }
	              #endif
		      }
		  }
		}
	
		if (write(out, outbuf, ui->peb_size_oob) != ui->peb_size_oob) {
			sys_errmsg("cannot write %d bytes to the output file", ui->peb_size_oob);
			goto out_free1;
		}

		first = 0;
	}

	free(outbuf);
	free(inbuf);
	return 0;

out_free1:
	free(outbuf);
out_free:
	free(inbuf);
	return -1;
}

/* init and layout data for parameter bin */
static int ubigen_bfnparam_init(const struct ubigen_info *ui,
			long long rbytes, int in, int out, unsigned char* outbuf,
			unsigned char* inbuf)
{
    int rd;
    unsigned int i, l, page_cnt, rlen,
                 first = 1, len = ui->peb_size;
    unsigned char *pbuf;

	while (rbytes) {
		if (rbytes < len)
			len = rbytes;
		rbytes -= len;

		l = len;
        page_cnt = len >> ui->page_shift;
        memset(outbuf, 0xFF, ui->peb_size_oob);

		rlen = l = len;
		pbuf = inbuf;
		if (first)
		{
		    rlen = l = l - ui->min_io_size;
			pbuf = inbuf + ui->min_io_size;
		}
		do {
			rd = read(in, pbuf + rlen - l, l);
			if (rd != l) {
				sys_errmsg("cannot read %d bytes from the input file", l);
				goto out_free1;
			}

			l -= rd;
		} while (l);

		for (i = 0; i < page_cnt; i++)
		{
		  ubigen_ecc_calc(ui, outbuf, inbuf, i);
		}

		/* write the remainder data which is less than one page */
		ubigen_ecc_calc_tail(ui, outbuf, inbuf, len);

		if (write(out, outbuf, ui->peb_size_oob) != ui->peb_size_oob) {
			sys_errmsg("cannot write %d bytes to the output file", ui->peb_size_oob);
			goto out_free1;
		}

		first = 0;
	}

	
	return 0;

out_free1:
	//free(outbuf);
	//free(inbuf);
	return -1;
}

/* write out bfn ldr partition without ubi information */
int ubigen_write_ldr_partition(const struct ubigen_info *ui,
			long long bytes, int in, int out)
{
    int rd;
	unsigned int l, rbytes;
	unsigned int Ldr2MIUPos = 0;
    unsigned int LdrSize = 0;
    unsigned int LdrEntryOffset = 0; // means the actually position offset from u8BINPos for bin file to store
	unsigned char *inbuf, *outbuf;

	inbuf = malloc(ui->peb_size);
	if (!inbuf)
		return sys_errmsg("cannot allocate %d bytes of memory",
				  ui->peb_size);
	outbuf = malloc(ui->peb_size_oob);
	if (!outbuf) {
		sys_errmsg("cannot allocate %d bytes of memory", ui->peb_size_oob);
	    free(inbuf);

		return -1;
	}
	memset(inbuf, 0x00, ui->peb_size);
	memset(outbuf, 0xFF, ui->peb_size_oob);
    
    l = 16;
	do {
		rd = read(in, outbuf + 16 - l, l);
		if (rd != l) {
			sys_errmsg("cannot read %d bytes from the input file", l);
			free(outbuf);
	        free(inbuf);

			return -1;
		}

		l -= rd;
	} while (l);

    Ldr2MIUPos = *((unsigned int*)outbuf);
    LdrSize = *((unsigned int*)(outbuf+12));
    LdrEntryOffset = *((unsigned int*)(outbuf+4));
    Ldr2MIUPos += LdrEntryOffset;
    LdrSize -= LdrEntryOffset;
	rbytes = bytes + ui->min_io_size - LdrEntryOffset;
	
    *((unsigned int*)(inbuf+0x00)) = LDRIDTAG;
    *((unsigned int*)(inbuf+0x04)) = Ldr2MIUPos;// the position we are going to put to MIU and jump
    *((unsigned int*)(inbuf+0x10)) = LDRIDTAG;
    *((unsigned int*)(inbuf+0x14)) = LdrSize;

    if (lseek(in, LdrEntryOffset, SEEK_SET) != LdrEntryOffset) {
        sys_errmsg("cannot seek file \"%s\"", in);
		free(outbuf);
        free(inbuf);

		return -1;
    }

       int ret = ubigen_bfnparam_init(ui, rbytes, in, out, outbuf, inbuf);
	free(inbuf);
	 return ret;
}

/* write out bfn app partition without ubi information */
int ubigen_write_app_partition(const struct ubigen_info *ui,
			long long bytes, int in, int out)
{
    int rd;
	unsigned int l, rbytes;
    unsigned int Bin2MIUPos = 0;
    unsigned int BinSize = 0;
    unsigned int APPEntryOffset = 0; // means the actually position offset from u8BINPos for bin file to store
    unsigned short HEADER_OFFSET = 0x100;
	unsigned char *inbuf, *outbuf;

	inbuf = malloc(ui->peb_size);
	if (!inbuf)
		return sys_errmsg("cannot allocate %d bytes of memory",
				  ui->peb_size);
	outbuf = malloc(ui->peb_size_oob);
	if (!outbuf) {
		sys_errmsg("cannot allocate %d bytes of memory", ui->peb_size_oob);
		free(inbuf);

		return -1;
	}
    	memset(inbuf, 0x00, ui->peb_size);
	memset(outbuf, 0xFF, ui->peb_size_oob);
    l = 16;
	do {
		rd = read(in, outbuf + 16 - l, l);
		if (rd != l) {
			sys_errmsg("cannot read %d bytes from the input file", l);
		    free(outbuf);
            free(inbuf);

		    return -1;
		}

		l -= rd;
	} while (l);

    Bin2MIUPos = *((unsigned int*)(outbuf));
    BinSize = *((unsigned int*)(outbuf+8));
    APPEntryOffset = *((unsigned int*)(outbuf+4));
    Bin2MIUPos += HEADER_OFFSET; //APPEntryOffset;
    BinSize -= HEADER_OFFSET;//APPEntryOffset;
    APPEntryOffset -= HEADER_OFFSET;
    *((unsigned int*)(inbuf+0x00)) = BINIDTAG;
    *((unsigned int*)(inbuf+0x04)) = Bin2MIUPos;
    *((unsigned int*)(inbuf+0x10)) = BINIDTAG; // the position we are going to put to MIU and jump
    *((unsigned int*)(inbuf+0x14)) = BinSize;
    *((unsigned int*)(inbuf+0x20)) = BINIDTAG; // the position we are going to put to MIU and jump
    *((unsigned int*)(inbuf+0x24)) = APPEntryOffset;

	rbytes = (bytes  - HEADER_OFFSET);

    if (lseek(in, HEADER_OFFSET, SEEK_SET) != HEADER_OFFSET) {
        sys_errmsg("cannot seek file \"%s\"", in);
		free(outbuf);
        free(inbuf);

		return -1;
    }

    unsigned int writeBytes = ui->peb_size;
    while(rbytes > 0)
    {
	 if (rbytes < (ui->peb_size-ui->min_io_size))
	 	writeBytes = rbytes+ui->min_io_size;

	 printf("[ + ] ------------------------------------------------\n");
	 printf("[ + ]    rbytes=%d\n", rbytes);
	 printf("[ + ] ------------------------------------------------\n");
	  
	 if(ubigen_bfnparam_init(ui, writeBytes, in, out, outbuf, inbuf) < 0)
	 	break;
	 rbytes -= (writeBytes-ui->min_io_size);
}
	free(outbuf);
	free(inbuf);
	if (rbytes!=0)
		return -1;
	else 
		return 0;
}
