/*
 * Copyright (C) 2008 Nokia Corporation
 * Copyright (c) International Business Machines Corp., 2006
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Generate emmc images.
 *
 * Authors: Artem Bityutskiy
 *          Oliver Lohmann
 */

#define PROGRAM_VERSION "1.0"
#define PROGRAM_NAME    "emmcnize"
#define WRITE_BUFFER_SIZE 0x6400000

#include <sys/stat.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <fcntl.h>

#include <mtd/ubi-media.h>
#include <libubigen.h>
#include <libiniparser.h>
#include "common.h"
#include "ubiutils-common.h"
#include "part_emmc.h"

struct args {
    const char *f_in;
    const char *f_out;
    int integrate;
    int out_fd;
    uint32_t image_seq;
    dictionary *dict;
};

static struct args args;

static int read_section(const struct ubigen_info *ui, const char *sname,
			struct ubigen_vol_info *vi, const char **img,
			struct stat *st)
{
	char buf[256];
	const char *p;

	*img = NULL;

	if (strlen(sname) > 128)
		return errmsg("too long section name \"%s\"", sname);

	/* Fetch the name of the volume image file */
	sprintf(buf, "%s:image", sname);
	p = iniparser_getstring(args.dict, buf, NULL);
	if (p) {
		*img = p;
		if (stat(p, st)){
			printf("cannot stat \"%s\" referred from section \"%s\"",
					  p, sname);
                    st->st_size = 0;
             }
		if (st->st_size == 0)
			printf("empty file \"%s\" referred from section \"%s\"",
				       p, sname);
            
	} else if (vi->type == UBI_VID_STATIC)
		return errmsg("image is not specified for static volume in section \"%s\"",
			      sname);

	/* Fetch volume id */
	sprintf(buf, "%s:vol_id", sname);
	vi->id = iniparser_getint(args.dict, buf, -1);
	if (vi->id == -1)
		return errmsg("\"vol_id\" key not found in section  \"%s\"", sname);
	if (vi->id < 0)
		return errmsg("negative volume ID %d in section \"%s\"",
			      vi->id, sname);
	if (vi->id > ui->max_volumes)
		return errmsg("too high volume ID %d in section \"%s\", max. is %d",
			      vi->id, sname, ui->max_volumes);

	printf("volume ID: %d\n", vi->id);

	/* Fetch volume size */
	sprintf(buf, "%s:vol_size", sname);
	p = iniparser_getstring(args.dict, buf, NULL);
	if (p) {
		vi->bytes = ubiutils_get_bytes(p);
		if (vi->bytes <= 0)
			return errmsg("bad \"vol_size\" key value \"%s\" (section \"%s\")",
				      p, sname);

		/* Make sure the image size is not larger than volume size */
		if (*img && st->st_size > vi->bytes)
			return errmsg("error in section \"%s\": size of the image file "
				      "\"%s\" is %lld, which is larger than volume size %lld",
				      sname, *img, (long long)st->st_size, vi->bytes);
		printf("volume size: %lld bytes\n", vi->bytes);
	} else {
		struct stat st;

		if (!*img)
			return errmsg("neither image file (\"image=\") nor volume size "
				      "(\"vol_size=\") specified in section \"%s\"", sname);

		if (stat(*img, &st))
			return sys_errmsg("cannot stat \"%s\"", *img);

		vi->bytes = st.st_size;

		if (vi->bytes == 0)
			return errmsg("file \"%s\" referred from section \"%s\" is empty",
				      *img, sname);

		normsg_cont("volume size was not specified in section \"%s\", assume"
			    " minimum to fit image \"%s\"", sname, *img);
		ubiutils_print_bytes(vi->bytes, 1);
		printf("\n");
	}

	/* Fetch volume name */
	sprintf(buf, "%s:vol_name", sname);
	p = iniparser_getstring(args.dict, buf, NULL);
	if (!p)
		return errmsg("\"vol_name\" key not found in section \"%s\"", sname);

	vi->name = p;
	vi->name_len = strlen(p);
	if (vi->name_len > UBI_VOL_NAME_MAX)
		return errmsg("too long volume name in section \"%s\", max. is %d characters",
			      vi->name, UBI_VOL_NAME_MAX);

	printf("volume name: %s\n\n", p);
	return 0;
}

int main(int argc, char * const argv[])
{
	int err = -1, sects, i;
	struct ubigen_info ui;
	struct ubigen_vol_info *vi;
        long long image_size=0, blank_size=0, per_write_size=0;

       ui.max_volumes = EMMC_RESERVED_FOR_MAP;

	args.out_fd = open(argv[2], O_CREAT | O_TRUNC | O_WRONLY,
					   S_IWUSR | S_IRUSR | S_IRGRP | S_IWGRP | S_IROTH);
	if (args.out_fd == -1)
	    return sys_errmsg("cannot open file \"%s\"", argv[2]);
    
	args.f_out = argv[2];
    
       args.f_in = argv[1];
	args.dict = iniparser_load(args.f_in);
	if (!args.dict) {
		errmsg("cannot load the input ini file \"%s\"", args.f_in);
		goto out_file;
	}

	printf("loaded the ini-file \"%s\"\n", args.f_in);

	/* Each section describes one volume */
	sects = iniparser_getnsec(args.dict);
	if (sects == -1) {
		errmsg("ini-file parsing error (iniparser_getnsec)");
		goto out_dict;
	}

	printf( "count of sections: %d\n", sects);
	if (sects == 0) {
		errmsg("no sections found the ini-file \"%s\"", args.f_in);
		goto out_dict;
	}

	if (sects > ui.max_volumes) {
		errmsg("too many sections (%d) in the ini-file \"%s\"",
		       sects, args.f_in);
		normsg("each section corresponds to an UBI volume, maximum "
		       "count of volumes is %d", ui.max_volumes);
		goto out_dict;
	}

	vi = calloc(sizeof(struct ubigen_vol_info), sects);
	if (!vi) {
		errmsg("cannot allocate memory");
		goto out_dict;
	}

/* First write the partition table */
	for (i = 0; i < sects; i++) {
		const char *sname_tbl = iniparser_getsecname(args.dict, i);
		const char *img_tbl = NULL;
		struct stat st_tbl;
		int uniqueid, k, err_tbl;
             emmc_driver_desc_t *ddb_base;
             emmc_partition_t *pdb_base; 
             ddb_base = (emmc_driver_desc_t *) malloc(sizeof(emmc_driver_desc_t));
             pdb_base = (emmc_partition_t *) malloc(sizeof(emmc_partition_t));
             memset(ddb_base, 0, sizeof(emmc_driver_desc_t));
             memset(pdb_base, 0, sizeof(emmc_partition_t));
            
             ddb_base->signature = EMMC_DRIVER_MAGIC;
             ddb_base->version = EMMC_PARTITIONTABLE_VERSION2;
             ddb_base->drvr_cnt = EMMC_RESERVED_FOR_MAP_V2;             
             pdb_base->signature = EMMC_PARTITION_MAGIC;
            
		if (!sname_tbl) {
			errmsg("ini-file parsing error (iniparser_getsecname)");
			goto out_free;
		}

		printf("parsing section \"%s\" tbl\n", sname_tbl);

		err_tbl = read_section(&ui, sname_tbl, &vi[i], &img_tbl, &st_tbl);
		if (err_tbl == -1)
			goto out_free;

		printf("adding volume %d tbl\n", vi[i].id);

		/*
		 * Make sure that volume ID and name is unique and that only
		 * one volume has auto-resize flag
		 */
		for (uniqueid = 0; uniqueid < i; uniqueid++) {
			if (vi[i].id == vi[uniqueid].id) {
				errmsg("volume IDs must be unique, but ID %d "
				       "in section \"%s\" is not",
				       vi[i].id, sname_tbl);
                           free(ddb_base);
                           free(pdb_base);
				goto out_free;
			}

			if (!strcmp(vi[i].name, vi[uniqueid].name)) {
				errmsg("volume name must be unique, but name "
				       "\"%s\" in section \"%s\" is not",
				       vi[i].name, sname_tbl);
                           free(ddb_base);
                           free(pdb_base);                
				goto out_free;
			}
		}

        //Do partition tbl write ops
            strcpy((char *)pdb_base->name, (const char *)vi[i].name);
            pdb_base->block_count = vi[i].bytes / 512;
            pdb_base->start_block = EMMC_PARTITION_START;

            for(k=0; k<i; k++)
                pdb_base->start_block += vi[k].bytes / 512;

            printf("section \"%s\" start at block %d\n", sname_tbl, pdb_base->start_block);
            if(i == 0)
                err_tbl = write(args.out_fd, ddb_base, 512);
            
            err_tbl = write(args.out_fd, pdb_base, 512);
            
            free(ddb_base);
            free(pdb_base); 
       }
    
/* Fill the blank partition tbl  */
       unsigned char *zbuf_tbl;
       int zerobytes;
       int blank_tbl;
       if(sects < EMMC_RESERVED_FOR_MAP_V2){
            zerobytes = (EMMC_PARTITION_START - sects - 1) * 512;
            zbuf_tbl =malloc(zerobytes);
            memset(zbuf_tbl, 0xFF, zerobytes);
            blank_tbl = write(args.out_fd, zbuf_tbl, zerobytes);
            if (blank_tbl != zerobytes) {
                errmsg("cannot write blank partition tbl\n");
                free(zbuf_tbl);
                goto out_free;
            }
            free(zbuf_tbl);
        }else{
            printf("Too much partitions!\n");
            goto out_done;
        }
       
       printf("***partition tbl wrote done!***\n");

/* Fill the left blank area by default */

       if ((argc>3) && (strcmp(argv[3], "-p") == 0))
            goto out_done;
           
/* Then write the image */
	for (i = 0; i < sects; i++) {
		const char *sname = iniparser_getsecname(args.dict, i);
		const char *img = NULL;
		struct stat st;
		int fd, j, rd;
              unsigned char *inbuf, *zbuf;

		if (!sname) {
			errmsg("ini-file parsing error (iniparser_getsecname)");
			goto out_free;
		}

		printf("parsing section \"%s\"\n", sname);

		err = read_section(&ui, sname, &vi[i], &img, &st);
		if (err == -1)
			goto out_free;

		printf("adding volume %d\n", vi[i].id);

		/*
		 * Make sure that volume ID and name is unique and that only
		 * one volume has auto-resize flag
		 */
		for (j = 0; j < i; j++) {
			if (vi[i].id == vi[j].id) {
				errmsg("volume IDs must be unique, but ID %d "
				       "in section \"%s\" is not",
				       vi[i].id, sname);
				goto out_free;
			}

			if (!strcmp(vi[i].name, vi[j].name)) {
				errmsg("volume name must be unique, but name "
				       "\"%s\" in section \"%s\" is not",
				       vi[i].name, sname);
				goto out_free;
			}
		}

		if (img) {
			printf("writing volume %d\n", vi[i].id);
			printf("image file: %s\n", img);
                    if(st.st_size){
			    fd = open(img, O_RDONLY);
			    if (fd == -1) {
				    sys_errmsg("cannot open \"%s\"", img);
				    goto out_free;
			    }

                        image_size = st.st_size;
                        inbuf = (unsigned char *)malloc(WRITE_BUFFER_SIZE);
                        while(image_size){
                                if (image_size >= WRITE_BUFFER_SIZE)
                                {
                                    per_write_size= WRITE_BUFFER_SIZE;
                                }
                                else
                                {
                                    per_write_size = image_size;
                                }                            
                                rd = read(fd, inbuf, per_write_size);                  
                                if(rd != per_write_size){
        				    errmsg("cannot read volume for section \"%s\"", sname);
        				    free(inbuf);
        				    close(fd);
        				    goto out_free;                
                                }                       
        			    
                                err = write(args.out_fd, inbuf, per_write_size);                     
                                if (err != per_write_size) {
            				    errmsg("cannot write volume for section \"%s\"", sname);
                                          free(inbuf);
                                          close(fd);
            				    goto out_free;
                                }
                                
                                image_size -= per_write_size;
                        }
                        close(fd);
                        free(inbuf);                
                    }
                    if(st.st_size < vi[i].bytes){
                        blank_size = vi[i].bytes - st.st_size;

                        zbuf = malloc(WRITE_BUFFER_SIZE);
                        memset(zbuf, 0xFF, WRITE_BUFFER_SIZE);

                        while(blank_size){
                            if (blank_size >= WRITE_BUFFER_SIZE)
                            {
                                per_write_size= WRITE_BUFFER_SIZE;
                            }
                            else
                            {
                                per_write_size = blank_size;
                            }

                            err = write(args.out_fd, zbuf, per_write_size);
                            if (err != per_write_size) {
    				    errmsg("cannot write blank for section \"%s\"", sname);
                               free(zbuf);
    				    goto out_free;
    			        }                            

                            blank_size -= per_write_size;
                        }                       

                        free(zbuf);
                    }
                        
		}
        
		printf("\n");
	}

out_done:
	printf("done\n");

	free(vi);
	iniparser_freedict(args.dict);
	close(args.out_fd);
	return 0;

out_free:
	free(vi);
out_dict:
	iniparser_freedict(args.dict);
out_file:
	close(args.out_fd);
	remove(args.f_out);
	return err;
}
