/*
 * Copyright (c) International Business Machines Corp., 2006
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Authors: Artem Bityutskiy (?и???кий ????м), Thomas Gleixner
 */

/*
 * UBI wear-leveling unit.
 *
 * This unit is responsible for wear-leveling. It works in terms of physical
 * eraseblocks and erase counters and knows nothing about logical eraseblocks,
 * volumes, etc. From this unit's perspective all physical eraseblocks are of
 * two types - used and free. Used physical eraseblocks are those that were
 * "get" by the 'ubi_wl_get_peb()' function, and free physical eraseblocks are
 * those that were put by the 'ubi_wl_put_peb()' function.
 *
 * Physical eraseblocks returned by 'ubi_wl_get_peb()' have only erase counter
 * header. The rest of the physical eraseblock contains only 0xFF bytes.
 *
 * When physical eraseblocks are returned to the WL unit by means of the
 * 'ubi_wl_put_peb()' function, they are scheduled for erasure. The erasure is
 * done asynchronously in context of the per-UBI device background thread,
 * which is also managed by the WL unit.
 *
 * The wear-leveling is ensured by means of moving the contents of used
 * physical eraseblocks with low erase counter to free physical eraseblocks
 * with high erase counter.
 *
 * The 'ubi_wl_get_peb()' function accepts data type hints which help to pick
 * an "optimal" physical eraseblock. For example, when it is known that the
 * physical eraseblock will be "put" soon because it contains short-term data,
 * the WL unit may pick a free physical eraseblock with low erase counter, and
 * so forth.
 *
 * If the WL unit fails to erase a physical eraseblock, it marks it as bad.
 *
 * This unit is also responsible for scrubbing. If a bit-flip is detected in a
 * physical eraseblock, it has to be moved. Technically this is the same as
 * moving it for wear-leveling reasons.
 *
 * As it was said, for the UBI unit all physical eraseblocks are either "free"
 * or "used". Free eraseblock are kept in the @wl->free RB-tree, while used
 * eraseblocks are kept in a set of different RB-trees: @wl->used,
 * @wl->prot.pnum, @wl->prot.aec, and @wl->scrub.
 *
 * Note, in this implementation, we keep a small in-RAM object for each physical
 * eraseblock. This is surely not a scalable solution. But it appears to be good
 * enough for moderately large flashes and it is simple. In future, one may
 * re-work this unit and make it more scalable.
 *
 * At the moment this unit does not utilize the sequence number, which was
 * introduced relatively recently. But it would be wise to do this because the
 * sequence number of a logical eraseblock characterizes how old is it. For
 * example, when we move a PEB with low erase counter, and we need to pick the
 * target PEB, we pick a PEB with the highest EC if our PEB is "old" and we
 * pick target PEB with an average EC if our PEB is not very "old". This is a
 * room for future re-works of the WL unit.
 *
 * FIXME: looks too complex, should be simplified (later).
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>
#include "ubi.h"

/* Number of physical eraseblocks reserved for wear-leveling purposes */
#define WL_RESERVED_PEBS 1

/**
 * ubi_wl_get_peb - get a physical eraseblock.
 * @ubi: UBI device description object
 * @dtype: type of data which will be stored in this physical eraseblock
 *
 * This function returns a physical eraseblock in case of success and a
 * negative error code in case of failure. Might sleep.
 */
int ubi_wl_get_peb(struct ubi_device *ubi, int dtype)
{
	int pnum, i, j;
	int err;
	struct ubi_ec_hdr *ec_hdr;

	ec_hdr = malloc(ubi->ec_hdr_alsize);
	if (!ec_hdr){
		printf("[%s]: malloc ec header failed\n", __func__);
		return -ENOMEM;
	}
	memset(ec_hdr, 0, ubi->ec_hdr_alsize);

	for(pnum = 0; pnum < ubi->peb_count; pnum++){
		i = pnum / (sizeof(unsigned char) * 8);
		j = pnum & (sizeof(unsigned char ) * 8 - 1);
		if((ubi->wl_bitmap[i] & (1 << j)) == 0){
			ubi->wl_bitmap[i] |= (1 << j);
			break;
		}
	}
	if(pnum == ubi->peb_count){
		printf("[%s]: No available peb\n", __func__);
		free(ec_hdr);
		return -1;
	}

	ec_hdr->ec = __cpu_to_be64(1);

	err = ubi_io_write_ec_hdr(ubi, pnum, ec_hdr);
	if(err){
		free(ec_hdr);
		return -1;
	}

	free(ec_hdr);
	//printf("[%s]: pnum %d-------------------------\n", __func__, pnum);
	return pnum;
}

/**
 * ubi_wl_put_peb - return a physical eraseblock to the wear-leveling unit.
 * @ubi: UBI device description object
 * @pnum: physical eraseblock to return
 * @torture: if this physical eraseblock has to be tortured
 *
 * This function is called to return physical eraseblock @pnum to the pool of
 * free physical eraseblocks. The @torture flag has to be set if an I/O error
 * occurred to this @pnum and it has to be tested. This function returns zero
 * in case of success, and a negative error code in case of failure.
 */
int ubi_wl_put_peb(struct ubi_device *ubi, int pnum, int torture)
{
	int err, i, j;

	//printf("[%s]: pnum %d-------------------------\n", __func__, pnum);

	err = ubi_io_sync_erase(ubi, pnum, torture);

	if(err < 0){
		printf("[%s]: erase peb %d failed\n", __func__, pnum);
		return err;
	}
	i = pnum / (sizeof(unsigned char) * 8);
	j = pnum & (sizeof(unsigned char ) * 8 - 1);

	ubi->wl_bitmap[i] &= ~(1 << j);
	return 0;
}

/**
 * ubi_wl_init_scan - initialize the wear-leveling unit using scanning
 * information.
 * @ubi: UBI device description object
 * @si: scanning information
 *
 * This function returns zero in case of success, and a negative error code in
 * case of failure.
 */
int ubi_wl_init(struct ubi_device *ubi)
{
	int size;
	int peb_count;

	if (ubi->avail_pebs < WL_RESERVED_PEBS) {
		ubi_err("no enough physical eraseblocks (%d, need %d)",
			ubi->avail_pebs, WL_RESERVED_PEBS);
		return -1;;
	}
	ubi->avail_pebs -= WL_RESERVED_PEBS;
	ubi->rsvd_pebs += WL_RESERVED_PEBS;

	peb_count = ubi->peb_count;
	peb_count = ALIGN(peb_count, sizeof(unsigned char) * 8);
	size = peb_count / (sizeof(unsigned char) * 8);

	ubi->wl_bitmap = (unsigned char *)malloc(size);
	if(!ubi->wl_bitmap){
		printf("[%s]: malloc wl_bitmap failed\n", __func__);
		return -1;
	}
	memset(ubi->wl_bitmap, 0, size);

	printf("[%s]: malloc %d byte for wl bitmap\n", __func__, size);

	return 0;
}
