/*
 * Copyright (c) International Business Machines Corp., 2006
 * Copyright (c) Nokia Corporation, 2006, 2007
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Author: Artem Bityutskiy (Битюцкий Артём)
 */

/*
 * This file includes volume table manipulation code. The volume table is an
 * on-flash table containing volume meta-data like name, number of reserved
 * physical eraseblocks, type, etc. The volume table is stored in the so-called
 * "layout volume".
 *
 * The layout volume is an internal volume which is organized as follows. It
 * consists of two logical eraseblocks - LEB 0 and LEB 1. Each logical
 * eraseblock stores one volume table copy, i.e. LEB 0 and LEB 1 duplicate each
 * other. This redundancy guarantees robustness to unclean reboots. The volume
 * table is basically an array of volume table records. Each record contains
 * full information about the volume and protected by a CRC checksum.
 *
 * The volume table is changed, it is first changed in RAM. Then LEB 0 is
 * erased, and the updated volume table is written back to LEB 0. Then same for
 * LEB 1. This scheme guarantees recoverability from unclean reboots.
 *
 * In this UBI implementation the on-flash volume table does not contain any
 * information about how many data static volumes contain. This information may
 * be found from the scanning data.
 *
 * But it would still be beneficial to store this information in the volume
 * table. For example, suppose we have a static volume X, and all its physical
 * eraseblocks became bad for some reasons. Suppose we are attaching the
 * corresponding MTD device, the scanning has found no logical eraseblocks
 * corresponding to the volume X. According to the volume table volume X does
 * exist. So we don't know whether it is just empty or all its physical
 * eraseblocks went bad. So we cannot alarm the user about this corruption.
 *
 * The volume table also stores so-called "update marker", which is used for
 * volume updates. Before updating the volume, the update marker is set, and
 * after the update operation is finished, the update marker is cleared. So if
 * the update operation was interrupted (e.g. by an unclean reboot) - the
 * update marker is still there and we know that the volume's contents is
 * damaged.
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "tool-util.h"
#include "ubi.h"
#include "linux/err.h"
#include "linux/crc32.h"

extern struct platform_info info;

/* Empty volume table record */
static struct ubi_vtbl_record empty_vtbl_record;

/**
 * ubi_change_vtbl_record - change volume table record.
 * @ubi: UBI device description object
 * @idx: table index to change
 * @vtbl_rec: new volume table record
 *
 * This function changes volume table record @idx. If @vtbl_rec is %NULL, empty
 * volume table record is written. The caller does not have to calculate CRC of
 * the record as it is done by this function. Returns zero in case of success
 * and a negative error code in case of failure.
 */
int ubi_change_vtbl_record(struct ubi_device *ubi, int idx,
			   struct ubi_vtbl_record *vtbl_rec)
{
	int i, err;
	uint32_t crc;
	struct ubi_volume *layout_vol;

	ubi_assert(idx >= 0 && idx < ubi->vtbl_slots);
	layout_vol = ubi->volumes[vol_id2idx(ubi, UBI_LAYOUT_VOLUME_ID)];

	if (!vtbl_rec)
		vtbl_rec = &empty_vtbl_record;
	else {
		crc = crc32(UBI_CRC32_INIT, vtbl_rec, UBI_VTBL_RECORD_SIZE_CRC);
		vtbl_rec->crc = __cpu_to_be32(crc);
	}

	memcpy(&ubi->vtbl[idx], vtbl_rec, sizeof(struct ubi_vtbl_record));
	printf("change vtbl--------------------\n");
	for (i = 0; i < UBI_LAYOUT_VOLUME_EBS; i++) {
		err = ubi_eba_unmap_leb(ubi, layout_vol, i);
		if (err)
			return err;

		err = ubi_eba_write_leb(ubi, layout_vol, i, ubi->vtbl, 0,
					ubi->vtbl_size, UBI_LONGTERM);
		if (err)
			return err;
	}
	return 0;
}

/**
 * create_vtbl - create a copy of volume table.
 * @ubi: UBI device description object
 * @si: scanning information
 * @copy: number of the volume table copy
 * @vtbl: contents of the volume table
 *
 * This function returns zero in case of success and a negative error code in
 * case of failure.
 */
static int create_vtbl(struct ubi_device *ubi, int copy, void *vtbl)
{
	int err, tries = 0;
	int pnum;
	static struct ubi_vid_hdr *vid_hdr;
	struct ubi_volume *vol;

	vol = ubi->volumes[vol_id2idx(ubi, UBI_LAYOUT_VOLUME_ID)];

	ubi_msg("create volume table (copy #%d)", copy + 1);

	vid_hdr = ubi_zalloc_vid_hdr(ubi);
	if (!vid_hdr)
		return -ENOMEM;

	pnum = ubi_wl_get_peb(ubi, 0);
	if (IS_ERR(pnum)) {
		err = PTR_ERR(pnum);
		goto out_free;
	}

	vol->eba_tbl[copy] = pnum;
	
	vid_hdr->vol_type = UBI_VID_DYNAMIC;
	vid_hdr->vol_id = __cpu_to_be32(UBI_LAYOUT_VOLUME_ID);
	vid_hdr->compat = UBI_LAYOUT_VOLUME_COMPAT;
	vid_hdr->data_size = vid_hdr->used_ebs =
			     vid_hdr->data_pad = __cpu_to_be32(0);
	vid_hdr->lnum = __cpu_to_be32(copy);
	vid_hdr->sqnum = __cpu_to_be64(next_sqnum(ubi));
	vid_hdr->leb_ver = __cpu_to_be32(0);

	/* The EC header is already there, write the VID header */
	err = ubi_io_write_vid_hdr(ubi, pnum, vid_hdr);
	if (err)
		goto out_free;

	/* Write the layout volume contents */
	err = ubi_io_write_data(ubi, vtbl, pnum, 0, ubi->vtbl_size);
	if (err)
		goto out_free;

	ubi_free_vid_hdr(ubi, vid_hdr);
	return err;


out_free:
	ubi_free_vid_hdr(ubi, vid_hdr);
	return err;
}

/**
 * create_empty_lvol - create empty layout volume.
 * @ubi: UBI device description object
 * @si: scanning information
 *
 * This function returns volume table contents in case of success and a
 * negative error code in case of failure.
 */
static struct ubi_vtbl_record *create_empty_lvol(struct ubi_device *ubi)
{
	int i;
	struct ubi_vtbl_record *vtbl;

	vtbl = malloc(ubi->vtbl_size);
	if (!vtbl)
		return ERR_PTR(-ENOMEM);
	memset(vtbl, 0, ubi->vtbl_size);
	ubi->vtbl = vtbl;

	for (i = 0; i < ubi->vtbl_slots; i++)
		memcpy(&vtbl[i], &empty_vtbl_record, UBI_VTBL_RECORD_SIZE);

	for (i = 0; i < UBI_LAYOUT_VOLUME_EBS; i++) {
		int err;

		err = create_vtbl(ubi, i, vtbl);
		if (err) {
			printf("[%s]: create vtbl %d failed\n", __func__, i);
			free(vtbl);
			return ERR_PTR(err);
		}
	}

	return vtbl;
}

int ubi_volume_init(struct ubi_device *ubi)
{
	int i, err;
	int reserved_pebs = 0;
	struct ubi_vtbl_record *vtbl;
	struct ubi_volume *vol;

	empty_vtbl_record.crc = __cpu_to_be32(0xf116c36b);

	/*
	 * The number of supported volumes is limited by the eraseblock size
	 * and by the UBI_MAX_VOLUMES constant.
	 */
	ubi->vtbl_slots = ubi->leb_size / UBI_VTBL_RECORD_SIZE;
	if (ubi->vtbl_slots > UBI_MAX_VOLUMES)
		ubi->vtbl_slots = UBI_MAX_VOLUMES;

	ubi->vtbl_size = ubi->vtbl_slots * UBI_VTBL_RECORD_SIZE;
	ubi->vtbl_size = ALIGN(ubi->vtbl_size, ubi->min_io_size);
	
	/* And add the layout volume */
	vol = malloc(sizeof(struct ubi_volume));
	if (!vol)
		return -ENOMEM;
	memset(vol, 0, sizeof(struct ubi_volume));

	vol->reserved_pebs = UBI_LAYOUT_VOLUME_EBS;
	vol->alignment = 1;
	vol->vol_type = UBI_DYNAMIC_VOLUME;
	vol->name_len = sizeof(UBI_LAYOUT_VOLUME_NAME) - 1;
	memcpy(vol->name, UBI_LAYOUT_VOLUME_NAME, vol->name_len + 1);
	vol->usable_leb_size = ubi->leb_size;
	vol->used_ebs = vol->reserved_pebs;
	vol->last_eb_bytes = vol->reserved_pebs;
	vol->used_bytes = (long long)vol->used_ebs * (ubi->leb_size - vol->data_pad);
	vol->vol_id = UBI_LAYOUT_VOLUME_ID;
	vol->ref_count = 1;
	
	vol->eba_tbl = malloc(vol->reserved_pebs * sizeof(int));
	for(i = 0; i < vol->reserved_pebs; i++)
		vol->eba_tbl[i] = UBI_LEB_UNMAPPED;

	ubi->volumes[vol_id2idx(ubi, vol->vol_id)] = vol;
	reserved_pebs += vol->reserved_pebs;
	ubi->vol_count += 1;
	vol->ubi = ubi;

	printf("[%s]: create layout volume for ubi part %s\n", __func__, ubi->ubi_name);

	if(info.ubilsbbakup){
		vol = malloc(sizeof(struct ubi_volume));
		if (!vol){
			printf("[%s]: malloc lsb backup volume failed\n", __func__);
			return -1;
		}

		vol->reserved_pebs = UBI_BACKUP_VOLUME_EBS;
		vol->alignment = 1;
		vol->vol_type = UBI_DYNAMIC_VOLUME;
		vol->name_len = sizeof(UBI_BACKUP_VOLUME_NAME) - 1;
		memcpy(vol->name, UBI_BACKUP_VOLUME_NAME, vol->name_len + 1);
		vol->usable_leb_size = ubi->leb_size;
		vol->used_ebs = vol->reserved_pebs;
		vol->last_eb_bytes = vol->reserved_pebs;
		vol->used_bytes = (long long)vol->used_ebs * (ubi->leb_size - vol->data_pad);
		vol->vol_id = UBI_BACKUP_VOLUME_ID;
		vol->ref_count = 1;

		ubi->volumes[vol_id2idx(ubi, vol->vol_id)] = vol;
		reserved_pebs += vol->reserved_pebs;
		ubi->vol_count += 1;
		vol->ubi = ubi;

		printf("[%s]: create lsb back up volume for ubi part %s, reserved peb=%d\n", __func__, ubi->ubi_name, vol->reserved_pebs);
	}

	ubi->rsvd_pebs += reserved_pebs;
	ubi->avail_pebs -= reserved_pebs;

	//create volume table
	ubi->vtbl = create_empty_lvol(ubi);
	return 0;
}
