/*
 * Copyright (c) International Business Machines Corp., 2006
 * Copyright (c) Nokia Corporation, 2007
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Author: Artem Bityutskiy (?и???кий ????м),
 *         Frank Haverkamp
 */

/*
 * This file includes UBI initialization and building of UBI devices.
 *
 * When UBI is initialized, it attaches all the MTD devices specified as the
 * module load parameters or the kernel boot parameters. If MTD devices were
 * specified, UBI does not attach any MTD device, but it is possible to do
 * later using the "UBI control device".
 *
 * At the moment we only attach UBI devices by scanning, which will become a
 * bottleneck when flashes reach certain large size. Then one may improve UBI
 * and add other methods, although it does not seem to be easy to do.
 */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include "../nand/mtd.h"
#include "ubi.h"
#include "common.h"

/* All UBI devices in system */
struct ubi_device *ubi_devices[UBI_MAX_DEVICES];


/**
 * attach_by_scanning - attach an MTD device using scanning method.
 * @ubi: UBI device descriptor
 *
 * This function returns zero in case of success and a negative error code in
 * case of failure.
 *
 * Note, currently this is the only method to attach UBI devices. Hopefully in
 * the future we'll have more scalable attaching methods and avoid full media
 * scanning. But even in this case scanning will be needed as a fall-back
 * attaching method if there are some on-flash table corruptions.
 */
static int attach_by_scanning(struct ubi_device *ubi)
{
	int err;

	ubi->bad_peb_count = 0;
	ubi->good_peb_count = ubi->peb_count;
	ubi->avail_pebs = ubi->peb_count;
	ubi->rsvd_pebs =0;
	ubi->max_ec = 0;
	ubi->mean_ec = 0;
	
	err = ubi_wl_init(ubi);
	if(err){
		printf("[%s]: failed\n", __func__);
		return err;
	}

	err = ubi_eba_init(ubi);
	if(err < 0){
		printf("[%s]: failed\n", __func__);
		return err;
	}
	
	err = ubi_volume_init(ubi);
	if(err){
		printf("[%s]: layout volume init failed\n", __func__);
		return err;
	}
	return 0;
}

/**
 * io_init - initialize I/O unit for a given UBI device.
 * @ubi: UBI device description object
 *
 * If @ubi->vid_hdr_offset or @ubi->leb_start is zero, default offsets are
 * assumed:
 *   o EC header is always at offset zero - this cannot be changed;
 *   o VID header starts just after the EC header at the closest address
 *     aligned to @io->hdrs_min_io_size;
 *   o data starts just after the VID header at the closest address aligned to
 *     @io->min_io_size
 *
 * This function returns zero in case of success and a negative error code in
 * case of failure.
 */
static int io_init(struct ubi_device *ubi)
{
	if (ubi->mtd->numeraseregions != 0) {
		/*
		 * Some flashes have several erase regions. Different regions
		 * may have different eraseblock size and other
		 * characteristics. It looks like mostly multi-region flashes
		 * have one "main" region and one or more small regions to
		 * store boot loader code or boot parameters or whatever. I
		 * guess we should just pick the largest region. But this is
		 * not implemented.
		 */
		ubi_err("multiple regions, not implemented");
		return -EINVAL;
	}

	if (ubi->vid_hdr_offset < 0)
		return -EINVAL;

       ubi->is_mlc = ubi->mtd->ismlc;
	/*
	 * Note, in this implementation we support MTD devices with 0x7FFFFFFF
	 * physical eraseblocks maximum.
	 */

	ubi->peb_size   = ubi->mtd->erasesize;
	ubi->peb_count  = mtd_div_by_eb(ubi->mtd->size, ubi->mtd);
	ubi->flash_size = ubi->mtd->size;

	if (ubi->mtd->block_isbad && ubi->mtd->block_markbad)
		ubi->bad_allowed = 1;

	ubi->min_io_size = ubi->mtd->writesize;
	ubi->hdrs_min_io_size = ubi->mtd->writesize >> ubi->mtd->subpage_sft;

	/*
	 * Make sure minimal I/O unit is power of 2. Note, there is no
	 * fundamental reason for this assumption. It is just an optimization
	 * which allows us to avoid costly division operations.
	 */
	if (!is_power_of_2(ubi->min_io_size)) {
		ubi_err("min. I/O unit (%d) is not power of 2",
			ubi->min_io_size);
		return -EINVAL;
	}

	/* Calculate default aligned sizes of EC and VID headers */
	ubi->ec_hdr_alsize = ALIGN(UBI_EC_HDR_SIZE, ubi->hdrs_min_io_size);
	ubi->vid_hdr_alsize = ALIGN(UBI_VID_HDR_SIZE, ubi->hdrs_min_io_size);

	if (ubi->vid_hdr_offset == 0)
		/* Default offset */
		ubi->vid_hdr_offset = ubi->vid_hdr_aloffset = ubi->ec_hdr_alsize;
	else {
		ubi->vid_hdr_aloffset = ubi->vid_hdr_offset & ~(ubi->hdrs_min_io_size - 1);
		ubi->vid_hdr_shift = ubi->vid_hdr_offset - ubi->vid_hdr_aloffset;
	}

	/* Similar for the data offset */
	ubi->leb_start = ubi->vid_hdr_offset + UBI_EC_HDR_SIZE;
	ubi->leb_start = ALIGN(ubi->leb_start, ubi->min_io_size);

	/* The shift must be aligned to 32-bit boundary */
	if (ubi->vid_hdr_shift % 4) {
		ubi_err("unaligned VID header shift %d", ubi->vid_hdr_shift);
		return -EINVAL;
	}

	/* Check sanity */
	if (ubi->vid_hdr_offset < UBI_EC_HDR_SIZE ||
	    ubi->leb_start < ubi->vid_hdr_offset + UBI_VID_HDR_SIZE ||
	    ubi->leb_start > ubi->peb_size - UBI_VID_HDR_SIZE ||
	    ubi->leb_start & (ubi->min_io_size - 1)) {
		ubi_err("bad VID header (%d) or data offsets (%d)", ubi->vid_hdr_offset, ubi->leb_start);
		return -EINVAL;
	}

	/*
	 * It may happen that EC and VID headers are situated in one minimal
	 * I/O unit. In this case we can only accept this UBI image in
	 * read-only mode.
	 */
	if (ubi->vid_hdr_offset + UBI_VID_HDR_SIZE <= ubi->hdrs_min_io_size) {
		ubi_warn("EC and VID headers are in the same minimal I/O unit, "
			 "switch to read-only mode");
		ubi->ro_mode = 1;
	}

	ubi->leb_size = ubi->peb_size - ubi->leb_start;

	if (!(ubi->mtd->flags & MTD_WRITEABLE)) {
		ubi_msg("MTD device %d is write-protected, attach in " 
			"read-only mode", ubi->mtd->index);
		ubi->ro_mode = 1;
	}
	return 0;
}

/**
 * find_mtd_device - open an MTD device by its name or number.
 * @mtd_dev: name or number of the device
 *
 * This function tries to open and MTD device described by @mtd_dev string,
 * which is first treated as an ASCII number, and if it is not true, it is
 * treated as MTD device name. Returns MTD device description object in case of
 * success and a negative error code in case of failure.
 */
struct mtd_info * open_mtd_device(const char *mtd_dev)
{
	struct mtd_info *mtd;
	int mtd_num;
	char *endp;

	mtd_num = strtoul(mtd_dev, &endp, 0);
	if (*endp != '\0' || mtd_dev == endp) {
		/*
		 * This does not look like an ASCII integer, probably this is
		 * MTD device name.
		 */
		mtd = get_mtd_device_nm(mtd_dev);
	} else
		mtd = get_mtd_device(NULL, mtd_num);
	return mtd;
}


/**
 * ubi_attach_mtd_dev - attach an MTD device.
 * @mtd_dev: MTD device description object
 * @ubi_num: number to assign to the new UBI device
 * @vid_hdr_offset: VID header offset
 *
 * This function attaches MTD device @mtd_dev to UBI and assign @ubi_num number
 * to the newly created UBI device, unless @ubi_num is %UBI_DEV_NUM_AUTO, in
 * which case this function finds a vacant device nubert and assings it
 * automatically. Returns the new UBI device number in case of success and a
 * negative error code in case of failure.
 *
 * Note, the invocations of this function has to be serialized by the
 * @ubi_devices_mutex.
 */
int ubi_attach_mtd_dev(struct mtd_info *mtd, int ubi_num, int vid_hdr_offset)
{
	struct ubi_device *ubi=NULL;
	int i=0, err=0;

	/*
	 * Check if we already have the same MTD device attached.
	 *
	 * Note, this function assumes that UBI devices creations and deletions
	 * are serialized, so it does not take the &ubi_devices_lock.
	 */
	for (i = 0; i < UBI_MAX_DEVICES; i++) {
		ubi = ubi_devices[i];
		if (ubi && mtd->index == ubi->mtd->index) {
			ubi_err("mtd%d is already attached to ubi%d",
				mtd->index, i);
			return -EEXIST;
		}
	}
	/*
	 * Make sure this MTD device is not emulated on top of an UBI volume
	 * already. Well, generally this recursion works fine, but there are
	 * different problems like the UBI module takes a reference to itself
	 * by attaching (and thus, opening) the emulated MTD device. This
	 * results in inability to unload the module. And in general it makes
	 * no sense to attach emulated MTD devices, so we prohibit this.
	 */
	if (mtd->type == MTD_UBIVOLUME) {
		ubi_err("refuse attaching mtd%d - it is already emulated on "
			"top of UBI", mtd->index);
		return -EINVAL;
	}

	if (ubi_num == UBI_DEV_NUM_AUTO) {
		/* Search for an empty slot in the @ubi_devices array */
		for (ubi_num = 0; ubi_num < UBI_MAX_DEVICES; ubi_num++)
			if (!ubi_devices[ubi_num])
				break;
		if (ubi_num == UBI_MAX_DEVICES) {
			dbg_err("only %d UBI devices may be created", UBI_MAX_DEVICES);
			return -ENFILE;
		}
	} else {
		if (ubi_num >= UBI_MAX_DEVICES)
			return -EINVAL;

		/* Make sure ubi_num is not busy */
		if (ubi_devices[ubi_num]) {
			dbg_err("ubi%d already exists", ubi_num);
			return -EEXIST;
		}
	}

	ubi = malloc(sizeof(struct ubi_device));
	if (!ubi)
		return -ENOMEM;

	// initialize ubi structure
	memset(ubi, 0, sizeof(struct ubi_device));

	ubi->mtd = mtd;
	ubi->ubi_num = ubi_num;
	ubi->vid_hdr_offset = vid_hdr_offset;
	ubi->autoresize_vol_id = -1;

	ubi_msg("attaching mtd%d to ubi%d", mtd->index, ubi_num);

	err = io_init(ubi);
	if (err)
		goto out_free;

	err = -ENOMEM;
	ubi->peb_buf1 = malloc(ubi->peb_size);
	if (!ubi->peb_buf1)
		goto out_free;


	err = attach_by_scanning(ubi);
	if (err) {
		dbg_err("failed to attach by scanning, error %d", err);
		goto out_free;
	}

	//from uif_init
	sprintf(ubi->ubi_name, UBI_NAME_STR "%d", ubi->ubi_num);

	ubi_msg("attached mtd%d to ubi%d", mtd->index, ubi_num);
	ubi_msg("MTD device name:            \"%s\"", mtd->name);
	ubi_msg("MTD device size:            %llu MiB", ubi->flash_size >> 20);
	ubi_msg("number of good PEBs:        %d", ubi->good_peb_count);
	ubi_msg("number of bad PEBs:         %d", ubi->bad_peb_count);
	ubi_msg("max. allowed volumes:       %d", ubi->vtbl_slots);
	ubi_msg("number of internal volumes: %d", UBI_INT_VOL_COUNT);
	ubi_msg("number of user volumes:     %d",
		ubi->vol_count - UBI_INT_VOL_COUNT);
	ubi_msg("available PEBs:             %d", ubi->avail_pebs);
	ubi_msg("total number of reserved PEBs: %d", ubi->rsvd_pebs);
	ubi_msg("number of PEBs reserved for bad PEB handling: %d",
		ubi->beb_rsvd_pebs);
	ubi_msg("max/mean erase counter: %d/%d", ubi->max_ec, ubi->mean_ec);

	ubi_devices[ubi_num] = ubi;
	return ubi_num;

out_detach:
	free(ubi->vtbl);
	
out_free:
	free(ubi->peb_buf1);
	free(ubi);
	return err;
}

void ubi_init(void)
{
	memset(ubi_devices, 0, sizeof(ubi_devices));
}

void ubi_exit(void)
{
	int i, j;
	for(i = 0; i < UBI_MAX_DEVICES; i++){
		if(ubi_devices[i]){
			if(ubi_devices[i]->mtd){
				if(ubi_devices[i]->mtd->name)
					free(ubi_devices[i]->mtd->name);
				free(ubi_devices[i]->mtd);
			}
			for(j = 0; j < UBI_MAX_VOLUMES+UBI_INT_VOL_COUNT; j++){
				if(ubi_devices[i]->volumes[j])
					free(ubi_devices[i]->volumes[j]);
			}
			if(ubi_devices[i]->wl_bitmap)
				free(ubi_devices[i]->wl_bitmap);
			free(ubi_devices[i]);
		}
	}
}







extern int ubi_start_update(struct ubi_device * ubi, struct ubi_volume * vol, long long bytes);
extern int ubi_more_update_data(struct ubi_device * ubi, struct ubi_volume * vol, const void * buf, int count);
/////////////////////////////////////////////////////////////
///////////////////////test UBI///////////////////////////////
void test_ubi1(void){
	extern struct mtd_info mtdinfo;
	struct ubi_mkvol_req req;
	int i,len;
	unsigned char *buf0, *buf1;

	for(i=0;i<UBI_MAX_DEVICES;i++)
		memset(&ubi_devices[i],0,sizeof(struct ubi_device));
	
	req.vol_type = UBI_DYNAMIC_VOLUME;
	req.vol_id = UBI_VOL_NUM_AUTO;
	req.alignment = 1;
	req.bytes = 0x1000000;
	strcpy(req.name, "TestVolume1");
	req.name_len = strlen(req.name);
	req.name[req.name_len] = '\0';
	req.padding1 = 0;

	printf("1-----------------\n");
	ubi_attach_mtd_dev(&mtdinfo, UBI_DEV_NUM_AUTO, 0);
	printf("2-----------------\n");
	ubi_create_volume(ubi_devices[0], &req);
	printf("3-----------------\n");

	req.vol_type = UBI_DYNAMIC_VOLUME;
	req.vol_id = UBI_VOL_NUM_AUTO;
	req.alignment = 1;
	req.bytes = 0x1000000;
	strcpy(req.name, "TestVolume2");
	req.name_len = strlen(req.name);
	req.name[req.name_len] = '\0';
	req.padding1 = 0;
	printf("4-----------------\n");
	ubi_create_volume(ubi_devices[0], &req);
	printf("5-----------------\n");

	len = 3*ubi_devices[0]->leb_size;
	buf0=malloc(len);
	buf1=malloc(len);
	for(i=0;i<len/512;i++){
		memset(buf0+i*512,i+1,512);
		memset(buf1+i*512,i+2,512);
	}
	printf("6-----------------\n");
	ubi_start_update(ubi_devices[0], ubi_devices[0]->volumes[0],len);
	ubi_start_update(ubi_devices[0], ubi_devices[0]->volumes[1],len);
	printf("7-----------------\n");
	ubi_more_update_data(ubi_devices[0], ubi_devices[0]->volumes[0], buf0, len/2);
	ubi_more_update_data(ubi_devices[0], ubi_devices[0]->volumes[1], buf1, len/2);
	printf("8-----------------\n");
	ubi_more_update_data(ubi_devices[0], ubi_devices[0]->volumes[0], buf0 + len/2, len/2);
	ubi_more_update_data(ubi_devices[0], ubi_devices[0]->volumes[1], buf1 + len/2, len/2);
	printf("----------%d %d\n", ubi_devices[0]->volumes[1]->upd_marker, len/2);
	
}

#include "jiffs2/load_kernel.h"
#include "../nand/partitions.h"
void test_ubi_mtdpart(void)
{
	extern struct mtd_info mtdinfo;
	struct mtd_info *info;
	struct mtd_partition mtd_part;
	struct ubi_mkvol_req req;
	int i,len;
	unsigned char *buf0, *buf1;
	struct part_info *part;
	struct mtd_dev *dev;
	u8 pnum;
	char buff1[64], buff2[64];
	
	printf("-----------------------------test ubi mtdpart----------------------------\n");
	printf("-----------------------------test ubi mtdpart----------------------------\n");
	printf("-----------------------------test ubi mtdpart----------------------------\n");
	
	for(i=0;i<UBI_MAX_DEVICES;i++)
		memset(&ubi_devices[i],0,sizeof(struct ubi_device));


	//ubi part UBI

	if (find_dev_and_part("UBI", &dev, &pnum, &part) != 0)
		return 1;
	
	printf("A------------%s %d\n", part->name, pnum);

	memset(buff1, 0, sizeof(buff1));
	sprintf(buff1, "mtd=%d", pnum);
	memset(&mtd_part, 0, sizeof(mtd_part));
	mtd_part.name = buff1;
	mtd_part.size = part->size;
	mtd_part.offset = part->offset;
	add_mtd_partitions(&mtdinfo, &mtd_part, 1);

	
	req.vol_type = UBI_DYNAMIC_VOLUME;
	req.vol_id = UBI_VOL_NUM_AUTO;
	req.alignment = 1;
	req.bytes = 0x1000000;
	strcpy(req.name, "TestVolume1");
	req.name_len = strlen(req.name);
	req.name[req.name_len] = '\0';
	req.padding1 = 0;

	info = open_mtd_device(mtd_part.name);
		printf("@@@@@@@@@@@@@@@ %d %s\n", info->index, info->name);
	ubi_attach_mtd_dev(info, UBI_DEV_NUM_AUTO, 0);
	ubi_create_volume(ubi_devices[0], &req);

	len = 3*ubi_devices[0]->leb_size;
	buf0=malloc(len);
	buf1=malloc(len);
	for(i=0;i<len/512;i++){
		memset(buf0+i*512,i+1,512);
		memset(buf1+i*512,i+2,512);
	}
//	ubi_start_update(ubi_devices[0], ubi_devices[0]->volumes[0],(long long) len);
//	ubi_more_update_data(ubi_devices[0], ubi_devices[0]->volumes[0], buf1, (long long)len/2);
//	ubi_more_update_data(ubi_devices[0], ubi_devices[0]->volumes[0], buf1 + len/2, (long long)len/2);


////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////
	//ubi part UBIRO

	if (find_dev_and_part("UBIRO", &dev, &pnum, &part) != 0)
		return 1;
	
	printf("B------------%s %d\n", part->name, pnum);

	memset(buff2, 0, sizeof(buff2));
	sprintf(buff2, "mtd=%d", pnum);
	memset(&mtd_part, 0, sizeof(mtd_part));
	mtd_part.name = buff2;
	mtd_part.size = part->size;
	mtd_part.offset = part->offset;
	add_mtd_partitions(&mtdinfo, &mtd_part, 1);
printf("C******** %s %d\n",mtd_part.name, pnum);
	
	req.vol_type = UBI_DYNAMIC_VOLUME;
	req.vol_id = UBI_VOL_NUM_AUTO;
	req.alignment = 1;
	req.bytes = 0x200000;
	strcpy(req.name, "UBIROTestVolume");
	req.name_len = strlen(req.name);
	req.name[req.name_len] = '\0';
	req.padding1 = 0;

	info = open_mtd_device(mtd_part.name);
	printf("D******** %s %s %d\n", info->name, mtd_part.name, info->index);
	ubi_attach_mtd_dev(info, UBI_DEV_NUM_AUTO, 0);
	ubi_create_volume(ubi_devices[1], &req);
	len = 3*ubi_devices[1]->leb_size;
	buf0=malloc(len);
	buf1=malloc(len);
	for(i=0;i<len/512;i++){
		memset(buf0+i*512,i+1,512);
		memset(buf1+i*512,i+2,512);
	}
	ubi_start_update(ubi_devices[1], ubi_devices[1]->volumes[0],(long long) len);
	ubi_more_update_data(ubi_devices[1], ubi_devices[1]->volumes[0], buf1, (long long)len/2);
	ubi_more_update_data(ubi_devices[1], ubi_devices[1]->volumes[0], buf1 + len/2, (long long)len/2);

}
/////////////////////////////////////////////////////////////
///////////////////////test UBI///////////////////////////////
