/*
 *  drivers/mtd/nand.c
 *
 *  Overview:
 *   This is the generic MTD driver for NAND flash devices. It should be
 *   capable of working with almost all NAND chips currently available.
 *   Basic support for AG-AND chips is provided.
 *
 *	Additional technical information is available on
 *	http://www.linux-mtd.infradead.org/doc/nand.html
 *
 *  Copyright (C) 2000 Steven J. Hill (sjhill@realitydiluted.com)
 *		  2002-2006 Thomas Gleixner (tglx@linutronix.de)
 *
 *  Credits:
 *	David Woodhouse for adding multichip support
 *
 *	Aleph One Ltd. and Toby Churchill Ltd. for supporting the
 *	rework for 2K page size chips
 *
 *  TODO:
 *	Enable cached programming for 2k page size chips
 *	Check, if mtd->ecctype should be set to MTD_ECC_HW
 *	if we have HW ecc support.
 *	The AG-AND chips have nice features for speed improvement,
 *	which are not supported yet. Read / program 4 pages in one go.
 *	BBT table is not serialized, has to be fixed
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include "nand/mtd.h"
#include "linux/mtd/nand.h"
#include "tool-util.h"
#include "environment.h"
#include "../../mstar/unfd/inc/common/drvNAND.h"
#include "../../mstar/spinand/inc/common/drvSPINAND.h"

/**
 * nand_default_block_markbad - [DEFAULT] mark a block bad
 * @mtd:	MTD device structure
 * @ofs:	offset from device start
 *
 * This is the default implementation, which can be overridden by
 * a hardware specific driver.
*/
extern struct platform_info info;

extern unsigned char *data_buf;
extern unsigned char *oob_buf;

static unsigned char *mtd_mainbuf;
static unsigned char *mtd_sparebuf;

extern void board_nand_exit(void);
extern void board_spinand_exit(void);

struct mtd_info mtdinfo;

/*
 *support env in raw nand
 *save env in nand need blocks
 */
unsigned int need_blk = 0;
U16 u16_ENV_Star_Blk;
U16 u16_ENV_Blk_Region;

/**
 * nand_transfer_oob - [Internal] Transfer oob to client buffer
 * @chip:	nand chip structure
 * @oob:	oob destination address
 * @ops:	oob ops structure
 * @len:	size of oob to transfer
 */
static uint8_t *nand_transfer_oob( uint8_t *oob, struct mtd_oob_ops *ops, size_t len)
{
	switch(ops->mode){
	case MTD_OOB_AUTO:
	case MTD_OOB_PLACE:
	case MTD_OOB_RAW:
		memcpy(oob, mtd_sparebuf + ops->ooboffs, len);
		break;
	}
	return oob + len;
}

/**
 * nand_fill_oob - [Internal] Transfer client buffer to oob
 * @chip:	nand chip structure
 * @oob:	oob data buffer
 * @ops:	oob ops structure
 */
static uint8_t *nand_fill_oob(uint8_t *oob, struct mtd_oob_ops *ops)
{
	size_t len = ops->ooblen;

	switch(ops->mode) {
	case MTD_OOB_PLACE:
	case MTD_OOB_RAW:
	case MTD_OOB_AUTO:
		memcpy(mtd_sparebuf + ops->ooboffs, oob, len);
		break;
	}
	return oob + len;
}

static void nand_sync(struct mtd_info *mtd)
{
	return;
}

/**
 * nand_block_isbad - [MTD Interface] Check if block at offset is bad
 * @mtd:	MTD device structure
 * @offs:	offset relative to mtd start
 */
static int nand_block_isbad(struct mtd_info *mtd, loff_t offs)
{
	/* Check for invalid offset */
	if (offs > mtd->size){
		printf("[%s]: Offset exceed nand size\n", __func__);
		return -1;
	}

	return 0;
}

/**
 * nand_block_markbad - [MTD Interface] Mark block at the given offset as bad
 * @mtd:	MTD device structure
 * @ofs:	offset relative to mtd start
 */
static int nand_block_markbad(struct mtd_info *mtd, loff_t ofs)
{
	printf("[%s]: OK\n", __func__);
	return 0;
}

/**
 * nand_block_markbad_e - [MTD Interface] Mark block at the given offset as bad, for debug
 * @mtd:	MTD device structure
 * @ofs:	offset relative to mtd start
 */
static int nand_block_markbad_e(struct mtd_info *mtd, loff_t ofs)
{
	printf("[%s]: OK\n", __func__);
	return 0;
}

static int nand_default_block_markbad(struct mtd_info *mtd, loff_t ofs)
{
	printf("[%s]: OK\n", __func__);
	return 0;
}

/**
 * nand_default_block_markbad_e - [DEFAULT] mark a block bad, for debug which does not write spare place.
 * @mtd:    MTD device structure
 * @ofs:    offset from device start
 *
 * This is the default implementation, which can be overridden by
 * a hardware specific driver.
*/
static int nand_default_block_markbad_e(struct mtd_info *mtd, loff_t ofs)
{
	printf("[%s]: OK\n", __func__);
	return 0;
}

static int nand_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	int ret;
	uint64_t addr = instr->addr;
	uint64_t len = instr->len, size;
	NAND_DRIVER *pNandDrv = (NAND_DRIVER *)drvNAND_get_DrvContext_address();
	while(len > 0){
		U32 RowIdx = addr / mtd->writesize;
		size = mtd->erasesize - (addr & (mtd->erasesize - 1));
		size = (size < len) ? size : len;

		if(info.nandtype == NAND){
			if(pNandDrv->u8_PlaneCnt > 1)
				ret = NC_EraseBlk2P(RowIdx);
			else
				ret = NC_EraseBlk(RowIdx);
		}
		else if(info.nandtype == SPINAND){
			ret = MDrv_SPINAND_BLOCK_ERASE(RowIdx);
		}
		if(ret){
			printf("[%s]: erase addr 0x%llX failed\n", __func__, addr);
			return -1;
		}
		len -= size;
		addr += size;
	}
	return 0;
}

static int nand_do_read_ops(struct mtd_info *mtd, loff_t from, struct mtd_oob_ops *ops)
{
	int ret, col;
	int len, ooblen, readlen, oobreadline;
	uint8_t *oob, *buf;
	U32 u32_RowIdx;
	NAND_DRIVER *pNandDrv = (NAND_DRIVER *)drvNAND_get_DrvContext_address();

    memset(mtd_mainbuf, 0xFF, mtd->writesize);
	memset(mtd_sparebuf, 0xFF, mtd->oobsize);

	buf = ops->datbuf;
	oob = ops->oobbuf;
	len = ops->len;
	ooblen = ops->ooblen;
	while(len > 0){
		col = from & (mtd->writesize - 1);
		readlen = mtd->writesize - col;
		readlen = (readlen < len) ? readlen : len;

		u32_RowIdx = from / mtd->writesize;
		if(info.nandtype == NAND){
			if(pNandDrv->u8_PlaneCnt > 1)
				ret = NC_ReadPages2P(u32_RowIdx, mtd_mainbuf, mtd_sparebuf, 1);
			else
				ret = NC_ReadPages(u32_RowIdx, mtd_mainbuf, mtd_sparebuf, 1);
		}
		else if(info.nandtype == SPINAND){
			ret = MDrv_SPINAND_Read(u32_RowIdx, mtd_mainbuf, mtd_sparebuf);
		}
		if(ret){
			printf("[%s]: read page %d failed\n", __func__, u32_RowIdx);
			return -1;
		}

		memcpy(buf, mtd_mainbuf + col, readlen);

		buf += readlen;

		if(oob){

			oobreadline = (mtd->oobsize < ooblen) ? mtd->oobsize : ooblen;
			if(ops->mode != MTD_OOB_RAW){
				oob = nand_transfer_oob(oob, ops, oobreadline);
			}
			else{
				buf = nand_transfer_oob(buf, ops, oobreadline);
			}
			ooblen -= oobreadline;
		}

		len -= readlen;
		from += readlen;
	}

	ops->retlen = ops->len - len;
	ops->oobretlen = ops->ooblen- ooblen;
	return 0;
}

static int nand_do_read_oob(struct mtd_info *mtd, loff_t from, struct mtd_oob_ops *ops)
{
	int ret, col;
	int ooblen, oobreadline;
	uint8_t *buf;
	U32 u32_RowIdx;
	NAND_DRIVER *pNandDrv = (NAND_DRIVER *)drvNAND_get_DrvContext_address();

    memset(mtd_sparebuf, 0xFF, mtd->oobsize);

	buf = ops->oobbuf;
	ooblen = ops->ooblen;
	u32_RowIdx = from / mtd->writesize;
	while(ooblen > 0){
		if(info.nandtype == NAND){
			if(pNandDrv->u8_PlaneCnt > 1)
				ret = NC_WritePages2P(u32_RowIdx, mtd_mainbuf, mtd_sparebuf, 1);
			else
				ret = NC_ReadPages(u32_RowIdx, mtd_mainbuf, mtd_sparebuf, 1);
		}
		else if(info.nandtype == SPINAND){
			ret = MDrv_SPINAND_Read(u32_RowIdx, mtd_mainbuf, mtd_sparebuf);
		}
		if(ret){
			printf("[%s]: read page %d failed\n", __func__, u32_RowIdx);
			return -1;
		}

		if(buf){
			oobreadline = (mtd->oobsize < ooblen) ? mtd->oobsize : ooblen;
			buf = nand_transfer_oob(buf, ops, oobreadline);
		}
		ooblen -= oobreadline;
		u32_RowIdx++;
	}

	ops->oobretlen = ops->ooblen -ooblen;
	return 0;

}

/**
 * nand_read_oob - [MTD Interface] NAND read data and/or out-of-band
 * @mtd:	MTD device structure
 * @from:	offset to read from
 * @ops:	oob operation description structure
 *
 * NAND read data and/or out-of-band data
 */
static int nand_read_oob (struct mtd_info *mtd, loff_t from, struct mtd_oob_ops *ops)
{
	int ret;

	ops->retlen = 0;

	/* Do not allow reads past end of device */
	if (ops->datbuf && (from + ops->len) > mtd->size) {
		printf("[%s]: Attempt read beyond end of device\n", __func__);
		return -EINVAL;
	}

	switch(ops->mode) {
	case MTD_OOB_PLACE:
	case MTD_OOB_AUTO:
	case MTD_OOB_RAW:
		break;

	default:
		printf("[%s]: UnKnown oob mode\n", __func__);
		return -1;
	}

	if (!ops->datbuf){
		ret = nand_do_read_oob(mtd, from, ops);
		}
	else
		ret = nand_do_read_ops(mtd, from, ops);
	if(ret < 0){
		printf("[%s]: read failed\n", __func__);
	}

	return ret;
}

/**
 * nand_read - [MTD Interface] MTD compability function for nand_do_read_ecc
 * @mtd:	MTD device structure
 * @from:	offset to read from
 * @len:	number of bytes to read
 * @retlen:	pointer to variable to store the number of read bytes
 * @buf:	the databuffer to put data
 *
 * Get hold of the chip and call nand_do_read
 */
static int nand_read(struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen, uint8_t *buf)
{
	struct mtd_oob_ops ops;
	int ret=0;

	/* Do not allow reads past end of device */
	if ((from + len) > mtd->size){
		printf("[%s], Error, Invalid size, from=%x, len=%x, mtd->size=%x", __func__, from, len, mtd->size);
		return -EINVAL;
	}
	if (!len)
		return 0;

	memset(&ops, 0, sizeof(struct mtd_oob_ops));
	ops.datbuf = buf;
	ops.len= len;
	ops.oobbuf = NULL;

	ret = nand_do_read_ops(mtd, from, &ops);
	if(ret < 0){
		printf("[%s]: read failed\n", __func__);
	}
	*retlen = ops.retlen;
	return ret;
}

static int nand_do_write_ops(struct mtd_info *mtd, loff_t to, struct mtd_oob_ops *ops)
{
	int ret, bytes, column;
	uint32_t writelen = ops->len;
	uint8_t *oob = ops->oobbuf;
	uint8_t *buf = ops->datbuf;
	U32 u32_RowIdx;

	memset(mtd_sparebuf, 0xFF, mtd->oobsize);

	while(writelen > 0){
		column = to & (mtd->writesize - 1);
		bytes = mtd->writesize - column;
		bytes = (bytes < writelen) ? bytes : writelen;
		u32_RowIdx = to / mtd->writesize;
		if (column || bytes < mtd->writesize) {
			memset(mtd_mainbuf, 0xFF, mtd->writesize);
		}
		memcpy(mtd_mainbuf + column, buf, bytes);
		if(oob){
			memset(mtd_sparebuf, 0xFF, mtd->oobsize);
			oob = nand_fill_oob(oob, ops);
		}
		if(info.nandtype == NAND){
			NAND_DRIVER *pNandDrv = (NAND_DRIVER*)drvNAND_get_DrvContext_address();
			if(pNandDrv->u8_PlaneCnt > 1){
				ret = NC_WritePages2P(u32_RowIdx, mtd_mainbuf, mtd_sparebuf, 1);
			}
			else{

				ret = NC_WritePages(u32_RowIdx, mtd_mainbuf, mtd_sparebuf, 1);
			}
		}
		else if(info.nandtype == SPINAND){
			ret = MDrv_SPINAND_Write(u32_RowIdx, mtd_mainbuf, mtd_sparebuf);
		}
		if(ret){
			printf("[%s]: read failed\n", __func__);
			return -1;
		}
		to += bytes;
		buf += bytes;
		writelen -= bytes;
	}

	ops->retlen = ops->len - writelen;
	ops->oobretlen = ops->ooblen;

	return 0;
}

static int nand_do_write_oob(struct mtd_info *mtd, loff_t to, struct mtd_oob_ops *ops)
{
	int ret;
	U32 u32_RowIdx;

	u32_RowIdx = to / mtd->writesize;

	memset(mtd_mainbuf, 0xFF, mtd->writesize);
	memset(mtd_sparebuf, 0xFF, mtd->oobsize);
	if(ops->oobbuf)
		nand_fill_oob(ops->oobbuf, ops);

	if(info.nandtype == NAND){
		NAND_DRIVER *pNandDrv = (NAND_DRIVER*)drvNAND_get_DrvContext_address();
		if(pNandDrv->u8_PlaneCnt > 1){
			ret = NC_WritePages2P(u32_RowIdx, mtd_mainbuf, mtd_sparebuf, 1);
		}
		else{
			ret = NC_WritePages(u32_RowIdx, mtd_mainbuf, mtd_sparebuf, 1);
		}
	}
	else if(info.nandtype == SPINAND){
		ret = MDrv_SPINAND_Write(u32_RowIdx, mtd_mainbuf, mtd_sparebuf);
	}

	ops->oobretlen = ops->ooblen;
	return 0;
}


/**
 * nand_write - [MTD Interface] NAND write with ECC
 * @mtd:	MTD device structure
 * @to:		offset to write to
 * @len:	number of bytes to write
 * @retlen:	pointer to variable to store the number of written bytes
 * @buf:	the data to write
 *
 * NAND write with ECC
 */
static int nand_write(struct mtd_info *mtd, loff_t to, size_t len, size_t *retlen, const uint8_t *buf)
{
	struct mtd_oob_ops ops;
	U32 u32_StartSector, u32_SectorCnt, ret;
	char cmd[512];
	char *ftl_SectorCnt, *ftl_StartSector;

	/* Do not allow reads past end of device */
	if ((to + len) > mtd->size){
		printf("[%s]: Attempt write beyond end of device\n", __func__);
		return -EINVAL;
	}
	if (!len)
		return 0;

	memset(&ops, 0, sizeof(struct mtd_oob_ops));
	ops.len = len;
	ops.datbuf = (uint8_t *)buf;
	ops.oobbuf = NULL;

	if(info.set_ftl == 1)
	{

		ftl_StartSector = getenv_uboot("ftl_StartSector");
		if (u32_StartSector < 0){
		printf("[%s]: please set ftl_StartSector\n", __func__);
		return -EINVAL;
		}
		u32_StartSector = atoi(ftl_StartSector);
		ftl_SectorCnt = getenv_uboot("ftl_SectorCnt");
		if (u32_SectorCnt < 0){
		printf("[%s]: please set ftl_SectorCnt\n", __func__);
		return -EINVAL;
		}
		u32_SectorCnt = atoi(ftl_SectorCnt);
		printf("this project have set ftl ftl_StartSector=%d,ftl_SectorCnt=%d\n",u32_StartSector,u32_SectorCnt);
		#if defined(__VER_UNFD_FTL__) && __VER_UNFD_FTL__
		drvNAND_WriteFAT((U32 *)buf, u32_StartSector, u32_SectorCnt);
		#endif
		return 0;
	}
	else
	{
		ret = nand_do_write_ops(mtd, to, & ops);
		*retlen = ops.retlen;
		if(ret < 0){
		printf("[%s]: write failed\n", __func__);
		}

	}

	return ret;
}


/**
 * nand_write_oob - [MTD Interface] NAND write data and/or out-of-band
 * @mtd:	MTD device structure
 * @to:		offset to write to
 * @ops:	oob operation description structure
 */
static int nand_write_oob(struct mtd_info *mtd, loff_t to,
			  struct mtd_oob_ops *ops)
{
	int ret ;

	ops->retlen = 0;

	/* Do not allow writes past end of device */
	if (ops->datbuf && (to + ops->len) > mtd->size) {
		printf("[%s]: Attempt write beyond end of device\n", __func__);
		return -EINVAL;
	}

	switch(ops->mode){
	case MTD_OOB_PLACE:
	case MTD_OOB_AUTO:
	case MTD_OOB_RAW:
		break;
	default:
		printf("[%s]: UnKnown oob mode\n", __func__);
		return -1;
	}

	if (!ops->datbuf)
		ret = nand_do_write_oob(mtd, to, ops);
	else
		ret = nand_do_write_ops(mtd, to, ops);

	return ret;
}
static void nand_get_flash_type_mstar_nand(struct mtd_info *mtd)
{
	NAND_DRIVER *pNandDrv = (NAND_DRIVER*)drvNAND_get_DrvContext_address();

	mtd->writesize = pNandDrv->u16_PageByteCnt * pNandDrv->u8_PlaneCnt;
	mtd->oobblock = pNandDrv->u16_PageByteCnt * pNandDrv->u8_PlaneCnt;  //kylin ??
	mtd->oobsize = pNandDrv->u16_SpareByteCnt * pNandDrv->u8_PlaneCnt;
	mtd->erasesize = pNandDrv->u16_BlkPageCnt * pNandDrv->u16_PageByteCnt * pNandDrv->u8_PlaneCnt;
	mtd->size = (uint64_t)pNandDrv->u16_PageByteCnt * pNandDrv->u16_BlkPageCnt * pNandDrv->u16_BlkCnt;
	mtd->ismlc = pNandDrv->u8_CellType ? 1 : 0;
}

static void nand_get_flash_type_mstar_spinand(struct mtd_info *mtd)
{
	SPINAND_FLASH_INFO_t *pSpinandInfo = (SPINAND_FLASH_INFO_t *)MDrv_SPINAND_get_SpinandInfo_address();

	mtd->writesize = pSpinandInfo->u16_PageByteCnt;
	mtd->oobblock = pSpinandInfo->u16_PageByteCnt;
	mtd->oobsize = pSpinandInfo->u16_SpareByteCnt;
	mtd->erasesize = pSpinandInfo->u16_PageByteCnt * pSpinandInfo->u16_BlkPageCnt;
	mtd->size = (uint64_t)pSpinandInfo->u16_PageByteCnt * pSpinandInfo->u16_BlkPageCnt * pSpinandInfo->u16_BlkCnt;
	mtd->ismlc = 0;
}

/**
 * nand_scan - [NAND Interface] Scan for the NAND device
 * @mtd:	MTD device structure
 * @maxchips:	Number of chips to scan for
 *
 * This fills out all the uninitialized function pointers
 * with the defaults.
 * The flash ID is read and the mtd/chip structures are
 * filled with the appropriate values.
 * The mtd->owner field must be set to the module of the caller
 *
 */
int nand_scan(struct mtd_info *mtd)
{
	struct nand_chip *this;

	if(info.nandtype == NAND){
		printf("Nand Scan:\n");
		nand_get_flash_type_mstar_nand(mtd);
	}
	else if(info.nandtype == SPINAND){
		printf("Spinand Scan:\n");
		nand_get_flash_type_mstar_spinand(mtd);
	}

	//size of data_buf & oob_buf is the same as block size;
	data_buf = (unsigned char *)malloc(mtd->erasesize);
	oob_buf = (unsigned char *)malloc(mtd->erasesize);
	if(!data_buf || !oob_buf){
		printf("[%s]: malloc nand util buffer failed\n", __func__);
		return -1;
	}

	//buffer used for mtd driver
	mtd_mainbuf= (unsigned char *)malloc(mtd->writesize);
	mtd_sparebuf= (unsigned char *)malloc(mtd->oobsize);
	if(!mtd_mainbuf || !mtd_sparebuf){
		printf("[%s]: malloc mtd buffer failed\n", __func__);
		return -1;
	}

	this = (struct nand_chip *)malloc(sizeof(struct nand_chip));
	memset(this, 0, sizeof(struct nand_chip));
	this->chipsize = mtd->size;
	this->phys_erase_shift = ffs(mtd->erasesize) - 1;
	mtd->priv = this;

	/* Fill in remaining MTD driver data */
	mtd->type = MTD_NANDFLASH;
	mtd->flags = MTD_CAP_NANDFLASH;
	mtd->lock = NULL;
	mtd->unlock = NULL;
	mtd->block_isbad = nand_block_isbad;
	mtd->block_markbad = nand_block_markbad;
	mtd->block_markbad_e = nand_block_markbad_e;
	mtd->point = NULL;
	mtd->unpoint = NULL;
	mtd->sync = nand_sync;
	mtd->erase = nand_erase;
	mtd->read = nand_read;
	mtd->read_oob = nand_read_oob;
	mtd->write = nand_write;
	mtd->write_oob = nand_write_oob;

	this->blocknum = this->chipsize >> this->phys_erase_shift;
	/* set bbt block number to 0.8% of total blocks, or blocks * (2 / 256) */
	mtd->bbt_block_num = ((this->blocknum >> 8) * 2);

	/* Set the bad block position */
	this->badblockpos = mtd->writesize > 512 ?
	    NAND_LARGE_BADBLOCK_POS : NAND_SMALL_BADBLOCK_POS;


	printf("###############################################\n");
	printf("#        MTD  INFO                            #\n");
	printf("###############################################\n");
	printf("   writesize: 0x%X\n", mtd->writesize);
	printf("   oobsize: 0x%X\n", mtd->oobsize);
	printf("   erasesize: 0x%X\n", mtd->erasesize);
	printf("   size: 0x%llX\n", mtd->size);
	printf("\n\n");

	//support env stored in raw nand
	/*
	if(info.envtype == NANDENV&&info.nandtype != SPINAND){
		// MSTAR start
		extern unsigned int cfg_env_offset;
		//cfg_env_offset=this->chipsize-(CONFIG_ENV_SIZE + 4*mtd->erasesize);
		need_blk = CONFIG_ENV_SIZE / mtd->erasesize;  //kylin ??
		if(CONFIG_ENV_SIZE % mtd->erasesize)
			need_blk += 1;

		cfg_env_offset = this->chipsize - ((need_blk * 2 + 2) + mtd->bbt_block_num) * mtd->erasesize;

		u16_ENV_Star_Blk = cfg_env_offset / mtd->erasesize;
		u16_ENV_Blk_Region = need_blk * 2 + 2;
		// MSTAR end
		printf("%s:%d [%d:%x:%x:%d:%d]\n", __func__, __LINE__, need_blk, CONFIG_ENV_SIZE,cfg_env_offset,u16_ENV_Star_Blk,u16_ENV_Blk_Region);
	}
	*/

	return 0;
}

void nand_exit(void)
{
	if(mtd_mainbuf)
		free(mtd_mainbuf);
	if(mtd_sparebuf)
		free(mtd_sparebuf);
	if(data_buf)
		free(data_buf);
	if(oob_buf)
		free(oob_buf);

	if(info.nandtype == NAND)
		board_nand_exit();
	else if(info.nandtype == SPINAND)
		board_spinand_exit();
}

#define MTD_TEST_R
void test_mtd(void)
{
#if defined (MTD_TEST_W)
	size_t len, retlen;
	unsigned char *buf;
	int i, j;
	len = mtdinfo.erasesize * 1 + mtdinfo.writesize * 2 + 0x100;
	buf = (unsigned char *)malloc(len);
	memset(buf, 0x5A, len);
	for(i = 0; i <  len / 256; i+=8){
		for(j = 0; j < 8; j++)
			memset(buf + (i + j)* 256, j + i/8  + 1,  mtdinfo.writesize);
	}
	mtdinfo.write(&mtdinfo, 0x2000, len, &retlen, buf);
#elif defined(MTD_TEST_E)
	size_t len, retlen;
	instr.addr = 0x100;
	instr.len = len;
	mtdinfo.erase(&mtdinfo, &instr);
#elif defined(MTD_TEST_R)
	size_t len, retlen;
	unsigned char *buf;
	int i;
	len = mtdinfo.erasesize * 1 + mtdinfo.writesize * 2 + 0x100;
	buf = (unsigned char *)malloc(len);
	mtdinfo.read(&mtdinfo, 0x2000, len, &retlen, buf);
		{
			int a, b;
			for(a=0;a<len;a+=16){
				for(b=0;b<16;b++){
					printf("0x%02X ", buf[a+b]);
				}
				printf("\n");

			}
			printf("\n\n\n");
			for(a=0;a<128;a+=16){
				for(b=0;b<16;b++){
					printf("0x%02X ", mtd_sparebuf[a+b]);
				}
				printf("\n");
			}
			printf("\n\n\n");
		}


#endif
}

void test_mtd2(void)
{
	unsigned char *data ,*oob,path[128];
	size_t retlen;
	int len = 913408, llen=2*1024*1024;
	int fd;
	data = (unsigned char *)malloc(llen);
	oob = (unsigned char *)malloc(llen);

	nand_read(&mtdinfo, 0, 913408, &retlen, data);
	memset(path,0,sizeof(path));
	sprintf(path, "%s", "/home/tao.zhou/xiaohui.zhu/readtest.bin");
	fd=open(path, O_RDWR | O_CREAT);
	if(fd<0){
		printf("open failed %s\n", path);
		return;
	}
	printf("tttt\n");
	write(fd, data, len);
	close(fd);
}


void test_mtd3(void)
{
	unsigned char *data ,*oob,path[128];
	size_t retlen;
	int len = 913408, llen=2*1024*1024;
	int fd;
	struct mtd_oob_ops ops;
	data = (unsigned char *)malloc(llen);
	oob = (unsigned char *)malloc(llen);

	printf("lllllllllllllllllllllllll0x%lX\n" ,(unsigned long)oob);

	memset(&ops, 0, sizeof(struct mtd_oob_ops));
	ops.datbuf = data;
	ops.len = len;
	ops.oobbuf=oob;
	ops.ooblen = 0x20000;
	ops.mode=MTD_OOB_RAW;

	nand_read_oob(&mtdinfo, 0, &ops);
	memset(path,0,sizeof(path));
	sprintf(path, "%s", "/home/tao.zhou/xiaohui.zhu/readtest.bin");
	fd=open(path, O_RDWR | O_CREAT);
	if(fd<0){
		printf("open failed %s\n", path);
		return;
	}
	printf("tttt\n");
	write(fd, data, len/2048*2176);
	close(fd);
}






void test_mtd4(void)
{
	unsigned char *data ,*oob,path[128];
	size_t retlen;
	int len = 913408, llen=2*1024*1024;
	int fd;
	struct mtd_oob_ops ops;
	data = (unsigned char *)malloc(llen);
	oob = (unsigned char *)malloc(llen);

	printf("lllllllllllllllllllllllll0x%lX\n" ,(unsigned long)oob);

	memset(&ops, 0, sizeof(struct mtd_oob_ops));
	ops.datbuf = data;
	ops.len = len;
	ops.oobbuf=oob;
	ops.ooblen = 57088;
	ops.mode=MTD_OOB_RAW;

	nand_read_oob(&mtdinfo, 0, &ops);
	printf("Hello\n");
	memset(path,0,sizeof(path));
	sprintf(path, "%s", "/home/tao.zhou/xiaohui.zhu/readtest.bin");
	fd=open(path, O_RDWR | O_CREAT);
	if(fd<0){
		printf("open failed %s\n", path);
		return;
	}
	printf("tttt\n");
	write(fd, oob, 57088);
	close(fd);
}

void test_mtd5(void)
{
	unsigned char *data ,*oob,path[128];
	size_t retlen;
	int len = 0x20800;
	int fd, i;
	struct mtd_oob_ops ops;
	data = (unsigned char *)malloc(len);
	oob = (unsigned char *)malloc(len);


/*

	for(i=0;i<len/512;i++){
		memset(data+512*i,i,512);
		memset(oob+32*i,i+1, 32);
	}


//	nand_write(&mtdinfo, 0x200, len, &retlen, data);


	memset(&ops, 0, sizeof(struct mtd_oob_ops));
	ops.datbuf = NULL;
	ops.len = len;
	ops.oobbuf=NULL;
	ops.ooblen = mtdinfo.oobsize;
	ops.mode=MTD_OOB_RAW;
	printf("Hi.................................\n");
	nand_write_oob(&mtdinfo, 0x200, &ops);

/////////////////////////////////////////////////////////////

	memset(&ops, 0, sizeof(struct mtd_oob_ops));
	ops.datbuf = data;
	ops.len = len;
	ops.oobbuf=NULL;
	ops.ooblen = mtdinfo.oobsize;
	ops.mode=MTD_OOB_RAW;
	printf("Hi.................................\n");
	nand_write_oob(&mtdinfo, 0x200, &ops);

///////////////////////////////////////////////////////////////////

	memset(&ops, 0, sizeof(struct mtd_oob_ops));
	ops.datbuf = NULL;
	ops.len = len;
	ops.oobbuf=oob;
	ops.ooblen = mtdinfo.oobsize;
	ops.mode=MTD_OOB_RAW;
	printf("Hi.................................\n");
	nand_write_oob(&mtdinfo, 0x200, &ops);

	memset(&ops, 0, sizeof(struct mtd_oob_ops));
	ops.datbuf = data;
	ops.len = len;
	ops.oobbuf=oob;
	ops.ooblen = mtdinfo.oobsize;
	ops.mode=MTD_OOB_RAW;
	printf("Hi.................................\n");
	nand_write_oob(&mtdinfo, 0x200, &ops);

//////////////////////////////////////////////////////////////////


	memset(&ops, 0, sizeof(struct mtd_oob_ops));
	ops.datbuf = NULL;
	ops.len = 0x800;
	ops.oobbuf=oob;
	ops.ooblen = mtdinfo.oobsize;
	ops.mode=MTD_OOB_RAW;
	nand_read_oob(&mtdinfo, 0, &ops);
	{
		int a,b;
		for(a=0;a<128;a+=16){
			for(b=0;b<16;b++){
				printf("0x%02X ", oob[a+b]);
			}
			printf("\n");
		}

	}


		for(i=0;i<0x2000/2048;i++){
			for(a=0;a<128;a+=16){
				for(b=0;b<16;b++){
					printf("0x%02X ", data[2048+i*2176+a+b]);
				}
				printf("\n");
			}
			printf("\n");
		}

*/
	memset(&ops, 0, sizeof(struct mtd_oob_ops));
	ops.datbuf = data;
	ops.len = 0x2000;
	ops.oobbuf=oob;
	ops.ooblen = mtdinfo.oobsize * 0x2000/2048*128;
	ops.mode=MTD_OOB_RAW;
	nand_read_oob(&mtdinfo, 0, &ops);
	{
		int a,b;
			for(a=0;a<2176*0x2000/2048;a+=16){
				for(b=0;b<16;b++){
					printf("0x%02X ", data[a+b]);
				}
				printf("\n");
			}
			printf("\n");
	}

}
