/*
 * Unsorted Block Image commands
 *
 *  Copyright (C) 2008 Samsung Electronics
 *  Kyungmin Park <kyungmin.park@samsung.com>
 *
 * Copyright 2008-2009 Stefan Roese <sr@denx.de>, DENX Software Engineering
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <errno.h>

#include "common.h"
#include "command.h"

#include "linux/err.h"
#include "../mtd/ubi/ubi.h"
#include "../mtd/nand/mtd.h"
#include "../mtd/nand/partitions.h"
#include "jiffs2/load_kernel.h"

#define DEV_TYPE_NONE		0
#define DEV_TYPE_NAND		1
#define DEV_TYPE_ONENAND	2
#define DEV_TYPE_NOR		3

/* Private own data */
static struct ubi_device *ubi;
static int ubi_initialized;

struct selected_dev {
	char part_name[80];
	int selected;
	int nr;
	struct mtd_info *mtd_info;
};
static struct selected_dev ubi_dev;

extern struct ubi_device *ubi_devices[UBI_MAX_DEVICES];
extern struct mtd_device *current_mtd_dev;
extern struct part_info* mtd_part_info(struct mtd_device *dev, unsigned int part_num);
extern int search_tftp_buffer(unsigned long *addr);
extern void release_tftp_buffer(unsigned long addr);


void ubi_get_part_name(char *name)
{
	struct part_info *part;
	int index;
	if(name && ubi){
		index =atoi(ubi->mtd->name + strlen("mtd="));
		part = mtd_part_info(current_mtd_dev, index);
		strcpy(name, part->name);
	}
}

static void ubi_dump_vol_info(const struct ubi_volume *vol)
{
	ubi_msg("volume information dump:");
	ubi_msg("vol_id          %d", vol->vol_id);
	ubi_msg("reserved_pebs   %d", vol->reserved_pebs);
	ubi_msg("alignment       %d", vol->alignment);
	ubi_msg("data_pad        %d", vol->data_pad);
	ubi_msg("vol_type        %d", vol->vol_type);
	ubi_msg("name_len        %d", vol->name_len);
	ubi_msg("usable_leb_size %d", vol->usable_leb_size);
	ubi_msg("used_ebs        %d", vol->used_ebs);
	ubi_msg("used_bytes      %lld", vol->used_bytes);
	ubi_msg("last_eb_bytes   %d", vol->last_eb_bytes);
	ubi_msg("corrupted       %d", vol->corrupted);
	ubi_msg("upd_marker      %d", vol->upd_marker);

	if (vol->name_len <= UBI_VOL_NAME_MAX &&
		strnlen(vol->name, vol->name_len + 1) == vol->name_len) {
		ubi_msg("name            %s", vol->name);
	} else {
		ubi_msg("the 1st 5 characters of the name: %c%c%c%c%c",
			vol->name[0], vol->name[1], vol->name[2],
			vol->name[3], vol->name[4]);
	}
}

static void display_volume_info(struct ubi_device *ubi)
{
	int i;

	for (i = 0; i < (ubi->vtbl_slots + UBI_INT_VOL_COUNT); i++) {
		if (!ubi->volumes[i])
			continue;	/* Empty record */
		ubi_dump_vol_info(ubi->volumes[i]);
	}
}

static void display_ubi_info(struct ubi_device *ubi)
{
	ubi_msg("MTD device name:            \"%s\"", ubi->mtd->name);
	ubi_msg("MTD device size:            %llu MiB", ubi->flash_size >> 20);
	ubi_msg("physical eraseblock size:   %d bytes (%d KiB)", ubi->peb_size, ubi->peb_size >> 10);
	ubi_msg("logical eraseblock size:    %d bytes", ubi->leb_size);
	ubi_msg("number of good PEBs:        %d", ubi->good_peb_count);
	ubi_msg("number of bad PEBs:         %d", ubi->bad_peb_count);
	ubi_msg("smallest flash I/O unit:    %d", ubi->min_io_size);
	ubi_msg("VID header offset:          %d (aligned %d)",
	    		ubi->vid_hdr_offset, ubi->vid_hdr_aloffset);
	ubi_msg("data offset:                %d", ubi->leb_start);
	ubi_msg("max. allowed volumes:       %d", ubi->vtbl_slots);
	ubi_msg("number of internal volumes: %d", UBI_INT_VOL_COUNT);
	ubi_msg("number of user volumes:     %d", ubi->vol_count - UBI_INT_VOL_COUNT);
	ubi_msg("available PEBs:             %d", ubi->avail_pebs);
	ubi_msg("total number of reserved PEBs: %d", ubi->rsvd_pebs);
	ubi_msg("number of PEBs reserved for bad PEB handling: %d", ubi->beb_rsvd_pebs);
	ubi_msg("max/mean erase counter: %d/%d", ubi->max_ec, ubi->mean_ec);
}

//mstar

int ubi_get_leb_size(void)
{
	return ubi->leb_size;
}

int ubi_get_avai_peb(void)
{
	return ubi->avail_pebs;
}

static int ubi_info(int layout)
{
	if (layout)
		display_volume_info(ubi);
	else
		display_ubi_info(ubi);

	return 0;
}

static int verify_mkvol_req(const struct ubi_device *ubi,
			    const struct ubi_mkvol_req *req)
{
	int n, err = EINVAL;

	if (req->bytes < 0 || req->alignment < 0 || req->vol_type < 0 || req->name_len < 0)
		goto bad;

	if ((req->vol_id < 0 || req->vol_id >= ubi->vtbl_slots) && req->vol_id != UBI_VOL_NUM_AUTO)
		goto bad;

	if (req->alignment == 0)
		goto bad;

	if (req->bytes == 0) {
		printf("No space left in UBI device!\n");
		err = ENOMEM;
		goto bad;
	}

	if (req->vol_type != UBI_DYNAMIC_VOLUME && req->vol_type != UBI_STATIC_VOLUME)
		goto bad;

	if (req->alignment > ubi->leb_size)
		goto bad;

	n = req->alignment % ubi->min_io_size;
	if (req->alignment != 1 && n)
		goto bad;

	if (req->name_len > UBI_VOL_NAME_MAX) {
		printf("Name too long!\n");
		err = ENAMETOOLONG;
		goto bad;
	}

	return 0;
bad:
	return err;
}

int ubi_create_vol(char *volume, size_t size, int dynamic)
{
	struct ubi_mkvol_req req;
	int err;

	if (dynamic)
		req.vol_type = UBI_DYNAMIC_VOLUME;
	else
		req.vol_type = UBI_STATIC_VOLUME;

	req.vol_id = UBI_VOL_NUM_AUTO;
	req.alignment = 1;
	req.bytes = size;

	//strcpy(req.name, volume);
	if(strlen(volume)>=sizeof(req.name)){
		strncpy(req.name, volume, sizeof(req.name)-1);
		req.name[sizeof(req.name)-1]='\0';
	}
	else{
		strcpy(req.name, volume);
	}
	req.name_len = strlen(volume);
	req.name[req.name_len] = '\0';
	req.padding1 = 0;
	/* It's duplicated at drivers/mtd/ubi/cdev.c */
	err = verify_mkvol_req(ubi, &req);
	if (err) {
		printf("verify_mkvol_req failed %d\n", err);
		return err;
	}
	printf("Creating %s volume %s of size %lu\n",
		dynamic ? "dynamic" : "static", volume, (unsigned long)size);
	/* Call real ubi create volume */
	return ubi_create_volume(ubi, &req);
}

struct ubi_volume *ubi_find_volume(char *volume)
{
	struct ubi_volume *vol = NULL;
	int i;

	if(ubi ==NULL)
		return NULL;
	
	for (i = 0; i < ubi->vtbl_slots; i++) {
		vol = ubi->volumes[i];
		if (vol && !strcmp(vol->name, volume))
			return vol;
        }

	printf("Volume %s not found!\n", volume);
	return NULL;
}

// MStar start
int ubi_volume_write_offset(char *volume, void *buf, size_t size, off_t offset)
{
	int err = -1;
	struct ubi_volume *vol;

	vol = ubi_find_volume(volume);
	if (vol == NULL)
		return ENODEV;
	//check "offset + size" is less than total size of volume
	if((offset + size) > (vol->reserved_pebs * ubi->leb_size)){
		printf("[%s]: write size exceeds volume size\n", __func__);
		printf("[%s]: Succeed volume size in Volume \"%s\"\n", __func__, volume);
		printf("[%s]: size: %d, offset %d, reserved pebs: %d, leb size: %d\n", __func__, size, offset, vol->reserved_pebs, ubi->leb_size);
		return 1;
	}
	//==================Partial Update=======================
	err = ubi_update_partial_data(ubi, vol, buf, offset, size);

	if(err)
		printf("[%s]\tCan't Partial update data\n", __func__);

	return err;	
}

int ubi_get_volume_size(char *volume, size_t* size)
{
	int i = 0, err = -1;
	int found = 0;
	struct ubi_volume *vol;

	for (i = 0; i < ubi->vtbl_slots; i++) {
		vol = ubi->volumes[i];
		if (vol && !strcmp(vol->name, volume)) {
			printf("Volume \"%s\" found at volume id %d\n", volume, i);
			found = 1;
			break;
		}
	}
	if (!found) {
		printf("%s volume not found\n", volume);
		return 1;
	}

	vol = ubi_find_volume(volume);
	if (vol == NULL)
		return ENODEV;
	
	*size = vol->used_bytes;

	return err;	
}

static int ubi_volume_write_cont(char *volume, void *buf, size_t size)
{
	int i = 0, err = -1;
	int found = 0;
	struct ubi_volume *vol;

	for (i = 0; i < ubi->vtbl_slots; i++) {
		vol = ubi->volumes[i];
		if (vol && !strcmp(vol->name, volume)) {
			printf("Volume \"%s\" found at volume id %d\n", volume, i);
			found = 1;
			break;
		}
	}
	if (!found) {
		printf("%s volume not found\n", volume);
		return 1;
	}

	vol = ubi_find_volume(volume);
	if (vol == NULL)
		return ENODEV;

	err = ubi_more_update_data(ubi, vol, buf, size);
	if (err < 0) {
		printf("Couldnt or partially wrote data \n");
		return err;
	}

	if (err) {
		size = err;

		err = ubi_check_volume(ubi, vol->vol_id);
		if ( err < 0 )
			return err;

		if (err) {
			ubi_warn("volume %d on UBI device %d is corrupted",
					vol->vol_id, ubi->ubi_num);
			vol->corrupted = 1;
		}

		vol->checked = 1;
	}
	
	return 0;
}

static int ubi_volume_write(char *volume, void *buf, size_t size, size_t total_size)
{
	int err = 1;
	int rsvd_bytes = 0;
	struct ubi_volume *vol;

	vol = ubi_find_volume(volume);
	if (vol == NULL)
		return ENODEV;

	rsvd_bytes = vol->reserved_pebs * (ubi->leb_size - vol->data_pad);
	if (size < 0 || size > rsvd_bytes) {
		printf("size > volume size! Aborting!\n");
		return EINVAL;
	}
printf("double free 1----------------------------\n");
	err = ubi_start_update(ubi, vol, (long long)total_size);
	if (err < 0) {
		printf("Cannot start volume update\n");
		return -err;
	}
printf("double free 1----------------------------2\n");

	err = ubi_more_update_data(ubi, vol, buf, size);
	if (err < 0) {
		printf("Couldnt or partially wrote data\n");
		return -err;
	}

	if (err) {
		size = err;

		err = ubi_check_volume(ubi, vol->vol_id);
		if (err < 0)
			return -err;

		if (err) {
			ubi_warn("volume %d on UBI device %d is corrupted", vol->vol_id, ubi->ubi_num);
			vol->corrupted = 1;
		}

		vol->checked = 1;
	}

	printf("%d bytes written to volume %s\n", size, volume);
	return 0;
}

// MStar start
int ubi_volume_partial_read(char *volume, char *buf, size_t size, size_t partition_offset)
{
	int err, lnum, off, len, tbuf_size, i = 0;
	size_t count_save = size;
	void *tbuf;
	unsigned long long tmp;
	struct ubi_volume *vol = NULL;
	loff_t offp = 0;
	size_t offset;
	size_t bnum;

	for (i = 0; i < ubi->vtbl_slots; i++) {
		vol = ubi->volumes[i];
		if (vol && !strcmp(vol->name, volume)) {
			//printf("Volume %s found at volume id %d\n",
			//	volume, vol->vol_id);
			break;
		}
	}
	if ((i == ubi->vtbl_slots)||(vol==NULL)) {
		printf("%s volume not found\n", volume);
		return -ENODEV;
	}

	//printf("read %d bytes from volume %d to %x(buf address) with offset 0x%x\n",
	//       (int) size, vol->vol_id, (unsigned)buf, (unsigned)partition_offset);

	if (vol->updating) {
		printf("[%s]: updating\n", __func__);
		return -EBUSY;
	}
	if (vol->upd_marker) {
		printf("[%s]: damaged volume, update marker is set\n", __func__);
		return -EBADF;
	}
	if (offp == vol->used_bytes)
		return 0;

	if (size == 0) {
		printf("[%s]: Read [%lu] bytes\n", __func__, (unsigned long) vol->used_bytes);
		size = vol->used_bytes;
	}

	if (vol->corrupted)
		printf("[%s]: read from corrupted volume %d", __func__, vol->vol_id);
	
	if (offp + size > vol->used_bytes)
		count_save = size = vol->used_bytes - offp;

	tbuf_size = vol->usable_leb_size;
	if (size < tbuf_size)
		tbuf_size = ALIGN(size, ubi->min_io_size);
	tbuf = malloc(tbuf_size);
	if (!tbuf) {
		printf("[%s]: NO MEM\n", __func__);
		return -ENOMEM;
	}
	len = size > tbuf_size ? tbuf_size : size;

	tmp = offp;
	//red flag CL613675
	offset=partition_offset%vol->usable_leb_size;
	bnum=partition_offset/vol->usable_leb_size;
	off = offset;//do_div(tmp, vol->usable_leb_size);
	lnum = tmp;
#if 1
	do {
		if (off + len >= vol->usable_leb_size)
			len = vol->usable_leb_size - off;

		err = ubi_eba_read_leb(ubi, vol, (lnum+bnum), tbuf, off, len, 0);

		if (err) {
			printf("read err %x\n", err);
			break;
		}
		off += len;
		if (off == vol->usable_leb_size) {
			lnum += 1;
			off -= vol->usable_leb_size;
		}

		size -= len;
		offp += len;

		memcpy(buf, tbuf, len);

		buf += len;
		len = size > tbuf_size ? tbuf_size : size;
	} while (size);
	if (err) {
		printf("read err %x\n", err);
	}
#else
    do {
		if (off + len >= vol->usable_leb_size)
			len = vol->usable_leb_size - off;

		err = ubi_eba_read_leb(ubi, vol, lnum, tbuf, off, len, 0);
        printf("\n\n1111====off=0x%x,len=0x%x,lnum=0x%x, tbuf=0x%x=====\n\n",off,len,lnum,tbuf);
		if (err) {
			printf("read err %x\n", err);
			break;
		}
		off += len;
		if (off == vol->usable_leb_size) {
			lnum += 1;
			off -= vol->usable_leb_size;
        printf("\n\n2222====off=0x%x,len=0x%x,vol->usable_leb_size=0x%x=====\n\n",off,len,vol->usable_leb_size);
		}

		size -= len;
		offp += len;

		memcpy(buf, tbuf, len);

		buf += len;
		len = size > tbuf_size ? tbuf_size : size;
	} while (size);
#endif

	free(tbuf);
    //return err ? err : count_save - size;
    return err;
}
// MStar end

static int ubi_volume_read(char *volume, char *buf, size_t size)
{
	int err, lnum, off, len, tbuf_size;
	size_t count_save = size;
	void *tbuf;
	unsigned long long tmp;
	struct ubi_volume *vol;
	loff_t offp = 0;

	vol = ubi_find_volume(volume);
	if (vol == NULL)
		return ENODEV;

	//printf("Read %d bytes from volume %s to %p\n", size, volume, buf);

	if (vol->updating) {
		printf("[%s]: updating\n", __func__);
		return EBUSY;
	}
	if (vol->upd_marker) {
		printf("[%s]: damaged volume, update marker is set\n", __func__);
		return EBADF;
	}
	if (offp == vol->used_bytes)
		return 0;

	if (size == 0) {
		printf("[%s]: No size specified -> Using max size (%lld)\n", __func__, vol->used_bytes);
		size = vol->used_bytes;
	}

	if (vol->corrupted)
		printf("[%s]: read from corrupted volume %d", __func__, vol->vol_id);
	if (offp + size > vol->used_bytes)
		count_save = size = vol->used_bytes - offp;

	tbuf_size = vol->usable_leb_size;
	if (size < tbuf_size)
		tbuf_size = ALIGN(size, ubi->min_io_size);
	tbuf = malloc(tbuf_size);
	if (!tbuf) {
		printf("[%s]: NO MEM\n", __func__);
		return ENOMEM;
	}
	len = size > tbuf_size ? tbuf_size : size;

	tmp = offp;
	off = do_div(tmp, vol->usable_leb_size);
	lnum = tmp;
	do {
		if (off + len >= vol->usable_leb_size)
			len = vol->usable_leb_size - off;

		err = ubi_eba_read_leb(ubi, vol, lnum, tbuf, off, len, 0);
		if (err) {
			printf("read err %x\n", err);
			err = -err;
			break;
		}
		off += len;
		if (off == vol->usable_leb_size) {
			lnum += 1;
			off -= vol->usable_leb_size;
		}

		size -= len;
		offp += len;

		memcpy(buf, tbuf, len);

		buf += len;
		len = size > tbuf_size ? tbuf_size : size;
	} while (size);

	free(tbuf);
	return err;
}

static int ubi_dev_scan(struct mtd_info *info, char *ubidev, const char *vid_header_offset)
{
	struct mtd_device *dev;
	struct part_info *part;
	struct mtd_partition mtd_part;
	struct mtd_info *slave;
	unsigned char mtd_dev[80];
	u8 pnum;
	int vid_hdr_offset = 0;
	char *endp;
	int i, err;

	if (find_dev_and_part(ubidev, &dev, &pnum, &part) != 0)
		return 1;

	memset(mtd_dev, 0, sizeof(mtd_dev));
	sprintf(mtd_dev, "mtd=%d", pnum);
	slave = (struct mtd_info *)open_mtd_device(mtd_dev);
	if(!IS_ERR(slave)){
		for(i = 0; i < UBI_MAX_DEVICES; i++){
			if(ubi_devices[i]->mtd->index == slave->index){
				printf("[%s]: part %s had been initialized\n", __func__, part->name);
				ubi = ubi_devices[i];
				return 0;
			}
		}
	}

	if(vid_header_offset)
		vid_hdr_offset = strtoul(vid_header_offset, &endp, 0);

	memset(&mtd_part, 0, sizeof(mtd_part));
	mtd_part.name = mtd_dev;
	mtd_part.size = part->size;
	mtd_part.offset = part->offset;
	add_mtd_partitions(info, &mtd_part, 1);

	slave = (struct mtd_info *)open_mtd_device(mtd_dev);
	if(!slave){
		printf("[%s]: open mtd device %d failed\n", __func__, mtd_dev);
	}
	err = ubi_attach_mtd_dev(slave, UBI_DEV_NUM_AUTO, vid_hdr_offset);
	if (err < 0) {
		printf("[%s]: failed\n", __func__);
		return err;
	}
	ubi_initialized = 1;

	return 0;
}

static int do_ubi(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
	size_t size = 0;
	ulong addr = 0;
	int i, err, ret = 0;

	if (argc < 2)
		return cmd_usage(cmdtp);

	if (mtdparts_init() != 0) {
		printf("Error initializing mtdparts!\n");
		return 1;
	}

	if (strcmp(argv[1], "part") == 0) {
		char mtd_dev[16];
		char part_name[16];
		struct mtd_device *dev;
		struct part_info *part;
		const char *vid_header_offset = NULL;
		u8 pnum;
		struct ubi_volume *vol;

		/* Print current partition */
		if (argc == 2) {
			if (!ubi_dev.selected) {
				printf("Error, no UBI device/partition selected!\n");
				return 1;
			}

			printf("Device %d: %s, partition %s\n", ubi_dev.nr, ubi_dev.mtd_info->name, ubi_dev.part_name);
			return 0;
		}

		if (argc < 3)
			return cmd_usage(cmdtp);

		/* todo: get dev number for NAND... */
		ubi_dev.nr = 0;

		/*
		 * Search the mtd device number where this partition
		 * is located
		 */
		if (find_dev_and_part(argv[2], &dev, &pnum, &part)) {
			printf("[%s]: Partition %s not found!\n", __func__, argv[2]);
			return 1;
		}
		sprintf(mtd_dev, "%s%d", MTD_DEV_TYPE(dev->id->type), dev->id->num);
		ubi_dev.mtd_info = get_mtd_device_nm(mtd_dev);
		if (IS_ERR(ubi_dev.mtd_info)) {
			printf("Partition %s not found on device %s!\n", argv[2], mtd_dev);
			return 1;
		}

		ubi_dev.selected = 1;

		if (argc > 3)
			vid_header_offset = argv[3];
		strcpy(ubi_dev.part_name, argv[2]);
		err = ubi_dev_scan(ubi_dev.mtd_info, ubi_dev.part_name,vid_header_offset);
		if (err) {
			printf("UBI init error %d\n", err);
			ubi_dev.selected = 0;
			return err;
		}
		memset(part_name, 0, sizeof(part_name));
		sprintf(part_name, "mtd=%d", pnum);
		for(i = 0; i < UBI_MAX_DEVICES; i++){
			if(ubi_devices[i])
				printf("%s\n", ubi_devices[i]->mtd->name);
			if(ubi_devices[i] && strcmp(ubi_devices[i]->mtd->name, part_name) == 0){
				ubi = ubi_devices[i];
				break;
			}
		}
		return 0;
	}

	if ((strcmp(argv[1], "part") != 0) && (!ubi_dev.selected)) {
		printf("Error, no UBI device/partition selected!\n");
		return 1;
	}

	if (strcmp(argv[1], "info") == 0) {
		int layout = 0;
		if (argc > 2 && !strncmp(argv[2], "l", 1))
			layout = 1;
		return ubi_info(layout);
	}

	if (strncmp(argv[1], "create", 6) == 0) {
		int dynamic = 1;	/* default: dynamic volume */

		/* Use maximum available size */
		size = 0;

		/* E.g., create volume size type */
		if (argc == 5) {
			if (strncmp(argv[4], "s", 1) == 0)
				dynamic = 0;
			else if (strncmp(argv[4], "d", 1) != 0) {
				printf("Incorrect type\n");
				return 1;
			}
			argc--;
		}
		/* E.g., create volume size */
		if (argc == 4) {
			size = strtoul(argv[3], NULL, 16);
			argc--;
		}
		/* Use maximum available size */
		if (!size) {
			size = ubi->avail_pebs * ubi->leb_size;
			printf("No size specified -> Using max size (%u)\n", size);
		}
		/* E.g., create volume */
		if (argc == 3)
			return ubi_create_vol(argv[2], size, dynamic);
	}

//MStar start
        if (strncmp(argv[1], "write_cont", 10) == 0) {
		if (argc < 5) {
			printf("Please see usage\n");
			return -EPERM;
		}

		addr = strtoul(argv[2], NULL, 16);
		size = strtoul(argv[4], NULL, 16);

		ret = search_tftp_buffer(&addr);
		if(ret)
			printf("[%s]: find tftp buffer address %lx for address %s\n", __func__, addr, argv[2]);
		else
			printf("[%s]: find no tftp buffer, use original address %lx\n", __func__, addr);
		
		err=ubi_volume_write_cont(argv[3], (void *)addr, size);

		addr = strtoul(argv[2], NULL, 16);
		release_tftp_buffer(addr);
		
		return err;
        }
//MStar end

	if (strncmp(argv[1], "write", 5) == 0) {
		if (argc < 5) {
			printf("Please see usage\n");
			return 1;
		}
		addr = strtoul(argv[2], NULL, 16);
		size = strtoul(argv[4], NULL, 16);
		
		ret = search_tftp_buffer(&addr);
		if(ret)
			printf("[%s]: find tftp buffer address %lx for address %s\n", __func__, addr, argv[2]);
		else
			printf("[%s]: find no tftp buffer, use original address %lx\n", __func__, addr);
		
		{
			size_t total_size =0;
			if(argc == 6)
			{
				total_size = strtoul(argv[5], NULL, 16);
				printf("total size = %d\n", total_size);
			}
			else
			{
				total_size = size;
			}
			err=ubi_volume_write(argv[3], (void *)addr, size, total_size);

			addr = strtoul(argv[2], NULL, 16);
			printf("cmd ubi test ---------------------------0x%lx\n", (unsigned long)addr);
			release_tftp_buffer(addr);

			return err;
		}
	}

	if(argc == 6){
		if (strncmp(argv[1], "partial_read", 12) == 0){
			size_t partition_offset;
			addr = strtoul(argv[2], NULL, 16);
			size= strtoul(argv[4], NULL, 16);
			partition_offset = strtoul(argv[5], NULL, 16);

			return ubi_volume_partial_read(argv[3], (char *)addr, size ,partition_offset);
		}
	}

	if (strncmp(argv[1], "read", 4) == 0) {
		size = 0;

		/* E.g., read volume size */
		if (argc == 5) {
			size = strtoul(argv[4], NULL, 16);
			argc--;
		}

		/* E.g., read volume */
		if (argc == 4) {
			addr = strtoul(argv[2], NULL, 16);
			argc--;
		}

		if (argc == 3)
			return ubi_volume_read(argv[3], (char *)addr, size);
	}

	if(strncmp(argv[1], "partial_write", 13) == 0 ){
		if(argc == 6){
			ulong off;
			size = strtoul(argv[4], NULL, 16);
			off = strtoul(argv[5], NULL, 16);
			addr = strtoul(argv[2], NULL, 16);
			
			ret = search_tftp_buffer(&addr);
			if(ret)
				printf("[%s]: find tftp buffer address %lx for address %s\n", __func__, addr, argv[2]);
			else
				printf("[%s]: find no tftp buffer, use original address %lx\n", __func__, addr);

			
			return ubi_volume_write_offset(argv[3], (void *)addr, size, off);

			addr = strtoul(argv[2], NULL, 16);
			release_tftp_buffer(addr);
		}
	}

	if (strncmp(argv[1], "read.e", 6) == 0) {
		ulong lebnum, pebnum, memaddr;
        int volnum, i, dumplebnum = 0, dpret = 0;
        struct ubi_volume *vol = NULL, *voltmp = NULL, *volterm = NULL;

		if(argc < 5)
			goto usage;

		memaddr = (int)strtoul(argv[2], NULL, 16);
		volnum = (int)strtoul(argv[3], NULL, 16);
		lebnum = (int)strtoul(argv[4], NULL, 16);

        if(argc > 5)
		    dumplebnum = (int)strtoul(argv[5], NULL, 16);

        if((volnum > 0) && (volnum < ubi->vtbl_slots))
              vol = ubi->volumes[volnum];

        if(vol){
            printf("Volume %s found at volume id %d\n", vol->name, volnum);
            goto readdump;
        }

        for (i = 0; i < ubi->vtbl_slots; i++) {
			voltmp = ubi->volumes[i];
			if (voltmp && (!strcmp(voltmp->name, argv[3]))) {
				printf("Volume %s found at volume id %d\n", argv[2], voltmp->vol_id);
                goto readdump;
            }
        }
        printf("volume not found!\n");
        goto usage;

readdump:

		for(i = 0;;i ++){
			if(vol)
				volterm = vol;
			else if(voltmp && (!strcmp(voltmp->name, argv[2])))
				volterm = voltmp;
			else{
				printf("volume not found!\n");
				goto usage;
			}
			pebnum = volterm->eba_tbl[lebnum + i];
			if(!pebnum)
			{
				printf("Can not found the corresponding Peb!\n");
				continue;
			}
			if(!dumplebnum)
				break;

			dpret = ubi_eba_read_leb(ubi, volterm, lebnum+i,
				(u_char*) memaddr, 0, volterm->usable_leb_size, 0);
			printf("lebnum:%ld, pebnum:%ld\n ", lebnum+i, pebnum);

			if(dumplebnum > 0){
				--dumplebnum;
				memaddr += volterm->usable_leb_size;
			}
			else
				break;
		}

		return dpret == 0 ? 1 : 0;
	}
usage:
	printf("Please see usage\n");
	return 1;
}

U_BOOT_CMD(
	ubi, 6, 1, do_ubi,
	"ubi commands",
	"part [part] [offset]\n"
		" - Show or set current partition (with optional VID"
		" header offset)\n"
	"ubi info [l[ayout]]"
		" - Display volume and ubi layout information\n"
	"ubi create[vol] volume [size] [type]"
		" - create volume name with size\n"
	"ubi write[vol] address volume size"
		" - Write volume from address with size\n"
	"ubi read[vol] address volume [size]"
		" - Read volume to address with size\n"
	"ubi partial_read/paritial_write address volume size offset\n"
		" - Write/Read volume from/to address with offset and size\n"
	"ubi remove[vol] volume"
		" - Remove volume\n"
	"[Legends]\n"
	" volume: character name\n"
	" size: specified in bytes\n"
	" type: s[tatic] or d[ynamic] (default=dynamic)"
);
