/*
 * (C) Copyright 2000
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

/*
 * Boot support
 */
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <fcntl.h>
#include <sys/stat.h>

#include "tool-util.h"
#include "common.h"
#include "command.h"
#include "nand.h"

#define CONFIG_SYS_MAX_NAND_DEVICE 1
#define TFTP_BUF_THRESHOLD (1 * 1024 * 1024)

extern int bintype;
extern char *root_directory;
extern char *image_directory;
extern nand_info_t nand_info[CONFIG_SYS_MAX_NAND_DEVICE];

struct tftp_buffer *tftp_buffer[TFTP_BUFFER_NUM]={0};

int search_tftp_buffer(unsigned long *addr)
{
	int i;
	for(i = 0; i < TFTP_BUFFER_NUM; i++){
		if(tftp_buffer[i] && (unsigned long)(tftp_buffer[i]->addr) == *addr){
			printf("[%s]: find tftp buffer index:%d buffer len:%ld\n", __func__, i, tftp_buffer[i]->len);
			*addr = (unsigned long)(tftp_buffer[i]->data);
			return 1;
		}
	}
	return 0;
}

void release_tftp_buffer(unsigned long addr)
{
	int i;
	for(i = 0; i < TFTP_BUFFER_NUM; i++){
		if(tftp_buffer[i] && (unsigned long)(tftp_buffer[i]->addr) == addr){
			printf("[%s]: free tftp buffer index: %d, addr=%lx\n", __func__, i, addr);
			free(tftp_buffer[i]);
			tftp_buffer[i] = NULL;
			return;
		}
	}
}

void do_tftprelease()
{
    int i;
	for(i = 0; i < TFTP_BUFFER_NUM; i++){
		if(tftp_buffer[i]){
			free(tftp_buffer[i]);
			tftp_buffer[i] = NULL;
			return;
		}
	}
}

int do_tftpb (cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int fd, i, ret;
	unsigned long len;
	char file[512], envbuf[16];
	char *endp;
	struct stat st;
	unsigned long addr_to_write;
    unsigned long data_buff_addr;

	if(argc < 3){
		printf("[%s]: invalid argument\n", __func__);
		return -1;
	}

	addr_to_write = strtoul(argv[1], &endp, 16);

    data_buff_addr = addr_to_write;
    ret = search_tftp_buffer(&data_buff_addr);
    if(ret == 1)
    {
        release_tftp_buffer(addr_to_write);
    }

	for(i = 0; i < TFTP_BUFFER_NUM; i++){
		if(!tftp_buffer[i])
			break;
		/*
            if a buffer has been allocated for the same DRAM addr, free it
            to prevent two or more buffers for same DRAM
            (since tftp writes to DRAM, no need to reserve old DRAM content)
            */
	        if(tftp_buffer[i]->addr == addr_to_write)
	        {
	            free(tftp_buffer[i]);
	            tftp_buffer[i] = NULL;
	            break;
	        }
	}
	if(i == TFTP_BUFFER_NUM){
		printf("[%s]: No available tftp buffer\n", __func__);
		return -1;
	}

    memset(&st, 0, sizeof(st));
	memset(file, 0, sizeof(file));
	sprintf(file, "%s%s", root_directory, argv[2]);
	stat(file, &st);

	if(st.st_size <= 0){
        memset(file, 0, sizeof(file));
	    sprintf(file, "%s%s%s", root_directory, image_directory, argv[2]);
	    stat(file, &st);
        if(st.st_size <= 0){
		    printf("[%s]: Wrong file size, please check if file %s exits\n", __func__, file);
		    return -1;
        }
	}

    printf("[%s]: buffer index:%d, filesize %lld\n", __func__, i, st.st_size);

	//Malloc less Memory
	len = st.st_size;
	if(bintype == NANDBIN)
		len = ALIGN(st.st_size,  nand_info[0].erasesize);

	if(len < TFTP_BUF_THRESHOLD/2)
		len = TFTP_BUF_THRESHOLD;
	else
		len = len * 2;

	tftp_buffer[i] = (struct tftp_buffer *)malloc(sizeof(struct tftp_buffer) + len);
	if(!tftp_buffer[i]){
		printf("[%s]: malloc tftp buffer failed\n", __func__);
		return -1;
	}

	printf("[%s]: malloc tftp buffer size: %ld\n", __func__, len);

	memset(tftp_buffer[i], 0x0, sizeof(struct tftp_buffer));
	memset(tftp_buffer[i]->data, 0xFF, len);

	tftp_buffer[i]->addr = addr_to_write;
	tftp_buffer[i]->len = (unsigned long)st.st_size;

	fd = open(file, O_RDWR);
	if(fd < 0){
		printf("[%s]: open file %s failed\n", __func__, file);
		goto out;
	}
	ret = read(fd, tftp_buffer[i]->data, st.st_size);
	if(ret < 0){
		printf("[%s]: read file %s failed\n", __func__, file);
		close(fd);
		goto out;
	}
	memset(envbuf, 0, sizeof(envbuf));
	sprintf(envbuf, "%llX", st.st_size);
	setenv_uboot("filesize", envbuf);
	memset(envbuf, 0, sizeof(envbuf));
	sprintf(envbuf, "%lX", tftp_buffer[i]->addr);
	setenv_uboot("fileaddr", envbuf);
	close(fd);
	return 0;

out:
	free(tftp_buffer[i]);
	tftp_buffer[i] = NULL;
	return -1;
}

U_BOOT_CMD(
	tftpboot,	3,	1,	do_tftpb,
	"boot image via network using TFTP protocol",
	"[loadAddress] [[hostIPaddr:]bootfilename]"
);

U_BOOT_CMD(
	releasebuf,	3,	1,	do_tftprelease,
	"boot image via network using TFTP protocol",
	"[loadAddress] [[hostIPaddr:]bootfilename]"
);
/////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////
void test_tftp(void)
{
	int i;
//	memset(tftp_buffer, 0, sizeof(tftp_buffer));
	run_command("tftp 0x80400000 images/ubifs/mboot_nand.bin", 0);
	for(i = 0; i < TFTP_BUFFER_NUM; i++){
		printf("--**1------%d----------------------\n", i);
		if(tftp_buffer[i]){
			printf("--**2----------------------------\n");
			printf("len = %lx, addr = %lx\n", tftp_buffer[i]->len, tftp_buffer[i]->addr);
		}
	}
	printf("--**5----------------------------\n");
}
