/*
 * Driver for NAND support, Rick Bronson
 * borrowed heavily from:
 * (c) 1999 Machine Vision Holdings, Inc.
 * (c) 1999, 2000 David Woodhouse <dwmw2@infradead.org>
 *
 * Ported 'dynenv' to 'nand env.oob' command
 * (C) 2010 Nanometrics, Inc.
 * 'dynenv' -- Dynamic environment offset in NAND OOB
 * (C) Copyright 2006-2007 OpenMoko, Inc.
 * Added 16-bit nand support
 * (C) 2004 Texas Instruments
 *
 * Copyright 2010 Freescale Semiconductor
 * The portions of this file whose copyright is held by Freescale and which
 * are not considered a derived work of GPL v2-only code may be distributed
 * and/or modified under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 */

#include <stdint.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <asm/byteorder.h>
#include <sys/stat.h>

#include "types.h"
#include "tool-util.h"
#include "common.h"
#include "command.h"
#include "jiffs2/load_kernel.h"
#include "nand.h"
#include "ubi/ubi-media.h"
#include "../mstar/spinand/inc/common/spinand.h"
#include "../mstar/unfd/inc/common/drvNAND.h"


#define CONFIG_SYS_MAX_NAND_DEVICE 1
#define CONFIG_SYS_MAXARGS		32

#define SBOOT_SIZE 0x10000
#define NAND_CLK_LEN   2

#define CONFIG_BFN_VERSION 5

extern struct mtd_device *current_mtd_dev;

extern char *defpath;
extern struct platform_info info;
extern unsigned int cfg_env_offset;

extern int drvNAND_ReadCISBlk(U32 u32_off, U8* pu8_DataBuf);
extern int nand_write_bootloader(U32 u32_Row,U8 * pu8_addr, U32 u32_size, U8 u8_BootStageId);
extern U32 drvNAND_WriteCIS_for_ROM(NAND_FLASH_INFO_t * pNandInfo);
extern int search_cis_in_DRAM(U8* pu8_CISAddr, U8* pu8_PPMAddr, U8* pu8_PNIAddr, NAND_FLASH_INFO_t *pNandInfoOut);
extern int MDrv_SPINAND_ReadCISBlk(U32 u32_off, U8* pu8_DataBuf);
extern int MDrv_SPINAND_write_bootloader(U32 u32_Row, U8 * pu8_addr, U32 u32_size, U8 u8_BootStageId);
extern U32 MDrv_SPINAND_WriteCIS_for_ROM(SPINAND_FLASH_INFO_TAG_t *pSpiNandInfoTag);
extern int MDrv_SPINAND_SearchCIS_in_DRAM(U8 *pu8_CISAddr, U8 *pu8_PPMAddr, U8 *pu8_PNIAddr, SPINAND_FLASH_INFO_TAG_t *pSpiNandInfoTagOut);
extern U32 MDrv_SPINAND_IsGoodBlk(U16 u16_PBA);
extern U32 drvNAND_NCISLDRAPP_WriteCISMIUParam(int MIUBLOCK, U8 *MIUBUFFER);

extern int search_tftp_buffer(unsigned long * addr);
extern void release_tftp_buffer(unsigned long addr);

EN_NAND_FS_FORMAT nand_fs_format = NAND_FS_FORMAT_NORMAL;

#define TOTAL_CIS_BLOCK 10

//attention, fix
#define VALID_CIS_BLOCK 2
//#define VALID_CIS_BLOCK 3

#define IF_UBI_PARTITION(name) if ((0 == strncmp(name, "UBI", 3)) ||(0 == strncmp(name, "ubi", 3)) )

int do_nsbootblk(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[]);
int write_partition_def_leap(ulong start, ulong end, ulong process, ulong pos);
int write_partition_def_snchk(ulong start, ulong end, ulong process, ulong pos, const char * partname);
int term_def_snchk();
int term_def_leap();

/* ------------------------------------------------------------------------- */

static int set_dev(int dev)
{
	if (dev < 0 || dev >= CONFIG_SYS_MAX_NAND_DEVICE || !nand_info[dev].name) {
		puts("No such device\n");
		return -1;
	}

	if (nand_curr_device == dev)
		return 0;

	printf("Device %d: %s", dev, nand_info[dev].name);
	puts("... is now current device\n");
	nand_curr_device = dev;

	return 0;
}

static inline int str2off(const char *p, loff_t *num)
{
	char *endptr=NULL;
	*num = strtoull(p, &endptr, 16);
	return *p != '\0' && *endptr == '\0';
}

static inline int str2long(const char *p, ulong *num)
{
	char *endptr;

	*num = strtoul(p, &endptr, 16);
	return *p != '\0' && *endptr == '\0';
}

static int get_part(const char *partname, int *idx, loff_t *off, loff_t *size)
{
	struct mtd_device *dev;
	struct part_info *part;
	u8 pnum;
	int ret;

	ret = mtdparts_init();
	if (ret)
		return ret;

	ret = find_dev_and_part(partname, &dev, &pnum, &part);
	if (ret)
		return ret;

	if (dev->id->type != MTD_DEV_TYPE_NAND) {
		puts("not a NAND device\n");
		return -1;
	}

	*off = part->offset;
	*size = part->size;
	*idx = dev->id->num;
    printf("%s:%d %s,idx=%d,off=0x%llx,size=0x%llx\n", __func__, __LINE__, partname, *idx, *off,*size);

	ret = set_dev(*idx);
	if (ret)
		return ret;

	return 0;
}

static int arg_off(const char *arg, int *idx, loff_t *off, loff_t *maxsize)
{
	if (!str2off(arg, off))
		return get_part(arg, idx, off, maxsize);

	if (*off >= nand_info[*idx].size) {
		puts("Offset exceeds device limit\n");
		return -1;
	}

	*maxsize = nand_info[*idx].size - *off;
	return 0;
}

static int arg_off_size(int argc, char *const argv[], int *idx,
			loff_t *off, loff_t *size)
{
	int ret;
	loff_t maxsize;
    printf("%s:%d %d,%d,%s\n", __func__, __LINE__, argc, *idx, argv[0]);
	if (argc == 0) {
		*off = 0;
		*size = nand_info[*idx].size;
		goto print;
	}

	ret = arg_off(argv[0], idx, off, &maxsize);
	if (ret)
		return ret;

	if (argc == 1) {
		*size = maxsize;
		goto print;
	}

	if (!str2off(argv[1], size)) {
		printf("'%s' is not a number\n", argv[1]);
		return -1;
	}

	if (*size > maxsize) {
		puts("Size exceeds partition or device limit\n");
		return -1;
	}

print:
	return 0;
}

//check_reserved_block: only check part specified by name
//reserved blocks: partition size * 3% + 2 blocks
static int check_reserved_blocks(const char *arg, loff_t writesize, int is_mlc)
{
	int idx;
	loff_t off, size;
	nand_info_t *nand = &nand_info[nand_curr_device];

	if(is_mlc)
		writesize = writesize * 2;

	if (!str2off(arg, &off)){
		if(get_part(arg, &idx, &off, &size) < 0){
			printf("check reserved block: get part failed\n");
			return 0;
		}
	}
	else
		return 0;

	if(size * 3 / 100 + 2 * nand->erasesize + writesize <= size)
		return 0;

	printf("\n--------------------- Warning!!! ---------------------\n");
	printf(" partition blocks: 0x%llx\n write blocks: 0x%llx\n reserved blocks: 0x%llx\n expected reserved blocks: 0x%llx\n",
		(u64)((size + nand->erasesize - 1) / nand->erasesize),
		(u64)((writesize + nand->erasesize - 1) / nand->erasesize),
		(u64)(size - writesize) / nand->erasesize,
		(u64)(size * 3 / 100 + 2 * nand->erasesize + nand->erasesize - 1) / nand->erasesize);
	printf(" \nPlease extend partition to 0x%llx Bytes(0x%llx blocks), and keep write size not changed!!\n\n",
		(u64)((size + size * 3 / 100 + nand->erasesize - 1) / nand->erasesize + 2) * nand->erasesize,
		(u64)(size + size * 3 / 100 + nand->erasesize - 1) / nand->erasesize + 2);

	sleep(2);

	return 0;
}

int do_nand(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
	int i=0, ret = 0, ftl_ops=0;
	ulong addr=0;
	loff_t off=0, size=0;
	char *cmd, *s=NULL;
	nand_info_t *nand=NULL;
	int dev = nand_curr_device;
	//Mstar Start
	int eraseloop=0;
	//Mstar End
	/* at least two arguments please */
	if (argc < 2)
		goto usage;

	cmd = argv[1];

	/* The following commands operate on the current device, unless
	 * overridden by a partition specifier.  Note that if somehow the
	 * current device is invalid, it will have to be changed to a valid
	 * one before these commands can run, even if a partition specifier
	 * for another device is to be used.
	 */
	nand = &nand_info[dev];

	//check nand device status
	if(nand->size == 0 || nand->erasesize == 0 || nand->writesize == 0 || nand->oobsize == 0){
		printf("Error: Unrecorgnized NAND\n");
		return 0;
	}

	/*
        * Syntax is:
        *   0    1     2       3    4
        *   nand bfn [miu/ldr/app] [blk addr][miu address]
        */
	if (strcmp(cmd, "bfn") == 0){
		printf("burn boot from nand data\n");
		if (argc < 5){
			printf("argument wrong!\n");
			return 0;
		}
		U32 size = 0;
		if(argc > 5)
			size = strtoul(argv[5], NULL, 16);

		U32 addr = strtoul(argv[4], NULL, 16);
		U32 blk = strtoul(argv[3], NULL, 16);
		U32 bfn_type = 0;

		//////////////////////////////////////////////
		ret = search_tftp_buffer(&addr);
		if(ret)
			printf("[%s]: find tftp buffer addr 0x%lx for addr %s\n", __func__, addr, argv[4]);
		//////////////////////////////////////////////

		if (strncmp(argv[2],"miu",3)==0){
			printf("burn the MIU param\n");
			bfn_type = 1;
		}
		else if (strncmp(argv[2],"ldr",3)==0){
			printf("burn the LDR param\n");
			bfn_type = 2;
		}
		else if (strncmp(argv[2],"app",3)==0){
			printf("burn the APP param\n");
			bfn_type = 3;
		}
		else if (strncmp(argv[2],"hash",4)==0){
			printf("burn the Hash0 param\n");
			bfn_type = 4;
		}
		else if (strncmp(argv[2],"newcis",6)==0){
			printf("burn the new CIS and MIU format\n");
			bfn_type = 5;
		}
		else if (strncmp(argv[2],"chkcis",6)==0){
			printf("check the new CIS and MIU format\n");
			bfn_type = 6;
		}
		else{
			printf("error boot from nand parameter\n");
			return 0;
		}
		ret = drvNAND_BFN_FLASH_INFO(bfn_type,blk,addr, size);

		//////////////////////////////////////////////
		addr = strtoul(argv[4], NULL, 16);
		printf("[%s]: release addr %s\n", __func__, argv[4]);
		release_tftp_buffer(addr);
		//////////////////////////////////////////////
		return ret == 0 ? 0 : 1;
	}

	/*
	 * Syntax is:
	 *   0    1     2       3    4
	 *   nand erase [clean] [off size]
	 */
	if (strncmp(cmd, "erase", 5) == 0 || strncmp(cmd, "scrub", 5) == 0 ||strcmp(cmd, "eloop") == 0 ) {
		int eloop = !strcmp(cmd, "eloop");
		nand_erase_options_t opts;
		/* "clean" at index 2 means request to write cleanmarker */
		int clean = argc > 2 && !strcmp("clean", argv[2]);
		int o = clean ? 3 : 2;
		int scrub = !strncmp(cmd, "scrub", 5);
		int part = 0;
		int chip = 0;
		int spread = 0;
		int args = 2;

		if (cmd[5] != 0) {
			if (!strcmp(&cmd[5], ".spread")) {
				spread = 1;
			} else if (!strcmp(&cmd[5], ".part")) {
				part = 1;
				args = 1;
			} else if (!strcmp(&cmd[5], ".chip")) {
				chip = 1;
				args = 0;
			} else if(!strcmp(&cmd[5], ".chipforce") || !strcmp(&cmd[5], ".force")){
				chip = 2;
				args = 0;
			} else if (!strcmp(&cmd[5], ".partforce")){
				chip = 2;
				args = 1;
			} else {
				goto usage;
			}
		}
		else{
			if(argc ==2){//nand erase
				chip = 1;
				args = 0;
			}
			else if((argc ==3)&&(clean==0)){//nand erase partition
				part = 1;
				args = 1;
			}
		}
		/*
		 * Don't allow missing arguments to cause full chip/partition
		 * erases -- easy to do accidentally, e.g. with a misspelled
		 * variable name.
		 */
		if ((argc != o + args)&&(clean == 0)){
			goto usage;
		}
		printf("\nNAND %s: ", cmd);
		/* skip first two or three arguments, look for offset and size */

		//Mstar Start
		if(strcmp(cmd, "eloop") == 0 ){
			clean = 0;
			o = 2;
			off = 0;
			//size  = 0x4000000;
			size = nand->size;
		}
		else{
		//Mstar End
			if (arg_off_size(argc - o, argv + o, &dev, &off, &size) != 0)
				return 1;
			//#if defined(CONFIG_ENV_IS_IN_NANDUBI)
			//get last partition name from cis pni
			if(info.envtype == UBIENV)
			{
				int part_idx;
				loff_t part_size, part_off;
				if(get_part("UBILD", &part_idx, &part_off, &part_size))
				{
					printf("[%s]: Get UBILD partition information Error\n", __func__);
					return -1;
				}
				else{
					//checking offset and size
					if(chip!= 2 && scrub == 0){
						if(off < part_off + part_size){
							if(size + off > part_off+ part_size)
								size -= part_off + part_size - off;
							else
								size = 0;
							off = part_off + part_size;
							printf("Skip those blocks before UBILD partition\n");
						}
					}
				}
			}
			//#endif
		}

		nand = &nand_info[dev];

		memset(&opts, 0, sizeof(opts));
		opts.offset = off;
		opts.length = size;
		opts.jffs2  = clean;
		opts.quiet  = 1;
		opts.spread = spread;

		if (scrub){
			opts.scrub = 1;
		}
		if(eloop){
			eraseloop = strtoul(argv[2], NULL, 16);
			printf("eraseloop is %x",eraseloop);
		}
		else{
			eraseloop = 1;
		}

		for(i=0; i<eraseloop; i++){
			ret = nand_erase_opts(nand, &opts);
			printf("%4x %s\n",i,ret ? "ERROR" : "OK");
		}
		return ret == 0 ? 0 : 1;
	}

	if (strncmp(cmd, "read", 4) == 0 || strncmp(cmd, "write", 5) == 0) {
		size_t rwsize;
		int read;

		if (argc < 4)
			goto usage;

		addr = strtoul(argv[2], NULL, 16);

		//////////////////////////////////////////////
		ret = search_tftp_buffer(&addr);
		if(ret)
			printf("[%s]: find tftp buffer addr 0x%lx for addr %s\n", __func__, addr, argv[2]);
		//////////////////////////////////////////////

		read = strncmp(cmd, "read", 4) == 0; /* 1 = read, 0 = write */

		if (arg_off_size(argc - 3, argv + 3, &dev, &off, &size) != 0)
			return 1;

		if(strncmp(argv[3], "UNFDBLK", 7) == 0 ||strncmp(argv[3], "unfdblk", 7) == 0)
		{
		    info.set_ftl = 1;
		    printf("***************set_ftl = 1**********\n");
		}
		else
		{
		    info.set_ftl = 0;
		    printf("***************set_ftl = 0**********\n");
		}

		s = strchr(cmd, '.');
		if(!read){
			if(s && !strcmp(s, ".slc"))
				check_reserved_blocks(argv[3], size, 1);
			else
				check_reserved_blocks(argv[3], size, 0);
		}

		nand_fs_format=NAND_FS_FORMAT_NORMAL;
		nand = &nand_info[dev];
		rwsize = size;

		if(!s){
			if (read)
			{
                printf("%s:%d enter\n", __func__, __LINE__);
				ret = nand_read (nand, off, &rwsize, (u_char *)addr);
			}
			else
			{
                printf("%s:%d enter\n", __func__, __LINE__);
				ret = nand_write(nand, off, &rwsize, (u_char *)addr);
			}
		}
		else if (!strcmp(s, ".jffs2") || !strcmp(s, ".e") || !strcmp(s, ".i")
			|| !strcmp(s, ".raw") || !strcmp(s, ".mkbin")|| !strcmp(s,".slc")
			|| !strcmp(s,".cis") || !strcmp(s, ".partial") || !strcmp(s, ".speed")) {
			//Mstar Start
			if(!strcmp(s, ".jffs2")){
				 nand_fs_format=NAND_FS_FORMAT_JFFS2;
			}
			else if(!strcmp(s, ".e")){
				 nand_fs_format=NAND_FS_FORMAT_E;
			}
			else if(!strcmp(s, ".i")){
				 nand_fs_format=NAND_FS_FORMAT_I;
			}
			else if(!strcmp(s, ".raw")){
				 nand_fs_format=NAND_FS_FORMAT_RAW;
			}
			else if(!strcmp(s, ".mkbin")){
				 nand_fs_format=NAND_FS_FORMAT_MKBIN;
			}

			//Mstar End
			if(!strcmp(s, ".cis")){
				if(read){
					if(info.nandtype == NAND)
						ret = drvNAND_ReadCISBlk(off,(U8*)addr);
					else if(info.nandtype == SPINAND)
						ret = MDrv_SPINAND_ReadCISBlk(off,(U8*)addr);
					if(0 == ret)
						rwsize = nand->erasesize;
					else
						ret = -1;
				}
				else{
					printf("[%s]: invalid command %s\n", __func__, s);
					ret = -1;
				}
			}
			else if(info.nandtype == NAND && strcmp(s, ".slc") == 0){
				int refresh = 1;
				if(info.boottype == ROMBOOT)
				{
					int part_idx;
					loff_t part_size, part_off;
					if(get_part("MBOOT", &part_idx, &part_off, &part_size)){
						printf("Get UBILD partition information Error\n");
						return -1;
					}
					//checking offset and size
					if(off < part_off + part_size){
						refresh = 0;
					}
				}
				if (read)
					ret = nand_read_slc_skip_bad(nand, off, &rwsize,
								 (u_char *)addr, refresh);
				else
					ret = nand_write_slc_skip_bad(nand, off, &rwsize,
								  (u_char *)addr, 0);
				if(ret == -117)
				{
					printf("Read correctable ecc bit\n");
					ret= 0;
				}
			}
			else if(info.nandtype == NAND && strcmp(s, ".partial") == 0)
			{
				if(argc != 6)
				{
					printf("require 6 arguments\n");
					goto usage;
				}
				loff_t loffset = strtoul(argv[5], NULL, 16);
				if(read)
				{
					ret = nand_read_partial_skip_bad(nand, off, &rwsize, loffset,
								 (u_char *)addr);
				}
				else
				{
					ret = nand_write_partial_skip_bad(nand, off, &rwsize, loffset, (u_char*) addr, 0);
				}
			}
			else{
				if (read)
				{
                    printf("%s:%d enter\n", __func__, __LINE__);
					ret = nand_read_skip_bad(nand, off, &rwsize, (u_char *)addr);
				}
				else
				{
                    printf("%s:%d enter\n", __func__, __LINE__);
					ret = nand_write_skip_bad(nand, off, &rwsize, (u_char *)addr, 0);
				}
			}
		}
		else if (!strcmp(s, ".oob")) {
			/* out-of-band data */
			mtd_oob_ops_t ops = {
				.oobbuf = (u8 *)addr,
				.ooblen = rwsize,
				.mode = MTD_OOB_RAW
			};
			if (read)
				ret = nand->read_oob(nand, off, &ops);
			else
				ret = nand->write_oob(nand, off, &ops);
		}
		else {
			printf("Unknown nand command suffix '%s'.\n", s);
			return 1;
		}

//////////////////////////////////////////////
		//addr = strtoul(argv[2], NULL, 16);
		//printf("[%s]: release addr %s\n", __func__, argv[2]);
		//release_tftp_buffer(addr);
//////////////////////////////////////////////

		info.set_ftl = 0;
		return ret == 0 ? 0 : 1;
	}
usage:
	return cmd_usage(cmdtp);
}

U_BOOT_CMD(
	nand, CONFIG_SYS_MAXARGS, 1, do_nand,
	"NAND sub-system",
	"info - show available NAND devices\n"
	"nand device [dev] - show or set current device\n"
	"nand read - addr off|partition size\n"
	"nand write - addr off|partition size\n"
	"    read/write 'size' bytes starting at offset 'off'\n"
	"    to/from memory address 'addr', skipping bad blocks.\n"
#ifdef CONFIG_CMD_NAND_YAFFS
	"nand write.yaffs - addr off|partition size\n"
	"    write 'size' bytes starting at offset 'off' with yaffs format\n"
	"    from memory address 'addr', skipping bad blocks.\n"
#endif
	"nand erase[.spread] [clean] off size - erase 'size' bytes "
	"from offset 'off'\n"
	"    With '.spread', erase enough for given file size, otherwise,\n"
	"    'size' includes skipped bad blocks.\n"
	"nand erase.part [clean] partition - erase entire mtd partition'\n"
	"nand erase.chip [clean] - erase entire chip'\n"
	"nand bad - show bad blocks\n"
	"nand dump[.oob] off - dump page\n"
	"nand scrub off size | scrub.part partition | scrub.chip\n"
	"    really clean NAND erasing bad blocks (UNSAFE)\n"
	"nand markbad off [...] - mark bad block(s) at offset (UNSAFE)\n"
	"nand biterr off - make a bit error at offset (UNSAFE)"
#ifdef CONFIG_CMD_NAND_LOCK_UNLOCK
	"\n"
	"nand lock [tight] [status]\n"
	"    bring nand to lock state or display locked pages\n"
	"nand unlock [offset] [size] - unlock section"
#endif
#ifdef CONFIG_ENV_OFFSET_OOB
	"\n"
	"nand env.oob - environment offset in OOB of block 0 of"
	"    first device.\n"
	"nand env.oob set off|partition - set enviromnent offset\n"
	"nand env.oob get - get environment offset\n"
#endif

#if defined(IF_IP_VERIFY) && IF_IP_VERIFY
	"nand ipverify - IP Verify test\n"
	"nand test readloop init blk\n"
	"nand test readloop test blk\n"
  #endif
	"nand dd nand2usb/usb2nand [usb port num] - dump/restore nand raw data between nand and usb\n"
);

#define MAX_PAGE_SIZE       (16384 + 1280)

unsigned char datbuf[MAX_PAGE_SIZE];

int check_all_0xff(unsigned char *buf, int len)
{
	int i;

	if(!buf){
		printf("[%s]:NULL pointer\n", __func__);
		return -1;
	}
	for(i = 0; i < len; i++){
		if(buf[i] != 0xFF)
			return 0;
	}
	return 1;
}

int address_last_none_empty_block(nand_info_t *nand, struct part_info *part, loff_t *res)
{
	int len, ret;
	loff_t pos;
	unsigned char *oobbuf;
	struct mtd_oob_ops opts;

	oobbuf = datbuf + nand->oobblock;
	memset(&opts, 0, sizeof(struct mtd_oob_ops));

	for(pos = part->offset + part->size - nand->erasesize; pos > part->offset; pos -= nand->erasesize){
		ret = nand->block_isbad(nand, pos);
		if(ret < 0){
			printf("[%s]: check bad failed\n", __func__);
			return -1;
		}
		if(ret)
			continue;

		for(len = 0; len < nand->erasesize; len += nand->oobblock){
			memset(datbuf, 0xFF, nand->oobblock + nand->oobsize);
			opts.datbuf = (unsigned char *)datbuf;
			opts.len = nand->oobblock;
			opts.oobbuf = (unsigned char *)oobbuf;
			opts.ooblen = nand->oobsize;
			opts.mode = MTD_OOB_RAW;
			ret = nand->read_oob(nand, pos + len, &opts);
			if(ret){
				printf("[%s]: read oob failed\n", __func__);
				return -1;
			}
			ret = check_all_0xff(datbuf, nand->oobblock + nand->oobsize);
			if(ret < 0){
				printf("[%s]: check spare all 0xFF failed\n", __func__);
				return ret;
			}
			if(!ret){
				break;
			}
		}
		if(len < nand->erasesize)
			break;
	}

	*res = pos;
	return 0;
}

int do_print_partition_valid_block(nand_info_t *nand, struct part_info *part, int res[3])
{
	int count = 0, ret;
	loff_t pos;
	size_t retlen;
	int idx = 0;

	memset(datbuf, 0, MAX_PAGE_SIZE);

	//mboot for romboot
	if((0 == strcmp("MBOOT", part->name)) || (0 == strcmp("mboot", part->name))){
		int sboot_blk = 0;

		//hiden cis
		for(pos = 0; pos < TOTAL_CIS_BLOCK * nand->erasesize; pos += nand->erasesize){
			memset(datbuf, 0xFF, nand->oobblock);
			ret = nand->read(nand, pos, nand->oobblock, &retlen, datbuf);
			if(ret < 0){
				printf("[%s]: Hidden CIS check bad failed\n", __func__);
				return -1;
			}
			if(check_all_0xff(datbuf, nand->oobblock) == 0)
				count = pos / nand->erasesize + 1;
		}
		if(res)
			res[0] = count;

		if(info.boottype == BFN){
			count = 0;
			idx++;
			//hiden cis
			for(pos = TOTAL_CIS_BLOCK; pos < (TOTAL_CIS_BLOCK+2) * nand->erasesize; pos += nand->erasesize){
				memset(datbuf, 0xFF, nand->oobblock);
				ret = nand->read(nand, pos, nand->oobblock, &retlen, datbuf);
				if(ret < 0){
					printf("[%s]: Hidden CIS check bad failed\n", __func__);
					return -1;
				}
				if(check_all_0xff(datbuf, nand->oobblock) == 0)
					count = pos / nand->erasesize + 1;
			}
			if(res)
				res[idx] = count - TOTAL_CIS_BLOCK;
		}

		//mboot
		count = 0;
		idx++;
		sboot_blk = do_nsbootblk(NULL, 0, 2, NULL);
		ret = address_last_none_empty_block(nand, part, &pos);

		if(ret < 0){
			printf("[%s]: get last none empty block failed\n", __func__);
		}
		for(; pos >= part->offset; pos -= nand->erasesize){
			ret = nand->block_isbad(nand, pos);
			if(ret < 0){
				printf(" %s mtd partition check bad failed\n", __func__);
				return -1;
			}
			if(ret)
				continue;
			count++;
			if(pos == 0)
				break;
		}

		count += sboot_blk;
		if(res)
			res[idx] = count;
		return 0;
	}

    if((0 == strcmp("IPL0", part->name)) || (0 == strcmp("ipl0", part->name))){
		int sboot_blk = 0;

		//hiden cis
		for(pos = 0; pos < part->offset; pos += nand->erasesize){
			memset(datbuf, 0xFF, nand->oobblock);
			ret = nand->read(nand, pos, nand->oobblock, &retlen, datbuf);
			if(ret < 0){
				printf("[%s]: Hidden CIS check bad failed\n", __func__);
				return -1;
			}
			if(check_all_0xff(datbuf, nand->oobblock) == 0)
				count = pos / nand->erasesize + 1;
		}
		if(res)
			res[0] = count;
        printf("%s:%d count=%d\n", __func__, __LINE__, count);
		//mboot
		count = 0;
		idx++;
		ret = address_last_none_empty_block(nand, part, &pos);

		if(ret < 0){
			printf("[%s]: get last none empty block failed\n", __func__);
		}
		for(; pos >= part->offset; pos -= nand->erasesize){
			ret = nand->block_isbad(nand, pos);
			if(ret < 0){
				printf(" %s mtd partition check bad failed\n", __func__);
				return -1;
			}
			if(ret)
				continue;
			count++;
			if(pos == 0)
				break;
		}

		if(res)
			res[idx] = count;
        printf("%s:%d count=%d\n", __func__, __LINE__, count);
		return 0;
	}

	//mtd partition
	ret = address_last_none_empty_block(nand, part, &pos);
	if(ret < 0){
		printf("get last none empty block failed\n");
	}
	for(; pos >= part->offset; pos -= nand->erasesize){
		ret = nand->block_isbad(nand, pos);
		if(ret < 0){
			printf(" %s mtd partition check bad failed\n", __func__);
			return -1;
		}
		if(ret)
			continue;
		count++;
		if(pos == 0)
			break;
	}
	if(res)
		res[0] = count;
    printf(" %s:%d count=%d\n", __func__, __LINE__, res[0]);
	return 0;
}

#define BFN_LDR_START 10
#define BFN_LDR_BLK 2

/* print ubi partition and environment parameter information */
int do_print_nandbin_info(cmd_tbl_t *cmdtp, int flag, int argc,char * const argv[])
{
	struct mtd_device *dev=NULL;
	struct part_info *part=NULL;
	nand_info_t *nand=NULL;
	u8 pnum=0;
	u64 page_size=0;
	u64 off_blk=0, size_blk=0;
	u64 end=0, end_blk=0;
	u32 partnum=0;
	u16 inc = 0;
	int res[3] = {0, 0, 0};
	int fd;
	char path[512];
	u64 bfn_mboot_total_blk = 0;
	u64 bfn_mboot_start_blk = 0;

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s", defpath, "leap.def");
	fd = open(path, O_RDWR | O_CREAT | O_TRUNC, S_IRWXU | S_IRWXG | S_IRWXO);
	if(fd < 0){
		printf("[%s]: open %s failed\n", __func__, path);
		return -1;
	}
	close(fd);
	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s", defpath, "snchk.def");
	fd = open(path, O_RDWR | O_CREAT | O_TRUNC, S_IRWXU | S_IRWXG | S_IRWXO);
	if(fd < 0){
		printf("[%s]: open %s failed\n", __func__, path);
		return -1;
	}
	close(fd);

#if 0
	if ((mtdparts_init() == 0) &&
		((find_dev_and_part("UBI", &dev, &pnum, &part) == 0) ||
		(find_dev_and_part("ubi", &dev, &pnum, &part) == 0))) {
		if (dev->id->type != MTD_DEV_TYPE_NAND) {
			puts("not a NAND device\n");
			return -1;
		}
	}

	if(part == NULL){
		printf("%s: Error: null part, at %d\n", __func__, __LINE__);
		return -1;
	}
#else
	dev = current_mtd_dev;
#endif

	if(!dev){
		printf("[%s]:Error, NULL mtd device pointer\n", __func__);
		return -1;
	}
	nand = &nand_info[0];
	page_size = nand->oobblock + nand->oobsize;

	for(partnum = 0; partnum < dev->num_parts; partnum++){
		part = (struct part_info *)mtd_part_info(dev, partnum);
		if(part == NULL)
		{
			printf("%s: Error: null part, at %d\n", __func__, __LINE__);
			return -1;
		}
		if((strcmp(part->name, "NA") == 0 ||strcmp(part->name, "na") == 0) && partnum == dev->num_parts - 1){
			term_def_leap();
			term_def_snchk();
			break;
		}

		//printf(" partname=%s\r\n",part->name);

		if(inc)
			printf("part#%d\n", partnum + inc);
		else
			printf("part#%d\n", partnum);

		if (part->size & (nand->erasesize - 1))
		{
			printf("Warning: %s partition size is not aligned with nand block size\r\n    part size=0x%llx, block size=0x%x\r\n",
				part->name, part->size, nand->erasesize);
		}

		if (part->offset & (nand->erasesize - 1))
		{
			printf("Warning: %s partition offset is not aligned with nand block size\r\n    part offset=0x%llx, block size=0x%x\r\n",
				part->name, part->offset, nand->erasesize);
		}

		off_blk = part->offset / nand->erasesize;
		size_blk = part->size / nand->erasesize;
		end = part->offset + part->size - 1;
		end_blk = (part->offset + part->size) / nand->erasesize - 1;

		do_print_partition_valid_block(nand, part, res);

		if((0 == strcmp(part->name, "MBOOT")) ||  (0 == strcmp(part->name, "mboot"))){
			int sboot_blk = 0;
			int sboot_len = 0;
			int sboot_off = 0;

			//dump cis partition info
			printf("cis partition: \r\n    start=0(by main:%d),  start=0(by main+spare:%lld)\r\n", nand->oobblock, page_size);
			printf("    start block=0(Hex:0)\r\n");
			printf("    size=0x%llx(main),  size=0x%llx(main+spare)\r\n",
					(u64)10 * nand->erasesize, (u64)((10 * nand->erasesize / nand->oobblock) * page_size));
			printf("    size block=%lld(Hex:0x%llx)\r\n", (u64)10, (u64)10);
			printf("    process block=%lld(Hex:0x%llx)\r\n", (u64)res[0], (u64)res[0]);
			printf("    end=0x%llx(main),  end=0x%llx(main+spare)\r\n",
				(u64)10 * nand->erasesize - 1, (u64)(((10 * nand->erasesize) / nand->oobblock) * page_size - 1));
			printf("    end block=%lld(Hex:0x%llx)\r\n\n", (u64)9, (u64)9);

			write_partition_def_leap((ulong)0, (ulong)9, (ulong)res[0], 0);
			write_partition_def_snchk((ulong)0, (ulong)9, (ulong)res[0], 0, "CIS");

            if(info.boottype == BFN){
				//dump ldr partition info
				inc++;
				printf("part#%d\n",partnum + inc);
				printf("ldr partition: \r\n    start=0x%llx(by main:%d),  start=0x%llx(by main+spare:%lld)\r\n",
					(u64)10 * nand->erasesize,nand->oobblock, (u64)((10 * nand->erasesize / nand->oobblock) * page_size), page_size);
				printf("    start block=10(Hex:10)\r\n");
				printf("    size=0x%llx(main),  size=0x%llx(main+spare)\r\n",
						(u64)2 * nand->erasesize, (u64)((2 * nand->erasesize / nand->oobblock) * page_size));
				printf("    size block=%lld(Hex:0x%llx)\r\n", (u64)2, (u64)2);
				printf("    process block=%lld(Hex:0x%llx)\r\n", (u64)res[1], (u64)res[1]);
				printf("    end=0x%llx(main),  end=0x%llx(main+spare)\r\n",
					(u64)12 * nand->erasesize - 1, (u64)(((12 * nand->erasesize) / nand->oobblock) * page_size - 1));
				printf("    end block=%lld(Hex:0x%llx)\r\n\n", (u64)11, (u64)11);

				write_partition_def_leap((ulong)10, (ulong)11, (ulong)res[1], inc);
				write_partition_def_snchk((ulong)10, (ulong)11, (ulong)res[1], inc, "LDR");
            }

			//dump mboot partition info
			sboot_blk = do_nsbootblk(NULL, 0, 2, NULL);
			sboot_len = sboot_blk * nand->erasesize;
			off_blk = (part->offset - sboot_len) / nand->erasesize;
			size_blk = (part->size + sboot_len) / nand->erasesize;

/*
			if(info.boottype == BFN){
				bfn_mboot_total_blk = size_blk + BFN_LDR_BLK;
				bfn_mboot_start_blk = BFN_LDR_START;
				continue;
			}
*/
			//end has not changed
			inc++;
			end = part->offset + part->size - 1;
			end_blk = (part->offset + part->size) / nand->erasesize - 1;
			printf("part#%d\n",partnum + inc);
			printf("%s partition: \r\n    start=0x%llx(by main:%d),  start=0x%llx(by main+spare:%lld)\r\n",
                   		 part->name, part->offset - sboot_len, nand->oobblock,
                   		 (((part->offset - sboot_len) / nand->oobblock) * page_size), page_size);
			printf("    start block=%lld(Hex:0x%llx)\r\n",
                   		 off_blk, off_blk);
        		printf("    size=0x%llx(main),  size=0x%llx(main+spare)\r\n",
                    		part->size + sboot_len, (((part->size + sboot_len) / nand->oobblock) * page_size));
        		printf("    size block=%lld(Hex:0x%llx)\r\n",
                    		size_blk, size_blk);
			printf("    process block=%lld(Hex:0x%llx)\r\n", (u64)res[inc], (u64)res[inc]);
        		printf("    end=0x%llx(main),  end=0x%llx(main+spare)\r\n",
				end, (((end + 1) / nand->oobblock) * page_size - 1));
        		printf("    end block=%lld(Hex:0x%llx)\r\n\n",
				end_blk, end_blk);

			write_partition_def_leap((ulong)off_blk, (ulong)end_blk, (ulong)res[inc], inc);
			write_partition_def_snchk((ulong)off_blk, (ulong)end_blk, (ulong)res[inc], inc, part->name);

			continue;
		}
/*
		if(info.boottype == BFN && ((0 == strcmp(part->name, "MBOOTBAK")) ||  (0 == strcmp(part->name, "mbootbak")))){
			end = part->offset + part->size - 1;
			end_blk = (part->offset + part->size) / nand->erasesize - 1;

			printf("part#%d\n",partnum + 1);
			printf("%s partition: \r\n    start=0x%llx(by main:%d),  start=0x%llx(by main+spare:%lld)\r\n",
                   		 "BFN LOADER & MBOOT & MBOOTBAK",
                   		 bfn_mboot_start_blk * nand->erasesize, nand->oobblock,
                   		 bfn_mboot_start_blk * nand->erasesize / nand->oobblock * page_size, page_size);
			printf("    start block=%lld(Hex:0x%llx)\r\n",
                   		 bfn_mboot_start_blk, bfn_mboot_start_blk);
			printf("    size=0x%llx(main),  size=0x%llx(main+spare)\r\n",
                    		(end_blk - bfn_mboot_start_blk + 1) * nand->erasesize,
                    		(end_blk - bfn_mboot_start_blk + 1) * nand->erasesize / nand->oobblock * page_size);
			printf("    size block=%lld(Hex:0x%llx)\r\n",
                    		end_blk - bfn_mboot_start_blk + 1, end_blk - bfn_mboot_start_blk + 1);
			printf("    process block=%lld(Hex:0x%llx)\r\n",
				(u64)res[0] + bfn_mboot_total_blk,
				(u64)res[0] + bfn_mboot_total_blk);
        		printf("    end=0x%llx(main),  end=0x%llx(main+spare)\r\n",
				end, (((end + 1) / nand->oobblock) * page_size - 1));
        		printf("    end block=%lld(Hex:0x%llx)\r\n\n",
				end_blk, end_blk);

			write_partition_def_leap((ulong)bfn_mboot_start_blk, (ulong)end_blk, (ulong)res[0] + bfn_mboot_total_blk, 1);
			write_partition_def_snchk((ulong)bfn_mboot_start_blk, (ulong)end_blk, (ulong)res[0] + bfn_mboot_total_blk, 1, "BFN");

			inc = 1;
			continue;
		}
*/
		printf("%s partition: \r\n    start=0x%llx(by main:%d),  start=0x%llx(by main+spare:%lld)\r\n",
			part->name, part->offset, nand->oobblock,
			((part->offset / nand->oobblock) * page_size), page_size);
		printf("    start block=%lld(Hex:0x%llx)\r\n",
			off_blk, off_blk);
		printf("    size=0x%llx(main),  size=0x%llx(main+spare)\r\n",
			part->size, ((part->size / nand->oobblock) * page_size));
		printf("    size block=%lld(Hex:0x%llx)\r\n",
			size_blk, size_blk);
		printf("    process block=%lld(Hex:0x%llx)\r\n", (u64)res[0], (u64)res[0]);
		printf("    end=0x%llx(main),  end=0x%llx(main+spare)\r\n",
			end, (((end + 1) / nand->oobblock) * page_size - 1));
		printf("    end block=%lld(Hex:0x%llx)\r\n\n",
			end_blk, end_blk);

		if(0 == strcmp(part->name, "TBL") || 0 == strcmp(part->name, "tbl") ||
			0 == strcmp(part->name, "CTRL") || 0 == strcmp(part->name, "ctrl"))
			res[0] = 1;


		if(partnum < dev->num_parts - 1){
			write_partition_def_leap((ulong)off_blk, (ulong)end_blk, (ulong)res[0], partnum + inc);
			write_partition_def_snchk((ulong)off_blk, (ulong)end_blk, (ulong)res[0], partnum + inc, part->name);
		}
		else{
			write_partition_def_leap((ulong)off_blk, (ulong)end_blk, (ulong)res[0], (partnum +inc) | 0x8000);
			write_partition_def_snchk((ulong)off_blk, (ulong)end_blk, (ulong)res[0], (partnum +inc) | 0x8000, part->name);
		}
	}

	printf("Nand Flash basic information: \r\n    main=%d bytes(Hex:0x%x),  spare=%d bytes(Hex:0x%x)\r\n",
		nand->oobblock, nand->oobblock, nand->oobsize, nand->oobsize);
	printf("    main+spare=%lld bytes(Hex:0x%llx),  page number per block=%d\r\n",
		page_size, page_size, (nand->erasesize / nand->oobblock));
	printf("    block size=0x%x(%dKB, main),  block size=0x%llx(main+spare)\r\n\n",
		nand->erasesize, (nand->erasesize >> 10), ((nand->erasesize / nand->oobblock) * page_size));

	return 0;
}
U_BOOT_CMD(
	bininfo, 2, 1,  do_print_nandbin_info,
	"bininfo -print parameter information of each partition and environment\n",
	"command: bininfo usbport\n"
);

/* print ubi partition and environment parameter information */
int do_print_nand_bin_info(cmd_tbl_t *cmdtp, int flag, int argc,char * const argv[])
{
	struct mtd_device *dev=NULL;
	struct part_info *part=NULL;
	nand_info_t *nand=NULL;
	u8 pnum=0;
	u64 page_size=0;
	u64 off_blk=0, size_blk=0;
	u64 end=0, end_blk=0;
	u32 partnum=0;
	u16 inc = 0;
	int res[3] = {0, 0, 0};
	int fd;
	char path[512];
	u64 bfn_mboot_total_blk = 0;
	u64 bfn_mboot_start_blk = 0;

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s", defpath, "leap.def");
	fd = open(path, O_RDWR | O_CREAT | O_TRUNC, S_IRWXU | S_IRWXG | S_IRWXO);
	if(fd < 0){
		printf("[%s]: open %s failed\n", __func__, path);
		return -1;
	}
	close(fd);
	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s", defpath, "snchk.def");
	fd = open(path, O_RDWR | O_CREAT | O_TRUNC, S_IRWXU | S_IRWXG | S_IRWXO);
	if(fd < 0){
		printf("[%s]: open %s failed\n", __func__, path);
		return -1;
	}
	close(fd);

	dev = current_mtd_dev;

	if(!dev){
		printf("[%s]:Error, NULL mtd device pointer\n", __func__);
		return -1;
	}
	nand = &nand_info[0];
	page_size = nand->oobblock + nand->oobsize;

	for(partnum = 0; partnum < dev->num_parts; partnum++){
		part = (struct part_info *)mtd_part_info(dev, partnum);
		if(part == NULL)
		{
			printf("%s: Error: null part, at %d\n", __func__, __LINE__);
			return -1;
		}
		if((strcmp(part->name, "NA") == 0 ||strcmp(part->name, "na") == 0) && partnum == dev->num_parts - 1){
			term_def_leap();
			term_def_snchk();
			break;
		}

		//printf(" partname=%s\r\n",part->name);

		if(inc)
			printf("part#%d\n", partnum + inc);
		else
			printf("part#%d\n", partnum);

		if (part->size & (nand->erasesize - 1))
		{
			printf("Warning: %s partition size is not aligned with nand block size\r\n    part size=0x%llx, block size=0x%x\r\n",
				part->name, part->size, nand->erasesize);
		}

		if (part->offset & (nand->erasesize - 1))
		{
			printf("Warning: %s partition offset is not aligned with nand block size\r\n    part offset=0x%llx, block size=0x%x\r\n",
				part->name, part->offset, nand->erasesize);
		}

		off_blk = part->offset / nand->erasesize;
		size_blk = part->size / nand->erasesize;
		end = part->offset + part->size - 1;
		end_blk = (part->offset + part->size) / nand->erasesize - 1;

		do_print_partition_valid_block(nand, part, res);

		if((0 == strcmp(part->name, "IPL0")) ||  (0 == strcmp(part->name, "ipl0"))){
			int sboot_off = 0;

            end = part->offset - 1;
			end_blk = part->offset/nand->erasesize - 1;
			//dump cis partition info
			printf("cis partition: \r\n    start=0(by main:%d),  start=0(by main+spare:%lld)\r\n", nand->oobblock, page_size);
			printf("    start block=0(Hex:0)\r\n");
			printf("    size=0x%llx(main),  size=0x%llx(main+spare)\r\n",
					(u64)part->offset, (u64)((part->offset/nand->oobblock) * page_size));
			printf("    size block=%lld(Hex:0x%llx)\r\n", (u64)part->offset/nand->erasesize, (u64)part->offset/nand->erasesize);
			printf("    process block=%lld(Hex:0x%llx)\r\n", (u64)res[0], (u64)res[0]);
			printf("    end=0x%llx(main),  end=0x%llx(main+spare)\r\n",
				end , (u64)((part->offset / nand->oobblock) * page_size - 1));
			printf("    end block=%lld(Hex:0x%llx)\r\n\n", end_blk, end_blk);

			write_partition_def_leap((ulong)0, (ulong)end_blk, (ulong)res[0], 0);
			write_partition_def_snchk((ulong)0, (ulong)end_blk, (ulong)res[0], 0, "CIS");

			//end has not changed
			inc++;
			end = part->offset + part->size - 1;
			end_blk = (part->offset + part->size) / nand->erasesize - 1;
			printf("part#%d\n",partnum + inc);
			printf("%s partition: \r\n    start=0x%llx(by main:%d),  start=0x%llx(by main+spare:%lld)\r\n",
                   		 part->name, part->offset, nand->oobblock,
                   		 ((part->offset/nand->oobblock) * page_size), page_size);
			printf("    start block=%lld(Hex:0x%llx)\r\n",
                   		 off_blk, off_blk);
        	printf("    size=0x%llx(main),  size=0x%llx(main+spare)\r\n",
                    		part->size, ((part->size / nand->oobblock) * page_size));
        	printf("    size block=%lld(Hex:0x%llx)\r\n",
                    		size_blk, size_blk);
			printf("    process block=%lld(Hex:0x%llx)\r\n", (u64)res[inc], (u64)res[inc]);
        	printf("    end=0x%llx(main),  end=0x%llx(main+spare)\r\n",
				end, (((end + 1) / nand->oobblock) * page_size - 1));
        	printf("    end block=%lld(Hex:0x%llx)\r\n\n",
				end_blk, end_blk);

			write_partition_def_leap((ulong)off_blk, (ulong)end_blk, (ulong)res[inc], inc);
			write_partition_def_snchk((ulong)off_blk, (ulong)end_blk, (ulong)res[inc], inc, part->name);

			continue;
		}

		printf("%s partition: \r\n    start=0x%llx(by main:%d),  start=0x%llx(by main+spare:%lld)\r\n",
			part->name, part->offset, nand->oobblock,
			((part->offset / nand->oobblock) * page_size), page_size);
		printf("    start block=%lld(Hex:0x%llx)\r\n",
			off_blk, off_blk);
		printf("    size=0x%llx(main),  size=0x%llx(main+spare)\r\n",
			part->size, ((part->size / nand->oobblock) * page_size));
		printf("    size block=%lld(Hex:0x%llx)\r\n",
			size_blk, size_blk);
		printf("    process block=%lld(Hex:0x%llx)\r\n", (u64)res[0], (u64)res[0]);
		printf("    end=0x%llx(main),  end=0x%llx(main+spare)\r\n",
			end, (((end + 1) / nand->oobblock) * page_size - 1));
		printf("    end block=%lld(Hex:0x%llx)\r\n\n",
			end_blk, end_blk);

		if(0 == strcmp(part->name, "TBL") || 0 == strcmp(part->name, "tbl") ||
			0 == strcmp(part->name, "CTRL") || 0 == strcmp(part->name, "ctrl"))
			res[0] = 1;


		if(partnum < dev->num_parts - 1){
			write_partition_def_leap((ulong)off_blk, (ulong)end_blk, (ulong)res[0], partnum + inc);
			write_partition_def_snchk((ulong)off_blk, (ulong)end_blk, (ulong)res[0], partnum + inc, part->name);
		}
		else{
			write_partition_def_leap((ulong)off_blk, (ulong)end_blk, (ulong)res[0], (partnum +inc) | 0x8000);
			write_partition_def_snchk((ulong)off_blk, (ulong)end_blk, (ulong)res[0], (partnum +inc) | 0x8000, part->name);
		}
	}

	printf("Nand Flash basic information: \r\n    main=%d bytes(Hex:0x%x),  spare=%d bytes(Hex:0x%x)\r\n",
		nand->oobblock, nand->oobblock, nand->oobsize, nand->oobsize);
	printf("    main+spare=%lld bytes(Hex:0x%llx),  page number per block=%d\r\n",
		page_size, page_size, (nand->erasesize / nand->oobblock));
	printf("    block size=0x%x(%dKB, main),  block size=0x%llx(main+spare)\r\n\n",
		nand->erasesize, (nand->erasesize >> 10), ((nand->erasesize / nand->oobblock) * page_size));

	return 0;
}

U_BOOT_CMD(
	nandbininfo, 2, 1,  do_print_nand_bin_info,
	"bininfo -print parameter information of each partition and environment\n",
	"command: bininfo usbport\n"
);

/* generates def file which decricbes partition information for nand programmer */
int write_partition_def_leap(ulong start, ulong end, ulong process, ulong pos)
{
	int fd=0, ret;
	u_char buf[16], filename[512];
	u_char def_header[16] = {0x47, 0x52, 0x4f, 0x55, 0x50, 0x20, 0x44, 0x45, 0x46, 0x49, 0x4e, 0x45, 0x32, 0x0, 0x0, 0x0};
	ulong cnt, last;

	cnt = pos & ~0x8000;
	last = pos & 0x8000;

	if(!process)
		process = 1;

	/* Write to usb for binary file making
	Initialize something and open usb file */
	memset(filename, 0, sizeof(filename));
	sprintf(filename, "%sleap.def", defpath);
	fd = open(filename, O_RDWR | O_APPEND);
	if (fd < 0){
		printf("[%s]: open file %s failed\n", __func__, filename);
		return -1;
	}

	if (cnt == 0){
		memcpy(buf, def_header, 0x10);
		ret = write(fd, (unsigned char *)buf, 0x10);
		if(ret < 0){
			printf("[%s]: write file %s failed\n", __func__, filename);
			goto out;
		}
	}

	*((ulong *)buf) = __cpu_to_le32(0x00000001);
	*((ulong *)(buf + 0x4)) = __cpu_to_le32(start);
	*((ulong *)(buf + 0x8)) = __cpu_to_le32(end);
	*((ulong *)(buf + 0xc)) = __cpu_to_le32(process);

	ret = write(fd, (unsigned char *) buf, 0x10);
	if(ret < 0){
		printf("[%s]: write file %s failed\n", __func__, filename);
		goto out;
	}

	if (last){
		memset(buf, 0xFF, 16);
		ret = write(fd, (unsigned char *) buf, 0x10);
		if(ret < 0){
			printf("[%s]: write file %s failed\n", __func__, filename);
			goto out;
		}
	}

	fsync(fd);
	close(fd);
	return 0;

out:
	close(fd);
	return ret;
}

/* generates def file which decricbes partition information for nand programmer */
int write_partition_def_snchk(ulong start, ulong end, ulong process, ulong pos, const char * partname)
{
	int fd=0, ret;
	u_char buf[16], filename[512];
	ulong cnt, last;

	cnt = pos & ~0x8000;
	last = pos & 0x8000;

	if(!process)
		process = 1;

	/* Write to usb for binary file making
	*Initialize something and open usb file
	*/
	memset(filename, 0, sizeof(filename));
	sprintf(filename, "%ssnchk.def", defpath);
	fd = open(filename, O_RDWR | O_APPEND);
	if (fd < 0){
		printf("[%s]: open file %s failed\n", __func__, filename);
		return -1;
	}

	*((ulong *)buf) = __cpu_to_le32(start);
	*((ulong *)(buf+ 0x4)) = __cpu_to_le32(end);
	*((ulong *)(buf + 0x8)) = __cpu_to_le32(process);
	memcpy(buf + 0xc, partname, 4);

	ret = write(fd, (unsigned char *) buf, 0x10);
	if(ret < 0){
		printf("[%s]: write file %s failed\n", __func__, filename);
		goto  out;
	}

	if (last){
		memset(buf, 0xFF, 16);
		ret = write(fd, (unsigned char *) buf, 0x10);
		if(ret < 0){
			printf("[%s]: write file %s failed\n", __func__, filename);
			goto  out;
		}
	}

	fsync(fd);
	close(fd);
	return 0;

out:
	close(fd);
	return ret;
}

int term_def_snchk()
{
	int fd = 0, ret = 0;
	u_char buf[16], filename[512];

	memset(filename, 0, sizeof(filename));
	sprintf(filename, "%ssnchk.def", defpath);
	fd = open(filename, O_RDWR | O_APPEND);
	if (fd < 0){
		printf("[%s]: open file %s failed\n", __func__, filename);
		return -1;
	}

	memset(buf, 0xFF, 0x10);

	ret = write(fd, (unsigned char *)buf, 0x10);
	if(ret < 0){
		printf("[%s]: write file %s failed\n", __func__, filename);
	}

out:
	fsync(fd);
	close(fd);
	return ret;
}

int term_def_leap()
{
	int fd=0, ret = 0;
	u_char buf[16], filename[512];

	memset(filename, 0, sizeof(filename));
	sprintf(filename, "%sleap.def", defpath);
	fd = open(filename, O_RDWR | O_APPEND);
	if (fd < 0){
		printf("[%s]: open file %s failed\n", __func__, filename);
		return -1;
	}

	memset(buf, 0xFF, 0x10);

	ret = write(fd, (unsigned char *)buf, 0x10);
	if(ret < 0){
		printf("[%s]: write file %s failed\n", __func__, filename);
	}

out:
	fsync(fd);
	close(fd);
	return ret;
}

/*
 * argv[1] = cis address
 * argv[2] = PPM address
 * argv[3] = PNI address
 * argv[4] = bootloader address
 * argv[5] = bootloader size
 * argv[6] = uboot address
 * argv[7] = uboot size
 *
 * */
int do_nandupdateCIS_BL_UBOOT(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
	long unsigned int u32_block_PBA = 0xA, u32_bl_PBA1, u32_bl_PBA2, u32_uboot_PBA0, u32_uboot_PBA1;
	long unsigned int u32_buf, u32_size, u32_Err = 0, u32_PPM_buf, u32_PNI_buf = 0, u32_tmp;
	NAND_DRIVER *pNandDrv = (NAND_DRIVER*)drvNAND_get_DrvContext_address();
	NAND_DRIVER *pNandDrv_tmp = (NAND_DRIVER*)malloc(sizeof(NAND_DRIVER));
	NAND_FLASH_INFO_t * pNandInfo = (NAND_FLASH_INFO_t* )malloc(512);
	extern SPI_NAND_DRIVER_t gtSpiNandDrv;
	SPINAND_FLASH_INFO_TAG_t *pSpiNandInfoTag = (SPINAND_FLASH_INFO_TAG_t *)malloc(512);
	SPINAND_FLASH_INFO_t *pSpiNandInfo = &pSpiNandInfoTag->tSpiNandInfo;
	PARTITION_INFO_t *pPartInfo;
	PARTITION_RECORD_t *pRecord;
	unsigned int u32_block_ubootbak;
	unsigned short u16_PartType;
	int k, ret, bootnum;
	unsigned char u8_HasPNI;

	if(argc < 3)
	{
		printf("Too few arguements\n");
		return cmd_usage(cmdtp);
	}

	//support k7a removing mbootbak
	if(info.disablembootbak)
		bootnum = 1;
	else
		bootnum = 2;

	u32_buf = strtoul(argv[1], NULL, 16);
	u32_PPM_buf = strtoul(argv[2], NULL, 16);

	/*
	 *seaech nni & ppm tftp buffer
	*/
	printf("[%s]: Search nni ppm pni tftp buffer\n", __func__);
	ret = search_tftp_buffer(&u32_buf);
	if(ret)
		printf("[%s]: find nni tftp buffer, addr=0x%lx\n", __func__, u32_buf);
	else
		printf("[%s]: find NO nni tftp buffer, using original addr=0x%lx\n", __func__, u32_buf);
	ret = search_tftp_buffer(&u32_PPM_buf);
	if(ret)
		printf("[%s]: find ppm tftp buffer, add=0x%lx\n", __func__, u32_PPM_buf);
	else
		printf("[%s]: find NO ppm tftp buffer, using original addr=0x%lx\n", __func__, u32_PPM_buf);


	if(argc >= 4)
		u32_PNI_buf = strtoul(argv[3], NULL, 16);

	/*
	 *search pni tftp buffer
	*/
	if(u32_PNI_buf){
		ret = search_tftp_buffer(&u32_PNI_buf);
		if(ret)
			printf("[%s]: find pni tftp buffer add0x%lx for addr %s\n", __func__, u32_PNI_buf, argv[3]);
		else
			printf("[%s]: find NO pni tftp buffer, using original addr=0x%lx\n", __func__, u32_PNI_buf);
	}

	if(info.nandtype == NAND)
		memcpy(pNandDrv_tmp, pNandDrv, sizeof(NAND_DRIVER));

	if(info.nandtype == NAND)
		u32_Err = search_cis_in_DRAM((U8*)u32_buf, (U8*)u32_PPM_buf, (U8*) u32_PNI_buf, pNandInfo);
	else if(info.nandtype == SPINAND)
		u32_Err = MDrv_SPINAND_SearchCIS_in_DRAM((U8*)u32_buf, (U8*)u32_PPM_buf, (U8*) u32_PNI_buf, pSpiNandInfoTag);
	if(u32_Err != 0){
		//search cis fail
		printf("[%s]: search cis in DRAM fail\n", __func__);
		goto Error;
	}

	if(info.nandtype == NAND)
		u8_HasPNI = pNandDrv->u8_HasPNI;
	else if(info.nandtype == SPINAND)
		u8_HasPNI = gtSpiNandDrv.u8_HasPNI;
	if(u8_HasPNI == 0 && argc >= 6)
	{
		printf("[%s]: search cis in DRAM fail\n", __func__);
		goto Error;
	}
	else
	{
		if(info.nandtype == NAND){
			pPartInfo = pNandDrv->pPartInfo;
			u16_PartType = UNFD_PART_UBOOT;
		}
		else if(info.nandtype == SPINAND){
			pPartInfo = &gtSpiNandDrv.tPartInfo;
			u16_PartType = USFD_PART_UBOOT;
		}
		pRecord = pPartInfo->records;

		while (pRecord - pPartInfo->records < pPartInfo->u16_PartCnt)
		{
			if (pRecord->u16_PartType == u16_PartType)
				break;

			pRecord++;
		}
		if(pRecord - pPartInfo->records == pPartInfo->u16_PartCnt)
		{
			printf("MBOOT partition not found\n");
			goto Error;
		}

		if(bootnum == 1)
			u32_block_ubootbak = pRecord->u16_StartBlk+pRecord->u16_BlkCnt+pRecord->u16_BackupBlkCnt;
		else if(bootnum == 2)
			u32_block_ubootbak = pRecord->u16_StartBlk+2+(pRecord->u16_BlkCnt+pRecord->u16_BackupBlkCnt-2)/2;
	}

	if(argc == 6)
	{
		u32_bl_PBA1 = u32_block_PBA;

		u32_buf = strtoul(argv[4], NULL, 16);
		u32_size = SBOOT_SIZE;

		ret = search_tftp_buffer(&u32_buf);
		if(ret)
			printf("[%s]: find bootloader tftp buffer, add=0x%lx\n", __func__, u32_buf);
		else
			printf("[%s]: find NO bootloader tftp buffer, using original addr=0x%lx\n", __func__, u32_buf);


		printf("write bootloader @ block %lX\n", u32_bl_PBA1);
		//Error Code is updated in nand_write_bootloader function
		if(info.nandtype == NAND)
			u32_Err = nand_write_bootloader(u32_bl_PBA1<< pNandDrv->u8_BlkPageCntBits, (U8*) u32_buf, u32_size, 0x00);
		else if(info.nandtype == SPINAND)
			u32_Err = MDrv_SPINAND_write_bootloader(u32_bl_PBA1*gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt, (U8*) u32_buf, u32_size, 0x00);
		if(u32_Err != 0){
			printf("[%s]: error: program bootloader @ block %lX fail\n", __func__, u32_bl_PBA1);
			goto Error;
		}

		u32_block_PBA = u32_bl_PBA1;

		if(info.nandtype == NAND)
			u32_tmp = (u32_size + (1<<(pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits-pNandDrv->u8_CellType))-1)
				>> (pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits - pNandDrv->u8_CellType);
		else if(info.nandtype == SPINAND)
			u32_tmp = (u32_size + gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt - 1)
				/(gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt);
		while(u32_tmp > 0)
		{
			if(info.nandtype == NAND){
				if(drvNAND_IsGoodBlk(u32_block_PBA)){
					u32_tmp --;
				}
			}
			else if(info.nandtype == SPINAND){
				if(MDrv_SPINAND_IsGoodBlk(u32_block_PBA)){
					u32_tmp --;
				}
			}
			u32_block_PBA ++;
		}

		if(bootnum == 1)
			u32_bl_PBA2 = u32_bl_PBA1;
		else if(bootnum == 2){
			u32_bl_PBA2 = u32_block_PBA;

			u32_buf = strtoul(argv[4], NULL, 16);
			u32_size = SBOOT_SIZE;

			ret = search_tftp_buffer(&u32_buf);
			if(ret)
				printf("[%s]: find bootloader tftp buffer, add=0x%lx\n", __func__, u32_buf);
			else
				printf("[%s]: find NO bootloader tftp buffer, using original addr=0x%lx\n", __func__, u32_buf);


			printf("write bootloader backup @ block %lX\n", u32_bl_PBA2);
			//Error Code is updated in nand_write_bootloader function
			if(info.nandtype == NAND)
				u32_Err = nand_write_bootloader(u32_bl_PBA2<< pNandDrv->u8_BlkPageCntBits, (U8*) u32_buf, u32_size, 0x00);
			else if(info.nandtype == SPINAND)
				u32_Err = MDrv_SPINAND_write_bootloader(u32_bl_PBA2*gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt, (U8*) u32_buf, u32_size, 0x00);
			if(u32_Err != 0){
				printf("[%s]: error: program bootloader backup @ block %lX fail\n", __func__, u32_bl_PBA2);
				goto Error;
			}

			u32_block_PBA = u32_bl_PBA2;

			if(info.nandtype == NAND)
				u32_tmp = (u32_size + (1<<(pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits-pNandDrv->u8_CellType))-1)
					>> (pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits - pNandDrv->u8_CellType);
			else if(info.nandtype == SPINAND)
				u32_tmp = (u32_size + gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt - 1)
					/(gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt);
			while(u32_tmp > 0)
			{
				if(info.nandtype == NAND){
					if(drvNAND_IsGoodBlk(u32_block_PBA)){
						u32_tmp --;
					}
				}
				else if(info.nandtype == SPINAND){
					if(MDrv_SPINAND_IsGoodBlk(u32_block_PBA)){
						u32_tmp --;
					}
				}
				u32_block_PBA ++;
			}
		}

		u32_uboot_PBA0 = u32_block_PBA;

		u32_buf = strtoul(argv[4], NULL, 16);
		ret = search_tftp_buffer(&u32_buf);
		u32_buf += SBOOT_SIZE;
		if(ret)
			printf("[%s]: find uboot tftp buffer, add=0x%lx\n", __func__, u32_buf);
		else
			printf("[%s]: find NO uboot tftp buffer, using original addr=0x%lx\n", __func__, u32_buf);

		u32_size = strtoul(argv[5], NULL, 16) - SBOOT_SIZE;

		for(k=u32_block_PBA ; k<u32_block_ubootbak ; k++)
		{
			if(info.nandtype == NAND && drvNAND_IsGoodBlk(k) == 1)
			{
				u32_Err = NC_EraseBlk(k<<pNandDrv->u8_BlkPageCntBits);
				if(u32_Err != UNFD_ST_SUCCESS)
				{
					drvNAND_MarkBadBlk(k);
				}
			}
			else if(info.nandtype == SPINAND && MDrv_SPINAND_IsGoodBlk(k) == 1)
			{
				u32_Err = MDrv_SPINAND_BLOCK_ERASE(k*gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt);
				if(u32_Err != ERR_SPINAND_SUCCESS)
				{
					//mark bad
				}
			}
		}

		printf("write uboot @ block %lX\n", u32_uboot_PBA0);
		if(info.nandtype == NAND)
			u32_Err = nand_write_bootloader(u32_uboot_PBA0 << pNandDrv->u8_BlkPageCntBits, (U8*) u32_buf, u32_size, 0xFF);
		else if(info.nandtype == SPINAND)
			u32_Err = MDrv_SPINAND_write_bootloader(u32_uboot_PBA0*gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt, (U8*) u32_buf, u32_size, 0xFF);
		if(u32_Err != 0){
			printf("error: program uboot @ block %lX fail\n", u32_uboot_PBA0);
			goto Error;
		}

		u32_block_PBA = u32_uboot_PBA0;

		//calcuate write contain length

		if(info.nandtype == NAND)
			u32_tmp = (u32_size + (1<<(pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits-pNandDrv->u8_CellType))-1)
				>> (pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits - pNandDrv->u8_CellType);
		else if(info.nandtype == SPINAND)
			u32_tmp = (u32_size + gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt - 1)
				/(gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt);

		while(u32_tmp > 0)
		{
			if(info.nandtype == NAND){
				if(drvNAND_IsGoodBlk(u32_block_PBA)){
					u32_tmp --;
				}

			}
			else if(info.nandtype == SPINAND){
				if(MDrv_SPINAND_IsGoodBlk(u32_block_PBA)){
					u32_tmp --;
				}
			}
			u32_block_PBA ++;
		}

		if(u32_block_PBA-1 >= u32_block_ubootbak)
		{
			printf("not enough block for uboot\n");
			goto Error;
		}

		if(bootnum == 1)
			u32_uboot_PBA1 = 0;
		else if(bootnum == 2){
			u32_block_PBA = u32_block_ubootbak;

			u32_uboot_PBA1 = u32_block_PBA;

			u32_buf = strtoul(argv[4], NULL, 16);
			ret = search_tftp_buffer(&u32_buf);
			u32_buf += SBOOT_SIZE;
			if(ret)
				printf("[%s]: find uboot tftp buffer, add=0x%lx\n", __func__, u32_buf);
			else
				printf("[%s]: find NO uboot tftp buffer, using original addr=0x%lx\n", __func__, u32_buf);

			u32_size = strtoul(argv[5], NULL, 16) - SBOOT_SIZE;

			printf("write uboot backup @ block %lX\n", u32_uboot_PBA1);
			//Error Code is updated in nand_write_bootloader function
			if(info.nandtype == NAND)
				u32_Err = nand_write_bootloader(u32_uboot_PBA1 << pNandDrv->u8_BlkPageCntBits, (U8*) u32_buf, u32_size, 0xFF);
			else if(info.nandtype == SPINAND)
				u32_Err = MDrv_SPINAND_write_bootloader(u32_uboot_PBA1*gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt, (U8*) u32_buf, u32_size, 0xFF);
			if(u32_Err != 0){
				printf("[%s]: error: program uboot backup @ block %lX fail\n", __func__, u32_uboot_PBA1);
				goto Error;
			}
		}

		if(info.nandtype == NAND){
			pNandInfo->u8_BL0PBA = (U8)(u32_bl_PBA1 & 0xFF);
			pNandInfo->u8_BL1PBA = (U8)(u32_bl_PBA2 & 0xFF);
			pNandInfo->u8_UBOOTPBA = (U8)(u32_uboot_PBA1 & 0xFF);
			printf("%X, %X, %X\n", pNandInfo->u8_BL0PBA, pNandInfo->u8_BL1PBA, pNandInfo->u8_UBOOTPBA);
		}
		else if(info.nandtype == SPINAND){
			pSpiNandInfo->u8_BL0PBA = (U8)(u32_bl_PBA1 & 0xFF);
			pSpiNandInfo->u8_BL1PBA = (U8)(u32_bl_PBA2 & 0xFF);
			pSpiNandInfo->u8_UBOOTPBA = (U8)(u32_uboot_PBA1 & 0xFF);
			printf("%X, %X, %X\n", pSpiNandInfo->u8_BL0PBA, pSpiNandInfo->u8_BL1PBA, pSpiNandInfo->u8_UBOOTPBA);
		}
	}

	if(argc == 8)
	{
		u32_bl_PBA1 = u32_block_PBA;

		u32_buf = strtoul(argv[4], NULL, 16);
		u32_size = strtoul(argv[5], NULL, 16);

		ret = search_tftp_buffer(&u32_buf);
		if(ret)
			printf("[%s]: find bootloader tftp buffer, add=0x%lx\n", __func__, u32_buf);
		else
			printf("[%s]: find NO bootloader tftp buffer, using original addr=0x%lx\n", __func__, u32_buf);



		printf("write bootloader @ block %lX\n", u32_bl_PBA1);
		if(info.nandtype == NAND)
			u32_Err = nand_write_bootloader(u32_bl_PBA1<< pNandDrv->u8_BlkPageCntBits, (U8*) u32_buf, u32_size, 0x00);
		else if(info.nandtype == SPINAND)
			u32_Err = MDrv_SPINAND_write_bootloader(u32_bl_PBA1*gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt, (U8*) u32_buf, u32_size, 0x00);
		if(u32_Err != 0){
			printf("[%s]: error: program bootloader @ block %lX fail\n", __func__, u32_bl_PBA1);
			goto Error;
		}

		u32_block_PBA = u32_bl_PBA1;

		if(info.nandtype == NAND)
			u32_tmp = (u32_size + (1<<(pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits-pNandDrv->u8_CellType))-1)
				>> (pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits - pNandDrv->u8_CellType);
		else if(info.nandtype == SPINAND)
			u32_tmp = (u32_size + gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt - 1)
				/(gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt);
		while(u32_tmp > 0)
		{
			if(info.nandtype == NAND){
				if(drvNAND_IsGoodBlk(u32_block_PBA))
					u32_tmp --;
			}
			else if(info.nandtype == SPINAND){
				if(MDrv_SPINAND_IsGoodBlk(u32_block_PBA))
					u32_tmp --;
			}
			u32_block_PBA ++;
		}

		if(bootnum == 1)
			u32_bl_PBA2 = u32_bl_PBA1;
		else{

			u32_bl_PBA2 = u32_block_PBA;

			u32_buf = strtoul(argv[4], NULL, 16);
			u32_size = strtoul(argv[5], NULL, 16);

			ret = search_tftp_buffer(&u32_buf);
			if(ret)
				printf("[%s]: find bootloader tftp buffer, add=0x%lx\n", __func__, u32_buf);
			else
				printf("[%s]: find NO bootloader tftp buffer, using original addr=0x%lx\n", __func__, u32_buf);

			printf("write bootloader backup @ block %lX\n", u32_bl_PBA2);

			if(info.nandtype == NAND)
				u32_Err = nand_write_bootloader(u32_bl_PBA2<< pNandDrv->u8_BlkPageCntBits, (U8*) u32_buf, u32_size, 0x00);
			else if(info.nandtype == SPINAND)
				u32_Err = MDrv_SPINAND_write_bootloader(u32_bl_PBA2*gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt, (U8*) u32_buf, u32_size, 0x00);
			if(u32_Err != 0){
				printf("error: program bootloader backup @ block %lX fail\n", u32_bl_PBA2);
				goto Error;
			}

			u32_block_PBA = u32_bl_PBA2;
			if(info.nandtype == NAND)
				u32_tmp = (u32_size + (1<<(pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits-pNandDrv->u8_CellType))-1)
				>> (pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits - pNandDrv->u8_CellType);
			else if(info.nandtype == SPINAND)
				u32_tmp = (u32_size + gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt - 1)
					/(gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt);

			while(u32_tmp > 0)
			{
				if(info.nandtype == NAND){
					if(drvNAND_IsGoodBlk(u32_block_PBA)){
						u32_tmp --;
					}
				}
				else if(info.nandtype == SPINAND){
					if(MDrv_SPINAND_IsGoodBlk(u32_block_PBA)){
						u32_tmp --;
					}
				}
				u32_block_PBA ++;
			}
		}

		u32_uboot_PBA0 = u32_block_PBA;

		u32_buf = strtoul(argv[6], NULL, 16);
		u32_size = strtoul(argv[7], NULL, 16);

		ret = search_tftp_buffer(&u32_buf);
		if(ret)
			printf("[%s]: find uboot tftp buffer, add=0x%lx\n", __func__, u32_buf);
		else
			printf("[%s]: find NO uboot tftp buffer, using original addr=0x%lx\n", __func__, u32_buf);


		for(k=u32_block_PBA ; k<u32_block_ubootbak ; k++)
		{
			if(info.nandtype == NAND && drvNAND_IsGoodBlk(k) == 1)
			{
				u32_Err = NC_EraseBlk(k<<pNandDrv->u8_BlkPageCntBits);
				if(u32_Err != UNFD_ST_SUCCESS)
				{
					drvNAND_MarkBadBlk(k);
				}
			}
			else if(info.nandtype == SPINAND && MDrv_SPINAND_IsGoodBlk(k) == 1)
			{
				u32_Err = MDrv_SPINAND_BLOCK_ERASE(k*gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt);
				if(u32_Err != ERR_SPINAND_SUCCESS)
				{
					//mark bad
				}
			}
		}

		printf("write uboot @ block %lX\n", u32_uboot_PBA0);

		if(info.nandtype == NAND)
			u32_Err = nand_write_bootloader(u32_uboot_PBA0 << pNandDrv->u8_BlkPageCntBits, (U8*) u32_buf, u32_size, 0xFF);
		else if(info.nandtype == SPINAND)
			u32_Err = MDrv_SPINAND_write_bootloader(u32_uboot_PBA0*gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt, (U8*) u32_buf, u32_size, 0xFF);
		if(u32_Err != 0){
			printf("error: program uboot @ block %lX fail\n", u32_uboot_PBA0);
			goto Error;
		}

		u32_block_PBA = u32_uboot_PBA0;

		//calcuate write contain length

		if(info.nandtype == NAND)
			u32_tmp = (u32_size + (1<<(pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits-pNandDrv->u8_CellType))-1)
				>> (pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits - pNandDrv->u8_CellType);
		else if(info.nandtype == SPINAND)
			u32_tmp = (u32_size + gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt - 1)
				/(gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt*gtSpiNandDrv.tSpinandInfo.u16_PageByteCnt);

		while(u32_tmp > 0)
		{
			if(info.nandtype == NAND){
				if(drvNAND_IsGoodBlk(u32_block_PBA)){
					u32_tmp --;
				}
			}
			else if(info.nandtype == SPINAND){
				if(MDrv_SPINAND_IsGoodBlk(u32_block_PBA)){
					u32_tmp --;
				}
			}
			u32_block_PBA ++;
		}

		if(u32_block_PBA-1 >= u32_block_ubootbak){
			printf("not enough block for uboot\n");
			goto Error;
		}

		if(bootnum == 1)
			u32_uboot_PBA1 = 0;
		else if(bootnum == 2){
			u32_block_PBA = u32_block_ubootbak;
			u32_uboot_PBA1 = u32_block_PBA;

			u32_buf = strtoul(argv[6], NULL, 16);
			u32_size = strtoul(argv[7], NULL, 16);

			ret = search_tftp_buffer(&u32_buf);
			if(ret)
				printf("[%s]: find uboot tftp buffer, add=0x%lx\n", __func__, u32_buf);
			else
				printf("[%s]: find NO uboot tftp buffer, using original addr=0x%lx\n", __func__, u32_buf);


			printf("write uboot backup @ block %lX\n", u32_uboot_PBA1);

			if(info.nandtype == NAND)
				u32_Err = nand_write_bootloader(u32_uboot_PBA1 << pNandDrv->u8_BlkPageCntBits, (U8*) u32_buf, u32_size, 0xFF);
			else if(info.nandtype == SPINAND)
				u32_Err = MDrv_SPINAND_write_bootloader(u32_uboot_PBA1*gtSpiNandDrv.tSpinandInfo.u16_BlkPageCnt, (U8*) u32_buf, u32_size, 0xFF);
			if(u32_Err != 0){
				printf("[%s]: error: program uboot backup @ block %lX fail\n", __func__, u32_uboot_PBA1);
				goto Error;
			}
		}

		if(info.nandtype == NAND){
			pNandInfo->u8_BL0PBA = (U8)(u32_bl_PBA1 & 0xFF);
			pNandInfo->u8_BL1PBA = (U8)(u32_bl_PBA2 & 0xFF);
			pNandInfo->u8_UBOOTPBA = (U8)(u32_uboot_PBA1 & 0xFF);
			printf("%X, %X, %X\n", pNandInfo->u8_BL0PBA, pNandInfo->u8_BL1PBA, pNandInfo->u8_UBOOTPBA);
		}
		else if(info.nandtype == SPINAND){
			pSpiNandInfo->u8_BL0PBA = (U8)(u32_bl_PBA1 & 0xFF);
			pSpiNandInfo->u8_BL1PBA = (U8)(u32_bl_PBA2 & 0xFF);
			pSpiNandInfo->u8_UBOOTPBA = (U8)(u32_uboot_PBA1 & 0xFF);
			printf("%X, %X, %X\n", pSpiNandInfo->u8_BL0PBA, pSpiNandInfo->u8_BL1PBA, pSpiNandInfo->u8_UBOOTPBA);
		}
	}

	if(info.nandtype == NAND)
		u32_Err = drvNAND_WriteCIS_for_ROM(pNandInfo);
	else if(info.nandtype == SPINAND)
		u32_Err = MDrv_SPINAND_WriteCIS_for_ROM(pSpiNandInfoTag);
	if(u32_Err != 0){
		printf("Write CIS for ROM fail\n");
		goto Error;
	}

	//Reinit NAND Flash
	//nand_init();

	/////////////////////////////////////////
	if(info.nandtype == NAND){
		pNandDrv->u8_BL0PBA = (U8)(u32_bl_PBA1 & 0xFF);
		pNandDrv->u8_BL1PBA = (U8)(u32_bl_PBA2 & 0xFF);
		pNandDrv->u8_UBOOTPBA = (U8)(u32_uboot_PBA1 & 0xFF);
	}
	else if(info.nandtype == SPINAND){
		gtSpiNandDrv.tSpinandInfo.u8_BL0PBA = (U8)(u32_bl_PBA1 & 0xFF);
		gtSpiNandDrv.tSpinandInfo.u8_BL1PBA = (U8)(u32_bl_PBA2 & 0xFF);
		gtSpiNandDrv.tSpinandInfo.u8_UBOOTPBA = (U8)(u32_uboot_PBA1 & 0xFF);
	}
	/////////////////////////////////////////


	/*
	 *release tftp buffer;
	 */
	/////////////////////////////////////////
	u32_buf = strtoul(argv[1], NULL, 16);
	release_tftp_buffer(u32_buf);
	u32_PPM_buf = strtoul(argv[2], NULL, 16);
	release_tftp_buffer(u32_PPM_buf);
	if(argc >= 4){
		u32_PNI_buf = strtoul(argv[3], NULL, 16);
		release_tftp_buffer(u32_PNI_buf);
	}
	if(argc == 6){
		u32_buf = strtoul(argv[4], NULL, 16);
		release_tftp_buffer(u32_buf);
	}
	if(argc == 8){
		u32_buf = strtoul(argv[4], NULL, 16);
		release_tftp_buffer(u32_buf);
		u32_buf = strtoul(argv[6], NULL, 16);
		release_tftp_buffer(u32_buf);
	}
	/////////////////////////////////////////

	if(info.nandtype == NAND){
		free(pNandDrv_tmp);
		free(pNandInfo);
	}
	else if(info.nandtype == SPINAND){
		free(pSpiNandInfoTag);
	}
	return 0;

Error:
	#if (ENABLE_MODULE_NAND_FLASH == 1)
	memcpy(pNandDrv, pNandDrv_tmp, sizeof(NAND_DRIVER));
	NC_Init();
	NC_Config();
	#endif
	return u32_Err ? -1 : 0;
}
U_BOOT_CMD(ncisbl, 10, 1, do_nandupdateCIS_BL_UBOOT,
	"write cis, bootloader and uboot or only CIS for ROM boot",
	"ncisbl CISAddr PPMAddr [PNIAddr] [bootloaderAddr] [bootloaderSize] [ubootAddr] [ubootSize]\n"
	"  -write CIS, PPM, PNI, bootloader, uboot from those memory address, CISaddr, PPMaddr, bootloaderAddr\n"
	"	and ubootAddr respectively with 'bootloaderSize' and 'ubootSize' bytes \n"
	"	assigned for bootloader and uboot programming."
);

int do_nand_update_CIS_HASH(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
	unsigned long u32_cis_buf = 0;
	unsigned long u32_ppm_buf = 0;
	unsigned long u32_hash_buf[3] = {0};
	unsigned long u32_block_hash[3] = {0};
	unsigned long u32_hash_size[3] = {0};
	unsigned int u32_block_PBA = 10;
	unsigned long u32_pni_buf = 0;
	unsigned int u32_Err = 0;
	unsigned int u32_tmp;
	unsigned int u32_BootSize;

	PARTITION_INFO_t *pPartInfo;
	PARTITION_RECORD_t *pRecord;
	unsigned int u32_block_ubootbak;
	NAND_DRIVER *pNandDrv = (NAND_DRIVER*)drvNAND_get_DrvContext_address();
	NAND_DRIVER *pNandDrv_tmp = (NAND_DRIVER*)malloc(sizeof(NAND_DRIVER));
	NAND_FLASH_INFO_t *pNandInfo = (NAND_FLASH_INFO_t* )malloc(512);
	PARTITION_INFO_t *pPartInfo_tmp = (PARTITION_INFO_t *)malloc(1024);
	PARTITION_RECORD_t *pRecord_tmp;

	int i, k;

	if( argc < 8 )
	{
		printf("Too few arguements\n");
		free(pPartInfo_tmp);
		free(pNandDrv_tmp);
		free(pNandInfo);
		return cmd_usage(cmdtp);
	}

	memcpy(pNandDrv_tmp, pNandDrv, sizeof(NAND_DRIVER));
	memset(pPartInfo_tmp, 0, 0x200);

	if(pNandDrv->u8_HasPNI == 1)
	{
		memcpy( pPartInfo_tmp, pNandDrv->pPartInfo, sizeof(*pNandDrv->pPartInfo) + pNandDrv->pPartInfo->u16_PartCnt * pNandDrv->pPartInfo->u16_UnitByteCnt);
	}

	u32_cis_buf = strtoul(argv[1], NULL, 16);
	u32_Err = search_tftp_buffer(&u32_cis_buf);
	if(u32_Err)
		printf("[%s]: find cis tftp buffer, addr=0x%lx\n", __func__, u32_cis_buf);
	else
		printf("[%s]: find NO cis tftp buffer, usingu32_cis_buf original addr=0x%lx\n", __func__, u32_cis_buf);
	u32_ppm_buf = strtoul(argv[2], NULL, 16);
	u32_Err = search_tftp_buffer(&u32_ppm_buf);
	if(u32_Err)
		printf("[%s]: find ppm tftp buffer, addr=0x%lx\n", __func__, u32_ppm_buf);
	else
		printf("[%s]: find NO ppm tftp buffer, using original addr=0x%lx\n", __func__, u32_ppm_buf);
	u32_pni_buf = strtoul(argv[3], NULL, 16);
	u32_Err = search_tftp_buffer(&u32_pni_buf);
	if(u32_Err)
		printf("[%s]: find pni tftp buffer, addr=0x%lx\n", __func__, u32_pni_buf);
	else
		printf("[%s]: find NO pni tftp buffer, using original addr=0x%lx\n", __func__, u32_pni_buf);
	u32_hash_buf[0] = strtoul(argv[4], NULL, 16);
	u32_Err = search_tftp_buffer(&u32_hash_buf[0]);
	if(u32_Err)
		printf("[%s]: find hash0 tftp buffer, addr=0x%lx\n", __func__, u32_hash_buf[0]);
	else
		printf("[%s]: find NO hash0 tftp buffer, using original addr=0x%lx\n", __func__, u32_hash_buf[0]);
	u32_hash_buf[1] = strtoul(argv[6], NULL, 16);
	u32_Err = search_tftp_buffer(&u32_hash_buf[1]);
	if(u32_Err)
		printf("[%s]: find hash1 tftp buffer, addr=0x%lx\n", __func__, u32_hash_buf[1]);
	else
		printf("[%s]: find NO hash1 tftp buffer, using original addr=0x%lx\n", __func__, u32_hash_buf[1]);
	u32_hash_buf[2] = u32_hash_buf[1] + SBOOT_SIZE;
	u32_hash_size[0] = strtoul(argv[5], NULL, 16);
	u32_hash_size[1] = SBOOT_SIZE;
	u32_hash_size[2] = strtoul(argv[7], NULL, 16) - SBOOT_SIZE;

	if(argc > 8)
		u32_BootSize = strtoul(argv[8], NULL, 16);
	else
		u32_BootSize = 0;

	//printf("nni addr=%Xh\n", u32_cis_buf);
	//printf("ppm addr=%Xh\n", u32_ppm_buf);
	//printf("pni addr=%Xh\n", u32_pni_buf);
	//printf("hash0 addr=%Xh, size=%Xh\n", u32_hash_buf[0], u32_hash_size[0]);
	//printf("hash1 addr=%Xh, size=%Xh\n", u32_hash_buf[1], u32_hash_size[1]);
	//printf("hash2 addr=%Xh, size=%Xh\n", u32_hash_buf[2], u32_hash_size[2]);

	u32_Err = search_cis_in_DRAM((U8*)u32_cis_buf, (U8*)u32_ppm_buf, (U8*) u32_pni_buf, pNandInfo);

	if(u32_Err != 0 || pNandDrv->u8_HasPNI == 0)
	{
		printf("search cis in DRAM fail\n");
		goto HASH_ERR;
	}

	//check partition information, if current information doesn't match with that in nand. issue nand erase.force or nand erase.chip
	pPartInfo = pNandDrv->pPartInfo;
	pRecord = pPartInfo->records;
	pRecord_tmp = pPartInfo_tmp->records;
	if(pPartInfo->u16_PartCnt != pPartInfo_tmp->u16_PartCnt)
	{
		printf("Current Partition Count %d doesn't not match with that in nand %d\n", pPartInfo->u16_PartCnt, pPartInfo_tmp->u16_PartCnt);
	}
	else
	{
		while (pRecord - pPartInfo->records < pPartInfo->u16_PartCnt)
		{
			if (	pRecord->u16_PartType != pRecord_tmp->u16_PartType ||
				pRecord->u16_StartBlk != pRecord_tmp->u16_StartBlk ||
				pRecord->u16_BlkCnt != pRecord_tmp->u16_BlkCnt ||
				pRecord->u16_BackupBlkCnt != pRecord_tmp->u16_BackupBlkCnt
				)
			{
				printf("Current Partition information doesn't not match with that in nand\n");

				printf("Previous Partition info:\n");
				printf("StartBlk: 0x%X\n", pRecord_tmp->u16_StartBlk);
				printf("BlkCnt: 0x%X\n", pRecord_tmp->u16_BlkCnt);
				printf("PartType: 0x%X\n", pRecord_tmp->u16_PartType);
				printf("BackupBlkCnt: 0x%X\n", pRecord_tmp->u16_BackupBlkCnt);


				printf("Current Partition info:\n");
				printf("StartBlk: 0x%X\n", pRecord->u16_StartBlk);
				printf("BlkCnt: 0x%X\n", pRecord->u16_BlkCnt);
				printf("PartType: 0x%X\n", pRecord->u16_PartType);
				printf("BackupBlkCnt: 0x%X\n", pRecord->u16_BackupBlkCnt);

				break;
			}
			pRecord++;
			pRecord_tmp ++;
		}
	}

	pPartInfo = pNandDrv->pPartInfo;
	pRecord = pPartInfo->records;
	while (pRecord - pPartInfo->records < pPartInfo->u16_PartCnt)
	{
		if (pRecord->u16_PartType == UNFD_PART_UBOOT)
			break;

		pRecord++;
	}
	if(pRecord - pPartInfo->records == pPartInfo->u16_PartCnt)
	{
		printf("MBOOT partition not found\n");
		goto HASH_ERR;
	}
	u32_block_ubootbak = pRecord->u16_StartBlk+4+(pRecord->u16_BlkCnt+pRecord->u16_BackupBlkCnt-4)/2;

	pNandInfo->u32_BootSize = u32_BootSize;

	for(i=0; i<3; i++)
	{
		//Search good block for hash[0-2]
		while(drvNAND_IsGoodBlk(u32_block_PBA) == 0)
			u32_block_PBA++;
		u32_block_hash[i] = u32_block_PBA;

		if(i == 2) // hash2, erase all blocks in case power loss
		{
			for(k=u32_block_PBA ; k<u32_block_ubootbak ; k++)
			{
				if(drvNAND_IsGoodBlk(k) == 1)
				{
					u32_Err = NC_EraseBlk(k<<pNandDrv->u8_BlkPageCntBits);
					if(u32_Err != UNFD_ST_SUCCESS)
					{
						drvNAND_MarkBadBlk(k);
					}
				}
			}
		}

		//Write hash[0-2]
		printf("write hash%d @ block %lX\n", i, u32_block_hash[i]);
		nand_write_bootloader(u32_block_hash[i]<< pNandDrv->u8_BlkPageCntBits, (U8*)u32_hash_buf[i], u32_hash_size[i], (0x80|i));
		pNandInfo->u8_HashPBA[i][0] = u32_block_hash[i];

		u32_tmp = (u32_hash_size[i]+(1<<(pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits-pNandDrv->u8_CellType))-1)
			>> (pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits-pNandDrv->u8_CellType);
		while(u32_tmp > 0)
		{
			if(drvNAND_IsGoodBlk(u32_block_PBA))
			{
				u32_tmp --;
			}
				u32_block_PBA ++;
		}

		if(i == 2) // hash2 backup
		{
			if(u32_block_PBA-1 >= u32_block_ubootbak)
			{
				printf("not enough block for hash2\n");
				goto HASH_ERR;
			}
			u32_block_PBA = u32_block_ubootbak;
		}
		//Search good block for hash[0-2] backup
		while(drvNAND_IsGoodBlk(u32_block_PBA) == 0)
			u32_block_PBA ++;
		u32_block_hash[i] = u32_block_PBA;

		//Write hash[0-2] backup
		printf("write hash%d backup @ block %lX\n", i, u32_block_hash[i]);
		nand_write_bootloader(u32_block_hash[i]<< pNandDrv->u8_BlkPageCntBits, (U8*)u32_hash_buf[i], u32_hash_size[i], (0x80|i));
		pNandInfo->u8_HashPBA[i][1] = u32_block_hash[i];

		u32_tmp = (u32_hash_size[i]+(1<<(pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits-pNandDrv->u8_CellType))-1)
			>> (pNandDrv->u8_BlkPageCntBits + pNandDrv->u8_PageByteCntBits-pNandDrv->u8_CellType);
		while(u32_tmp > 0)
		{
			if(drvNAND_IsGoodBlk(u32_block_PBA))
			{
				u32_tmp --;
			}
          	  u32_block_PBA ++;
		}
	}

	pNandInfo->u8_BL0PBA = 0;
	pNandInfo->u8_BL1PBA = 0;
	pNandInfo->u8_UBOOTPBA = 0;

	u32_Err = drvNAND_WriteCIS_for_ROM(pNandInfo);

	if(u32_Err != 0)
	{
		printf("Update CIS for ROM fail\n");
		goto HASH_ERR;
	}

	pNandDrv->u8_BL0PBA = 0;
	pNandDrv->u8_BL1PBA = 0;
	pNandDrv->u8_UBOOTPBA = 0;
	for(i = 0; i < 3; i++){
		pNandDrv->u8_HashPBA[i][0] = pNandInfo->u8_HashPBA[i][0];
		pNandDrv->u8_HashPBA[i][1] = pNandInfo->u8_HashPBA[i][1];
	}

	printf("[%s]: Hash[0][0]=0x%x Hash[0][1]=0x%x\n", __func__, pNandDrv->u8_HashPBA[0][0], pNandDrv->u8_HashPBA[0][1]);
	printf("[%s]: Hash[1][0]=0x%x Hash[1][1]=0x%x\n", __func__, pNandDrv->u8_HashPBA[1][0], pNandDrv->u8_HashPBA[1][1]);
	printf("[%s]: Hash[2][0]=0x%x Hash[2][1]=0x%x\n", __func__, pNandDrv->u8_HashPBA[2][0], pNandDrv->u8_HashPBA[2][1]);

	free(pPartInfo_tmp);
	free(pNandDrv_tmp);
	free(pNandInfo);

	u32_cis_buf = strtoul(argv[1], NULL, 16);
	release_tftp_buffer(u32_cis_buf);
	u32_ppm_buf = strtoul(argv[2], NULL, 16);
	release_tftp_buffer(u32_ppm_buf);
	u32_pni_buf = strtoul(argv[3], NULL, 16);
	release_tftp_buffer(u32_pni_buf);
	u32_hash_buf[0] = strtoul(argv[4], NULL, 16);
	release_tftp_buffer(u32_hash_buf[0]);
	u32_hash_buf[1] = strtoul(argv[6], NULL, 16);
	release_tftp_buffer(u32_hash_buf[1]);

	return 0;

HASH_ERR:
	memcpy(pNandDrv, pNandDrv_tmp, sizeof(NAND_DRIVER));
	NC_Init();
	NC_Config();
	free(pPartInfo_tmp);
	free(pNandDrv_tmp);
	free(pNandInfo);
	return u32_Err;
}

U_BOOT_CMD(ncishash, 9, 1, do_nand_update_CIS_HASH,
	"write cis and hash[0-2] for Secure ROM boot",
	"<CISAddr> <PPMAddr> <PNIAddr> <Hash0Addr> <Hash0Size> <MbootAddr> <MbootSize> [<Hash1Size>]\n"
	"CISAddr   DRAM address of CIS\n"
	"PPMAddr   DRAM address of PPM\n"
	"PNIAddr   DRAM address of PNI\n"
	"Hash0Addr DRAM address of hash0\n"
	"Hash0Size Total size of the hash0\n"
	"MbootAddr DRAM address of MbootAddr [Hash1 + Bootram + uboot]\n"
	"MbootSize Total size of the MbootSize\n"
	"[Hash1Size Total size of the hash1]\n"
);


#define  CONFIG_STB_BFN 1

#if defined(CONFIG_STB_BFN) && (CONFIG_STB_BFN == 1)
#define HASH_LDR_TagInSpare 0x42000000
#define HASH_APP_TagInSpare 0x42010000

static  U8 cmd_u8MainBuf[8192*4];
static  U8 cmd_u8SpareBuf[640*4];

static int do_nand_update_HASH_STBBFN(U32* u32WritetBlk,U32 u32HashDAddr,S32 size,U32 u32HashTagInSpare)
{
	U32 u32Row_offset = 0;
	U32 u32Ret = UNFD_ST_SUCCESS;
	U32 i =0;
	NAND_DRIVER *pNandDrv = (NAND_DRIVER*)drvNAND_get_DrvContext_address();
	U8 *au8_PageBuf = (U8*)cmd_u8MainBuf;
	U8 *au8_SpareBuf = (U8*)cmd_u8SpareBuf;
	U32 	u32HashDAddr_Tmp = u32HashDAddr;
	int ret = 0;
	U32 u32_write_size = 0;

	printf("[%s]: Search BFN Hash buffer\n", __func__);
	ret = search_tftp_buffer(&u32HashDAddr);
	if(ret)
		printf("[%s]: find BFN Hash buffer, addr=0x%lx\n", __func__, u32HashDAddr);
	else
		printf("[%s]: find NO BFN Hash buffer, using original addr=0x%lx\n", __func__, u32HashDAddr_Tmp);

	memset(au8_PageBuf, '\0', pNandDrv->u16_PageByteCnt);

	//each while loop operates one blk data
	while(size > 0) {
		printf("Hash left %lx\n",size);

		//Find good blk
		while(drvNAND_IsGoodBlk(*u32WritetBlk) == 0){
			printf("blk %ld is bad\n",(*u32WritetBlk));
			u32Ret = NC_EraseBlk((*u32WritetBlk)*(U32)pNandDrv->u16_BlkPageCnt);
			if(u32Ret != UNFD_ST_SUCCESS)
			{
				drvNAND_MarkBadBlk((*u32WritetBlk));
			}
			(*u32WritetBlk)++;
		}

		//Erase before write
		if(NC_EraseBlk((*u32WritetBlk)*(U32)pNandDrv->u16_BlkPageCnt)!=UNFD_ST_SUCCESS)
		{
			printf("erase blk error %ld\n",(*u32WritetBlk));
			drvNAND_MarkBadBlk((*u32WritetBlk));
			(*u32WritetBlk)++;
			continue;
		}

		printf("Start write Hash data @blk%ld\n",(*u32WritetBlk));
        u32_write_size = 0;
		for(u32Row_offset=0;u32Row_offset<(U32)pNandDrv->u16_BlkPageCnt;u32Row_offset++){
			memset(au8_SpareBuf, 0xFF, pNandDrv->u16_SpareByteCnt);
			memcpy(au8_SpareBuf+4, &u32HashTagInSpare, 0x4);
			if (NC_WritePages((*u32WritetBlk)*(U32)pNandDrv->u16_BlkPageCnt+u32Row_offset, (U8 *)((u32HashDAddr+u32Row_offset*(U32)pNandDrv->u16_PageByteCnt)), au8_SpareBuf, 1)!=UNFD_ST_SUCCESS){
				printf("HASH NC_WritePages failed @blk%ld\n",(*u32WritetBlk));
				NC_EraseBlk((*u32WritetBlk));
				drvNAND_MarkBadBlk((*u32WritetBlk));
				(*u32WritetBlk)++;
				continue;
			}
			u32_write_size += pNandDrv->u16_PageByteCnt;
			if(u32_write_size >= size)
				break;
		}

		//write one blk content ok
		printf("Write blk%ld ok\n",(*u32WritetBlk));

		size -= (U32)pNandDrv->u16_BlkPageCnt*(U32)pNandDrv->u16_PageByteCnt;
		u32HashDAddr += (U32)pNandDrv->u16_BlkPageCnt*(U32)pNandDrv->u16_PageByteCnt;
		u32HashTagInSpare++;
		(*u32WritetBlk)++;

	}

	(*u32WritetBlk)--;
	return UNFD_ST_SUCCESS;

	release_tftp_buffer(u32HashDAddr_Tmp);

	return u32Ret;
}

static int do_nand_update_MIUCIS_LDRAPP(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
	int ret;
	unsigned long u32_cis_buf = 0;
	unsigned long u32_ppm_buf = 0;
	U32 u32_hash_buf[3] = {0};
	U32 u32_hash_size[3] = {0};
	U32 u32_block_PBA = 10;
	unsigned long u32_pni_buf = 0;
	unsigned int u32_Err = 0;
	unsigned int u32_CIS_end_blk=0;
	int part_idx;
	loff_t part_size, part_off;

	NAND_DRIVER *pNandDrv = (NAND_DRIVER*)drvNAND_get_DrvContext_address();
	NAND_DRIVER *pNandDrv_tmp = (NAND_DRIVER*)malloc(sizeof(NAND_DRIVER));
	NAND_FLASH_INFO_t *pNandInfo = (NAND_FLASH_INFO_t* )malloc(512);

	PARTITION_INFO_t *pPartInfo_tmp = (PARTITION_INFO_t *)malloc(512);
	unsigned char  u8_SkipCISWrite = 0;//, u8_HasPNI;

	if( argc < 4 ){
		printf("Too few arguements\n");
		free(pPartInfo_tmp);
		free(pNandDrv_tmp);
		free(pNandInfo);
		return cmd_usage(cmdtp);
	}

	u32_hash_buf[0] = strtoul(argv[4], NULL, 16);
	u32_hash_buf[1] = strtoul(argv[5], NULL, 16);
	u32_hash_buf[2] = strtoul(argv[7], NULL, 16);
	u32_hash_size[0] = 64*1024;
	u32_hash_size[1] = strtoul(argv[6], NULL, 16);
	u32_hash_size[2] = strtoul(argv[8], NULL, 16);

	printf("bfn_miu addr=0x%lXh, size=0x%lXh\n", u32_hash_buf[0], u32_hash_size[0]);
	printf("bfn_ldr addr=0x%lXh, size=0x%lXh\n", u32_hash_buf[1], u32_hash_size[1]);
	printf("bfn_app addr=0x%lXh, size=0x%lXh\n", u32_hash_buf[2], u32_hash_size[2]);

	// =================================================
	// program LDR LDR_backup APP APP_backup
	// =================================================
	//PRG_BOOT:

	//step 2, write LDR
	do_nand_update_HASH_STBBFN(&u32_block_PBA,u32_hash_buf[1],(S32)u32_hash_size[1],HASH_LDR_TagInSpare);
	printf("LDR write end blk %ld\n",u32_block_PBA);

	//step 3, write backup LDR
	u32_block_PBA++;
	do_nand_update_HASH_STBBFN(&u32_block_PBA,u32_hash_buf[1],(S32)u32_hash_size[1],HASH_LDR_TagInSpare);
	printf("Backup LDR write end blk %ld\n",u32_block_PBA);

	if(get_part("MBOOT", &part_idx, &part_off, &part_size)){
		printf("Get UBILD partition information Error\n");
		return -1;
	}

	printf("part_idx = 0x%x, part_off = 0x%llx, part_size = 0x%llx\n",part_idx, part_off, part_size);

    u32_block_PBA = part_off/(pNandDrv->u16_BlkPageCnt * pNandDrv->u16_PageByteCnt);

	//step 4, write APP/BIN
	//u32_block_PBA++;
	printf("APP write start blk %ld\n",u32_block_PBA);
	do_nand_update_HASH_STBBFN(&u32_block_PBA,u32_hash_buf[2],(S32)u32_hash_size[2],HASH_APP_TagInSpare);
	printf("APP write end blk %ld\n",u32_block_PBA);

    if(get_part("MBOOTBAK", &part_idx, &part_off, &part_size)){
		printf("Get UBILD partition information Error\n");
		return -1;
	}

	printf("part_idx = 0x%x, part_off = 0x%llx, part_size = 0x%llx\n",part_idx, part_off, part_size);

    u32_block_PBA = part_off/(pNandDrv->u16_BlkPageCnt * pNandDrv->u16_PageByteCnt);

	//step 5, write backup APP/BIN
	//u32_block_PBA++;
	printf("Backup APP write start blk %ld\n",u32_block_PBA);
	do_nand_update_HASH_STBBFN(&u32_block_PBA,u32_hash_buf[2],(S32)u32_hash_size[2],HASH_APP_TagInSpare);
	printf("Backup APP write end blk %ld\n",u32_block_PBA);

	// =================================================
	// program CIS x2
	// =================================================
	//PRG_CIS:
	if(u8_SkipCISWrite == 0){
		int ret;

		pNandInfo->u8_BL0PBA = 0;
		pNandInfo->u8_BL1PBA = 0;
		pNandInfo->u8_UBOOTPBA = 0;

		u32_cis_buf = strtoul(argv[1], NULL, 16);
		u32_ppm_buf = strtoul(argv[2], NULL, 16);
		u32_pni_buf = strtoul(argv[3], NULL, 16);

		printf("[%s]: Search nni & ppm & pni buffer\n", __func__);
		search_tftp_buffer(&u32_cis_buf);
		search_tftp_buffer(&u32_ppm_buf);
		search_tftp_buffer(&u32_pni_buf);

		u32_Err = search_cis_in_DRAM((U8*)u32_cis_buf, (U8*)u32_ppm_buf, (U8*) u32_pni_buf, pNandInfo);

		//state for program CIS
		u32_Err = drvNAND_WriteCIS_for_ROM(pNandInfo);
		if(u32_Err != 0){
			printf("Update CIS for ROM fail\n");
			goto HASH_ERR;
		}

		u32_cis_buf = strtoul(argv[1], NULL, 16);
		u32_ppm_buf = strtoul(argv[2], NULL, 16);
		u32_pni_buf = strtoul(argv[3], NULL, 16);

		release_tftp_buffer(u32_cis_buf);
		release_tftp_buffer(u32_ppm_buf);
		release_tftp_buffer(u32_pni_buf);

        #if CONFIG_BFN_VERSION < 5
		search_tftp_buffer(&u32_hash_buf[0]);
		drvNAND_NCISLDRAPP_WriteCISMIUParam(0, u32_hash_buf[0]);
		drvNAND_NCISLDRAPP_WriteCISMIUParam(1, u32_hash_buf[0]);
		u32_hash_buf[0] = strtoul(argv[4], NULL, 16);
		release_tftp_buffer(u32_hash_buf[0]);
		#endif

	}

	#if CONFIG_BFN_VERSION == 5
	u32_block_PBA = 2;
	printf("miu write start blk %ld\n",u32_block_PBA);
	do_nand_update_HASH_STBBFN(&u32_block_PBA,u32_hash_buf[0],(S32)u32_hash_size[0],HASH_APP_TagInSpare);
	printf("miu write end blk %ld\n",u32_block_PBA);
	u32_block_PBA++;
	printf("miu backup write start blk %ld\n",u32_block_PBA);
	do_nand_update_HASH_STBBFN(&u32_block_PBA,u32_hash_buf[0],(S32)u32_hash_size[0],HASH_APP_TagInSpare);
	printf("miu backup write end blk %ld\n",u32_block_PBA);
	#endif

	// Reinit NAND Flash

	nand_init();

	#if defined(__VER_UNFD_FTL__) && __VER_UNFD_FTL__
	printf("[%s]: FTL init ......\n", __func__);
	// init
	ret = nand_Init_FTL();
	if(UNFD_ST_SUCCESS != ret)
	{
	   printf("nand_Init_FTL failed!!!\n");
	   return -1;
	}
	#endif //__VER_UNFD_FTL__

	printf("BFN image writen end blk = %ld\n",u32_block_PBA);

	free(pNandDrv_tmp);
	free(pNandInfo);
	free(pPartInfo_tmp);
	return 0;

HASH_ERR:
	memcpy(pNandDrv, pNandDrv_tmp, sizeof(NAND_DRIVER));
	NC_Init();
	NC_Config();
	free(pNandDrv_tmp);
	free(pNandInfo);
	free(pPartInfo_tmp);
	return u32_Err;
}

U_BOOT_CMD(ncisldrapp, 9, 1, do_nand_update_MIUCIS_LDRAPP,
    "write cismiu and LDR0 LDR1 for STB BFN",
    "<CISAddr> <PPMAddr> <PNIAddr> <MIUAddr> <LDRAddr> <LDRSize> <APPAddr> <APPSize> \n"
    "CISAddr   DRAM address of CIS\n"
    "PPMAddr   DRAM address of PPM\n"
    "PNIAddr   DRAM address of PNI\n"
    "LDRAddr   DRAM address of bfn_ldr.bin\n"
    "LDRSize   Total size of the bfn_ldr.bin\n"
    "APPAddr   DRAM address of bfn_app.bin\n"
    "APPSize   Total size of the bfn_app.bin\n"
    "MIUAddr   DRAM address of bfn_miu.bin\n"
    "\n"
);
#endif

int do_nsbootblk(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
	if(info.nandtype == NAND){
		NAND_DRIVER *pNandDrv = (NAND_DRIVER*)drvNAND_get_DrvContext_address();
		if(pNandDrv->u8_BL0PBA != 0){
			if(info.disablembootbak)
				return 1;
			else
				return 2;
		}
		else if(pNandDrv->u8_HashPBA[0][0] != 0)
			return 4;
		else{
			if(info.boottype == ROMBOOT){
				printf("[%s]: fail\n", __func__);
				return -1;
			}
			else if(info.boottype == BFN){
				return 0;
			}
		}
	}
	else if(info.nandtype == SPINAND){
		extern SPI_NAND_DRIVER_t gtSpiNandDrv;
		if(gtSpiNandDrv.tSpinandInfo.u8_BL0PBA != 0){
			if(info.disablembootbak)
				return 1;
			else
				return 2;
		}
		else if(gtSpiNandDrv.tSpinandInfo.u8_HashPBA[0][0] != 0)
			return 4;
		else{
			printf("[%s]: fail\n", __func__);
			return -1;
		}
	}
}

U_BOOT_CMD(nsbootblk, 1, 1, do_nsbootblk,
	"Show how many block count sboot is\n",
	""
);




////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////nand  cmd test////////////////////////
void nand_cmd_test(void)
{
	extern char *root_directory, *image_directory;
	unsigned char *nni_buf, *pni_buf, *ppm_buf, *mboot_buf;
	struct stat st;
	int fd, ret;
	char cmd[128], path[128];

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s%s", root_directory, image_directory, "SPINANDINFO.sni");
//	sprintf(path, "%s%s%s", root_directory, image_directory, "NANDINFO.nni");
	printf("---------path %s\n", path);
	stat(path, &st);
	nni_buf = malloc(st.st_size);
	fd=open(path, O_RDWR);
	if(fd<0){
		printf("fail 1\n");
		return;
	}
	ret = read(fd, nni_buf, st.st_size);
	if(ret<0){
		printf("fail 2\n");
		return;
	}


	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s%s", root_directory, image_directory, "PARTITION_v2_default.pni");
	stat(path, &st);
	pni_buf = malloc(st.st_size);
	fd=open(path, O_RDWR);
	if(fd<0){
		printf("fail 3\n");
		return;
	}
	ret = read(fd, pni_buf, st.st_size);
	if(ret<0){
		printf("fail 4\n");
		return;
	}

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s%s", root_directory, image_directory, "PAIRPAGEMAP_v2.ppm");
	stat(path, &st);
	ppm_buf = malloc(st.st_size);
	fd=open(path, O_RDWR);
	if(fd<0){
		printf("fail 5\n");
		return;
	}
	ret = read(fd, ppm_buf, st.st_size);
	if(ret<0){
		printf("fail 6\n");
		return;
	}

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s%s", root_directory, image_directory, "mboot_nand.bin");
	stat(path, &st);
	mboot_buf = malloc(st.st_size);
	fd=open(path, O_RDWR);
	if(fd<0){
		printf("fail 5\n");
		return;
	}
	ret = read(fd, mboot_buf, st.st_size);
	if(ret<0){
		printf("fail 6\n");
		return;
	}
	printf("mboot path:%s, mboot len %lld\n", path, st.st_size);



	memset(cmd, 0, sizeof(cmd));
	printf("**          u32 %d %ld\n", sizeof(U32), (long)pni_buf);
	sprintf(cmd, "ncisbl %lx %lx %lx %lx %llx", (long)nni_buf, (long)ppm_buf, (long)pni_buf, (long)mboot_buf, st.st_size);
	run_command(cmd, 0);
}

void nand_cmd_test2(void)
{
	extern char *root_directory, *image_directory;
	unsigned char *buf;
	struct stat st;
	int fd, ret;
	char cmd[128], path[128];

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s%s", root_directory, image_directory, "uImage");
	printf("---------path %s\n", path);
	stat(path, &st);
	buf = malloc(st.st_size);
	fd=open(path, O_RDWR);
	if(fd<0){
		printf("fail 1\n");
		return;
	}
	ret = read(fd, buf, st.st_size);
	if(ret<0){
		printf("fail 2\n");
		return;
	}
	run_command("tftp 0x80400000 images/ubifs/uImage");
	sprintf(cmd, "nand write.e %s KL %s", "0x80400000", "$(filesize)");
	run_command(cmd, 0);

//	sprintf(cmd, "nand write.e %x KL %x", (long)buf, st.st_size);
//	run_command(cmd, 0);
}

int writeSpinandCIS(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
    U32 u32_CISAddr = 0;
    U32 u32_CISDataAddr = 0;
    U32 u32_PartInfo = 0;
    U32 u32_PartInfoData = 0;
    int SPINAND_SUCCESS = 0;
    SPINAND_FLASH_INFO_TAG_t pSpiNandInfoTagOut;

    u32_CISDataAddr = u32_CISAddr = strtoul(argv[1], NULL, 0);
    u32_PartInfoData = u32_PartInfo = strtoul(argv[2], NULL, 0);
    search_tftp_buffer(&u32_CISDataAddr);
	search_tftp_buffer(&u32_PartInfoData);

    if(MDrv_SPINAND_SearchCIS_in_DRAM((u8 *)u32_CISDataAddr, NULL, (u8 *)u32_PartInfoData, &pSpiNandInfoTagOut) != SPINAND_SUCCESS)
    {
        printf("SearchCIS_in_DRAM fail");
        return -1;
    }

    if(MDrv_SPINAND_WriteCIS_for_ROM(&pSpiNandInfoTagOut) != ERR_SPINAND_SUCCESS)
    {
        printf("WriteCIS_for_ROM fail");
        return -1;
    }

    release_tftp_buffer(u32_CISAddr);
	release_tftp_buffer(u32_PartInfo);

    printf("Write CIS success!!\n");

    return 0;
}

U_BOOT_CMD(
    writecis,  CONFIG_SYS_MAXARGS,    1,    writeSpinandCIS,
    "Search CIS in dram then write to spinand.",
    ""
);

int writeNandCIS(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
    U32 u32_CISAddr = 0;
    U32 u32_CISDataAddr = 0;
    U32 u32_PartInfo = 0;
    U32 u32_PartInfoData = 0;
    NAND_FLASH_INFO_t pNandInfoTagOut;

    u32_CISDataAddr = u32_CISAddr = strtoul(argv[1], NULL, 0);
    u32_PartInfoData = u32_PartInfo = strtoul(argv[2], NULL, 0);
    search_tftp_buffer(&u32_CISDataAddr);
	search_tftp_buffer(&u32_PartInfoData);

    if(search_cis_in_DRAM((u8 *)u32_CISDataAddr, NULL, (u8 *)u32_PartInfoData, &pNandInfoTagOut) != UNFD_ST_SUCCESS)
    {
        printf("search_cis_in_DRAM fail");
        return -1;
    }

    if(drvNAND_WriteCIS_for_ROM(&pNandInfoTagOut) != UNFD_ST_SUCCESS)
    {
        printf("drvNAND_WriteCIS_for_ROM fail");
        return -1;
    }

    release_tftp_buffer(u32_CISAddr);
	release_tftp_buffer(u32_PartInfo);

    printf("Write CIS success!!\n");

    return 0;

}

U_BOOT_CMD(
    cis,  CONFIG_SYS_MAXARGS,    1,    writeNandCIS,
    "write CIS",
    ""
);

////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
