//<MStar Software>
//******************************************************************************
// MStar Software
// Copyright (c) 2010 - 2012 MStar Semiconductor, Inc. All rights reserved.
// All software, firmware and related documentation herein ("MStar Software") are
// intellectual property of MStar Semiconductor, Inc. ("MStar") and protected by
// law, including, but not limited to, copyright law and international treaties.
// Any use, modification, reproduction, retransmission, or republication of all
// or part of MStar Software is expressly prohibited, unless prior written
// permission has been granted by MStar.
//
// By accessing, browsing and/or using MStar Software, you acknowledge that you
// have read, understood, and agree, to be bound by below terms ("Terms") and to
// comply with all applicable laws and regulations:
//
// 1. MStar shall retain any and all right, ownership and interest to MStar
//    Software and any modification/derivatives thereof.
//    No right, ownership, or interest to MStar Software and any
//    modification/derivatives thereof is transferred to you under Terms.
//
// 2. You understand that MStar Software might include, incorporate or be
//    supplied together with third party`s software and the use of MStar
//    Software may require additional licenses from third parties.
//    Therefore, you hereby agree it is your sole responsibility to separately
//    obtain any and all third party right and license necessary for your use of
//    such third party`s software.
//
// 3. MStar Software and any modification/derivatives thereof shall be deemed as
//    MStar`s confidential information and you agree to keep MStar`s
//    confidential information in strictest confidence and not disclose to any
//    third party.
//
// 4. MStar Software is provided on an "AS IS" basis without warranties of any
//    kind. Any warranties are hereby expressly disclaimed by MStar, including
//    without limitation, any warranties of merchantability, non-infringement of
//    intellectual property rights, fitness for a particular purpose, error free
//    and in conformity with any international standard.  You agree to waive any
//    claim against MStar for any loss, damage, cost or expense that you may
//    incur related to your use of MStar Software.
//    In no event shall MStar be liable for any direct, indirect, incidental or
//    consequential damages, including without limitation, lost of profit or
//    revenues, lost or damage of data, and unauthorized system use.
//    You agree that this Section 4 shall still apply without being affected
//    even if MStar Software has been modified by MStar in accordance with your
//    request or instruction for your use, except otherwise agreed by both
//    parties in writing.
//
// 5. If requested, MStar may from time to time provide technical supports or
//    services in relation with MStar Software to you for your use of
//    MStar Software in conjunction with your or your customer`s product
//    ("Services").
//    You understand and agree that, except otherwise agreed by both parties in
//    writing, Services are provided on an "AS IS" basis and the warranty
//    disclaimer set forth in Section 4 above shall apply.
//
// 6. Nothing contained herein shall be construed as by implication, estoppels
//    or otherwise:
//    (a) conferring any license or right to use MStar name, trademark, service
//        mark, symbol or any other identification;
//    (b) obligating MStar or any of its affiliates to furnish any person,
//        including without limitation, you and your customers, any assistance
//        of any kind whatsoever, or any information; or
//    (c) conferring any license or right under any intellectual property right.
//
// 7. These terms shall be governed by and construed in accordance with the laws
//    of Taiwan, R.O.C., excluding its conflict of law rules.
//    Any and all dispute arising out hereof or related hereto shall be finally
//    settled by arbitration referred to the Chinese Arbitration Association,
//    Taipei in accordance with the ROC Arbitration Law and the Arbitration
//    Rules of the Association by three (3) arbitrators appointed in accordance
//    with the said Rules.
//    The place of arbitration shall be in Taipei, Taiwan and the language shall
//    be English.
//    The arbitration award shall be final and binding to both parties.
//
//******************************************************************************
//<MStar Software>

///////////////////////////////////////////////////////////////////////////////////////////////////
///
/// file    cmd_secure.c
/// @brief  SCS Main Function
/// @author MStar Semiconductor Inc.
///////////////////////////////////////////////////////////////////////////////////////////////////


//-------------------------------------------------------------------------------------------------
//  Include Files
//-------------------------------------------------------------------------------------------------
#include <stdio.h>
#include <stdint.h>
#include <fcntl.h>
#include <string.h>
#include <malloc.h>

#include "MsDebug.h"
#include "MsRawIO.h"
#include "MsSystem.h"
#include "secure/MsSignature.h"

#include "dictionary.h"
#include "libiniparser.h"
#include "tool-util.h"

//-------------------------------------------------------------------------------------------------
//  Debug
//-------------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------------
//  Local Defines
//-------------------------------------------------------------------------------------------------
#define NUMBER_OF_SECURE_INFO 2 // 2 : One is for continue mode, and the other one is for interleave mode

//secure info in spi flash
#define FLASH_DEFAULT_TARGET_SPI  E_RAW_DATA_IN_SPI
#define FLASH_DEFAULT_PARTITION_SPI SPI_DEFAULT_PARTITION
#define FLASH_DEFAULT_VOLUME_SPI SPI_DEFAULT_VOLUME
#define SECTOR_SIZE_SPI   0x10000 

//secure info in nand
#define FLASH_DEFAULT_TARGET_NAND  E_RAW_DATA_IN_NAND
#define FLASH_DEFAULT_PARTITION_NAND NAND_DEFAULT_PARTITION
#define FLASH_DEFAULT_VOLUME_NAND NAND_DEFAULT_VOLUME
#define SECTOR_SIZE_NAND   ubi_get_leb_size()

//secure info in emmc
#define FLASH_DEFAULT_TARGET_EMMC E_RAW_DATA_IN_MMC
#define FLASH_DEFAULT_PARTITION_EMMC MMC_DEFAULT_PARTITION
#define FLASH_DEFAULT_VOLUME_EMMC MMC_DEFAULT_VOLUME
#define SECTOR_SIZE_EMMC   0x200
#define SECURE_INFOR_BACK_OFFSET_EMMC 0x6000

extern int ubi_get_volume_size(char *, size_t* );
extern int ubi_get_leb_size(void);
extern uint32_t  crc32 (uint32_t crc, const unsigned char *p, unsigned int len);
//-------------------------------------------------------------------------------------------------
//  Local Structurs
//-------------------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------------------
//  Global Variables
//-------------------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------------------
//  Local Variables
//-------------------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------------
//  extern function
//-------------------------------------------------------------------------------------------------
extern struct platform_info info;
extern dictionary *dict;
extern int ubi_get_leb_size();
extern int search_tftp_buffer(unsigned long *addr);
extern void release_tftp_buffer(unsigned long addr);
//-------------------------------------------------------------------------------------------------
//  inline
//-------------------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------------------
//  Local function
//-------------------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------------------
//-------------------------------------------------------------------------------------------------
int get_signature_offset(unsigned int *u32SigOffset,unsigned int *u32SigBkOffset)
{
	int ret = 0;
	unsigned int u32Size = 0;
    
	ret = MsApiChunkHeader_GetValue(CH_SECURITY_INFO_AP_ROM_OFFSET,u32SigOffset);


	if(ret != 0)
	{
		printf("[%s]: MsApiChunkHeader_GetValue fail!\n", __func__);
		return -1;
	}

	if(info.rawdata== TOOL_RAWDATA_SPI){
		ret = getSpiSize(&u32Size);
		*u32SigOffset = u32Size - ((*u32SigOffset) * SECTOR_SIZE_SPI);
	}
	else if(info.rawdata == TOOL_RAWDATA_NAND){
		check_ubi_partition(NAND_DEFAULT_VOLUME, NAND_DEFAULT_PARTITION);
		ubi_get_volume_size(NAND_DEFAULT_VOLUME, &u32Size);
		printf("[%s]: u32Size : 0x%x\n", __func__, u32Size);
		printf("[%s]: ubi_get_leb_size : 0x%x\n", __func__, SECTOR_SIZE_NAND);
		*u32SigOffset = u32Size - ((*u32SigOffset)*SECTOR_SIZE_NAND);
	}
	else if(info.rawdata == TOOL_RAWDATA_MMC){
		ret = get_mmc_partsize(FLASH_DEFAULT_VOLUME_EMMC, &u32Size);
		*u32SigOffset = u32Size - ((*u32SigOffset) * SECTOR_SIZE_EMMC);
	}
	else{
		printf("[%s]: Unknown secure store type\n", __func__);
		return -1;
	}

	if(info.rawdata == TOOL_RAWDATA_MMC)
		*u32SigBkOffset=*u32SigOffset+SECURE_INFOR_BACK_OFFSET_EMMC;
	else if(info.rawdata == TOOL_RAWDATA_NAND)
		*u32SigBkOffset=*u32SigOffset+SECTOR_SIZE_NAND;
	else if(info.rawdata== TOOL_RAWDATA_SPI)
		*u32SigBkOffset=*u32SigOffset+SECTOR_SIZE_SPI;
    
	if(EN_SUCCESS == ret){
		ret=0;
	}
	else{
		printf("[%s]: failed\n", __func__);
		ret=-1;
	}
    
	return ret;
}

int SignatureSave_Android(SECURITY_INFO_ANDROID *pBufferAddr)
{   
	int ret=-1;
	int ret_bk =-1;
	unsigned int u32SigOffset = 0;  
	unsigned int u32SigBkOffset = 0; 
	unsigned int u32SecuritySize= 0;
	   
	//Here, we check the CRC of SECUREITY_INFO, and the check range is from "pBufferAddr->data" to "pBufferAddr->data_interleave"
	u32SecuritySize = sizeof(_SECURITY_INFO_ANDROID_DATA) * NUMBER_OF_SECURE_INFO;

	if(pBufferAddr==NULL)
	{
		UBOOT_ERROR("The input parameter pBufferAddr' is a null pointer\n");
		return -1;
	}

	if(info.rawdata== TOOL_RAWDATA_SPI)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_SPI, FLASH_DEFAULT_PARTITION_SPI, FLASH_DEFAULT_VOLUME_SPI);
	else if(info.rawdata== TOOL_RAWDATA_NAND)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_NAND, FLASH_DEFAULT_PARTITION_NAND, FLASH_DEFAULT_VOLUME_NAND);
	else if(info.rawdata== TOOL_RAWDATA_MMC)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_EMMC, FLASH_DEFAULT_PARTITION_EMMC, FLASH_DEFAULT_VOLUME_EMMC);
	else{
		printf("[%s]: Unknown secure media\n", __func__);
		return -1;
	}
	
	if(ret != 0)
	{
		UBOOT_ERROR("raw_io_config setting fail!\n");
		return -1;
	}

	ret = get_signature_offset(&u32SigOffset,&u32SigBkOffset);
	if(ret != 0)
	{
		UBOOT_ERROR("get_signature_offset fail!\n");
		return -1;
	}

	UBOOT_DEBUG("u32SigOffset : 0x%x\n",u32SigOffset);
	UBOOT_DEBUG("u32SigBkOffset : 0x%x\n",u32SigBkOffset);

	// update CRC
	pBufferAddr->crc = crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize);
	ret = raw_write((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_ANDROID));
	ret_bk = raw_write((unsigned int)pBufferAddr,u32SigBkOffset,sizeof(SECURITY_INFO_ANDROID));

	if(EN_SUCCESS == ret || EN_SUCCESS == ret_bk)
	{
		UBOOT_DEBUG("ret : %d , ret_bk : %d \n",ret,ret_bk);
		ret = 0;       
	}
	else
	{
		UBOOT_ERROR("SignatureSave fail\n");
		ret= -1;
	}

	return ret;
}

int SignatureSave_Android_Store(SECURITY_INFO_ANDROID_SECURESTORE *pBufferAddr)
{   
	int ret=-1;
	int ret_bk =-1;
	unsigned int u32SigOffset = 0;  
	unsigned int u32SigBkOffset = 0; 
	unsigned int u32SecuritySize= 0;
	   
	//Here, we check the CRC of SECUREITY_INFO, and the check range is from "pBufferAddr->data" to "pBufferAddr->data_interleave"
	u32SecuritySize = sizeof(_SECURITY_INFO_ANDROID_SECURESTORE_DATA) * NUMBER_OF_SECURE_INFO;

	if(pBufferAddr==NULL)
	{
		UBOOT_ERROR("The input parameter pBufferAddr' is a null pointer\n");
		return -1;
	}

	if(info.rawdata == TOOL_RAWDATA_SPI)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_SPI, FLASH_DEFAULT_PARTITION_SPI, FLASH_DEFAULT_VOLUME_SPI);
	else if(info.rawdata == TOOL_RAWDATA_NAND)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_NAND, FLASH_DEFAULT_PARTITION_NAND, FLASH_DEFAULT_VOLUME_NAND);
	else if(info.rawdata == TOOL_RAWDATA_MMC)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_EMMC, FLASH_DEFAULT_PARTITION_EMMC, FLASH_DEFAULT_VOLUME_EMMC);
	else{
		printf("[%s]: Unknown secure media\n", __func__);
		return -1;
	}

	if(ret != 0)
	{
		UBOOT_ERROR("raw_io_config setting fail!\n");
		return -1;
	}

	ret = get_signature_offset(&u32SigOffset,&u32SigBkOffset);
	if(ret != 0)
	{
		UBOOT_ERROR("get_signature_offset fail!\n");
		return -1;
	}

	UBOOT_DEBUG("u32SigOffset : 0x%x\n",u32SigOffset);
	UBOOT_DEBUG("u32SigBkOffset : 0x%x\n",u32SigBkOffset);

	// update CRC
	pBufferAddr->crc = crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize);
	ret = raw_write((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_ANDROID_SECURESTORE));
	ret_bk = raw_write((unsigned int)pBufferAddr,u32SigBkOffset,sizeof(SECURITY_INFO_ANDROID_SECURESTORE));

	if(EN_SUCCESS == ret || EN_SUCCESS == ret_bk)
	{
		UBOOT_DEBUG("ret : %d , ret_bk : %d \n",ret,ret_bk);
		ret = 0;       
	}
	else
	{
		UBOOT_ERROR("SignatureSave fail\n");
		ret= -1;
	}

	return ret;
}

int SignatureSave_None_Android(SECURITY_INFO_NONE_ANDROID *pBufferAddr)
{   
	int ret=-1;
	int ret_bk =-1;
	unsigned int u32SigOffset = 0;  
	unsigned int u32SigBkOffset = 0; 
	unsigned int u32SecuritySize= 0;
	   
	//Here, we check the CRC of SECUREITY_INFO, and the check range is from "pBufferAddr->data" to "pBufferAddr->data_interleave"
	u32SecuritySize = sizeof(_SECURITY_INFO_NONE_ANDROID_DATA) * NUMBER_OF_SECURE_INFO;

	if(pBufferAddr==NULL)
	{
		UBOOT_ERROR("The input parameter pBufferAddr' is a null pointer\n");
		return -1;
	}

	if(info.rawdata == TOOL_RAWDATA_SPI)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_SPI, FLASH_DEFAULT_PARTITION_SPI, FLASH_DEFAULT_VOLUME_SPI);
	else if(info.rawdata == TOOL_RAWDATA_NAND)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_NAND, FLASH_DEFAULT_PARTITION_NAND, FLASH_DEFAULT_VOLUME_NAND);
	else if(info.rawdata == TOOL_RAWDATA_MMC)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_EMMC, FLASH_DEFAULT_PARTITION_EMMC, FLASH_DEFAULT_VOLUME_EMMC);
	else{
		printf("[%s]: Unknown secure media\n", __func__);
		return -1;
	}

	if(ret != 0)
	{
		UBOOT_ERROR("raw_io_config setting fail!\n");
		return -1;
	}

	ret = get_signature_offset(&u32SigOffset,&u32SigBkOffset);
	if(ret != 0)
	{
		UBOOT_ERROR("get_signature_offset fail!\n");
		return -1;
	}

	UBOOT_DEBUG("u32SigOffset : 0x%x\n",u32SigOffset);
	UBOOT_DEBUG("u32SigBkOffset : 0x%x\n",u32SigBkOffset);

	// update CRC
	pBufferAddr->crc = crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize);
	ret = raw_write((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_NONE_ANDROID));
	ret_bk = raw_write((unsigned int)pBufferAddr,u32SigBkOffset,sizeof(SECURITY_INFO_NONE_ANDROID));

	if(EN_SUCCESS == ret || EN_SUCCESS == ret_bk)
	{
		UBOOT_DEBUG("ret : %d , ret_bk : %d \n",ret,ret_bk);
		ret = 0;       
	}
	else
	{
		UBOOT_ERROR("SignatureSave fail\n");
		ret= -1;
	}

	return ret;
}


int SignatureLoad_Android(SECURITY_INFO_ANDROID *pBufferAddr)
{
	int ret = -1;
	int flag1=0, flag2=0;    
	unsigned int u32SigOffset = 0;
	unsigned int u32SigBkOffset = 0;     
	unsigned int u32SecuritySize= 0;
	UBOOT_TRACE("IN\n");

	//Here, we check the CRC of SECUREITY_INFO, 
	//and the check range include "pBufferAddr->data" and "pBufferAddr->data_interleave"
	u32SecuritySize = sizeof(_SECURITY_INFO_ANDROID_DATA) * NUMBER_OF_SECURE_INFO;

	if(pBufferAddr==NULL)
	{
		UBOOT_ERROR("The input parameter pBufferAddr' is a null pointer\n");
		return -1;
	}

	if(info.rawdata == TOOL_RAWDATA_SPI)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_SPI, FLASH_DEFAULT_PARTITION_SPI, FLASH_DEFAULT_VOLUME_SPI);
	else if(info.rawdata == TOOL_RAWDATA_NAND)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_NAND, FLASH_DEFAULT_PARTITION_NAND, FLASH_DEFAULT_VOLUME_NAND);
	else if(info.rawdata == TOOL_RAWDATA_MMC)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_EMMC, FLASH_DEFAULT_PARTITION_EMMC, FLASH_DEFAULT_VOLUME_EMMC);
	else{
		printf("[%s]: Unknown secure media\n", __func__);
		return -1;
	}
	if(ret != 0)
	{
		UBOOT_ERROR("raw_io_config setting fail!\n");
		return -1;
	}

	ret = get_signature_offset(&u32SigOffset,&u32SigBkOffset);
	if(ret != 0)
	{
		UBOOT_ERROR("get_signature_offset fail!\n");
		return -1;
	}
   
	ret = raw_read((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_ANDROID));
    
	if( (EN_SUCCESS == ret) && (pBufferAddr->crc == crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize)) )
		flag1=1;
    
	ret = raw_read((unsigned int)pBufferAddr,u32SigBkOffset,sizeof(SECURITY_INFO_ANDROID));
    
	if( (EN_SUCCESS == ret) && (pBufferAddr->crc == crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize)) )
		flag2=1;

	if( (flag2==0) && (flag1!=0) )
	{
		ret = raw_read((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_ANDROID));
		if( (EN_SUCCESS == ret) && (pBufferAddr->crc == crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize)))
		{
			ret = raw_write((unsigned int)pBufferAddr,u32SigBkOffset,sizeof(SECURITY_INFO_ANDROID));
		}
		else
		{
			UBOOT_ERROR("raw_read fail or caculate crc fail!\n");
			return -1;
		}
	}

	if((flag1==0)&&(flag2!=0))
	{
		ret = raw_write((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_ANDROID));
	}

	if(EN_SUCCESS == ret)
	{
		ret=0;
		UBOOT_TRACE("OK\n");
	}
	else
	{
		ret=-1;
		UBOOT_ERROR("SignatureLoad fail\n");
	}

	return ret;
}

int SignatureLoad_Android_Store(SECURITY_INFO_ANDROID_SECURESTORE *pBufferAddr)
{
	int ret = -1;
	int flag1=0, flag2=0;    
	unsigned int u32SigOffset = 0;
	unsigned int u32SigBkOffset = 0;     
	unsigned int u32SecuritySize= 0;
	UBOOT_TRACE("IN\n");

	//Here, we check the CRC of SECUREITY_INFO, 
	//and the check range include "pBufferAddr->data" and "pBufferAddr->data_interleave"
	u32SecuritySize = sizeof(_SECURITY_INFO_ANDROID_SECURESTORE_DATA) * NUMBER_OF_SECURE_INFO;

	if(pBufferAddr==NULL)
	{
		UBOOT_ERROR("The input parameter pBufferAddr' is a null pointer\n");
		return -1;
	}

	if(info.rawdata == TOOL_RAWDATA_SPI)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_SPI, FLASH_DEFAULT_PARTITION_SPI, FLASH_DEFAULT_VOLUME_SPI);
	else if(info.rawdata == TOOL_RAWDATA_NAND)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_NAND, FLASH_DEFAULT_PARTITION_NAND, FLASH_DEFAULT_VOLUME_NAND);
	else if(info.rawdata == TOOL_RAWDATA_MMC)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_EMMC, FLASH_DEFAULT_PARTITION_EMMC, FLASH_DEFAULT_VOLUME_EMMC);
	else{
		printf("[%s]: Unknown secure media\n", __func__);
		return -1;
	}
	if(ret != 0)
	{
		UBOOT_ERROR("raw_io_config setting fail!\n");
		return -1;
	}

	ret = get_signature_offset(&u32SigOffset,&u32SigBkOffset);
	if(ret != 0)
	{
		UBOOT_ERROR("get_signature_offset fail!\n");
		return -1;
	}
   
	ret = raw_read((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_ANDROID_SECURESTORE));
    
	if( (EN_SUCCESS == ret) && (pBufferAddr->crc == crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize)) )
		flag1=1;
    
	ret = raw_read((unsigned int)pBufferAddr,u32SigBkOffset,sizeof(SECURITY_INFO_ANDROID_SECURESTORE));
    
	if( (EN_SUCCESS == ret) && (pBufferAddr->crc == crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize)) )
		flag2=1;

	if( (flag2==0) && (flag1!=0) )
	{
		ret = raw_read((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_ANDROID_SECURESTORE));
		if( (EN_SUCCESS == ret) && (pBufferAddr->crc == crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize)))
		{
			ret = raw_write((unsigned int)pBufferAddr,u32SigBkOffset,sizeof(SECURITY_INFO_ANDROID_SECURESTORE));
		}
		else
		{
			UBOOT_ERROR("raw_read fail or caculate crc fail!\n");
			return -1;
		}
	}

	if((flag1==0)&&(flag2!=0))
	{
		ret = raw_write((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_ANDROID_SECURESTORE));
	}

	if(EN_SUCCESS == ret)
	{
		ret=0;
		UBOOT_TRACE("OK\n");
	}
	else
	{
		ret=-1;
		UBOOT_ERROR("SignatureLoad fail\n");
	}

	return ret;
}

int SignatureLoad_None_Android(SECURITY_INFO_NONE_ANDROID*pBufferAddr)
{
	int ret = -1;
	int flag1=0, flag2=0;    
	unsigned int u32SigOffset = 0;
	unsigned int u32SigBkOffset = 0;     
	unsigned int u32SecuritySize= 0;
	UBOOT_TRACE("IN\n");

	//Here, we check the CRC of SECUREITY_INFO, 
	//and the check range include "pBufferAddr->data" and "pBufferAddr->data_interleave"
	u32SecuritySize = sizeof(_SECURITY_INFO_NONE_ANDROID_DATA) * NUMBER_OF_SECURE_INFO;

	if(pBufferAddr==NULL)
	{
		UBOOT_ERROR("The input parameter pBufferAddr' is a null pointer\n");
		return -1;
	}

	if(info.rawdata == TOOL_RAWDATA_SPI)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_SPI, FLASH_DEFAULT_PARTITION_SPI, FLASH_DEFAULT_VOLUME_SPI);
	else if(info.rawdata == TOOL_RAWDATA_NAND)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_NAND, FLASH_DEFAULT_PARTITION_NAND, FLASH_DEFAULT_VOLUME_NAND);
	else if(info.rawdata == TOOL_RAWDATA_MMC)
		ret = raw_io_config(FLASH_DEFAULT_TARGET_EMMC, FLASH_DEFAULT_PARTITION_EMMC, FLASH_DEFAULT_VOLUME_EMMC);
	else{
		printf("[%s]: Unknown secure media\n", __func__);
		return -1;
	}
	if(ret != 0)
	{
		UBOOT_ERROR("raw_io_config setting fail!\n");
		return -1;
	}

	ret = get_signature_offset(&u32SigOffset,&u32SigBkOffset);
	if(ret != 0)
	{
		UBOOT_ERROR("get_signature_offset fail!\n");
		return -1;
	}
   
	ret = raw_read((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_NONE_ANDROID));
    
	if( (EN_SUCCESS == ret) && (pBufferAddr->crc == crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize)) )
		flag1=1;
    
	ret = raw_read((unsigned int)pBufferAddr,u32SigBkOffset,sizeof(SECURITY_INFO_NONE_ANDROID));
    
	if( (EN_SUCCESS == ret) && (pBufferAddr->crc == crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize)) )
		flag2=1;

	UBOOT_DEBUG("flag 1=%d, flad 2=%d\n", flag1, flag2);

	if( (flag2==0) && (flag1!=0) )
	{
		ret = raw_read((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_NONE_ANDROID));
		if( (EN_SUCCESS == ret) && (pBufferAddr->crc == crc32(0, (unsigned char const *)&pBufferAddr->data,u32SecuritySize)))
		{
			ret = raw_write((unsigned int)pBufferAddr,u32SigBkOffset,sizeof(SECURITY_INFO_NONE_ANDROID));
		}
		else
		{
			UBOOT_ERROR("raw_read fail or caculate crc fail!\n");
			return -1;
		}
	}

	if((flag1==0)&&(flag2!=0))
	{
		ret = raw_write((unsigned int)pBufferAddr,u32SigOffset,sizeof(SECURITY_INFO_NONE_ANDROID));
	}

	if(EN_SUCCESS == ret)
	{
		ret=0;
		UBOOT_TRACE("OK\n");
	}
	else
	{
		ret=-1;
		UBOOT_ERROR("SignatureLoad fail\n");
	}

	return ret;
}

#define OBJECT_NAME argv[1]
#define OBJECT_DRAM_ADDR s_addr
static int do_save_secure_info_android(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int ret=0;
	char s_addr[16];
	unsigned long l_addr; 
	SUB_SECURE_INFO *pSubInfo;
	SECURITY_INFO_ANDROID *pSecureInfo = (SECURITY_INFO_ANDROID *)malloc(sizeof(SECURITY_INFO_ANDROID));

	if(argc<3)
	{
		cmd_usage(cmdtp);
		return -1;
	}

	memset(s_addr, 0, sizeof(s_addr));
	l_addr = strtoul(argv[2], NULL, 16);
	ret = search_tftp_buffer(&l_addr);
	sprintf(s_addr, "%lx", l_addr);

	pSubInfo =(SUB_SECURE_INFO *)strtoul(OBJECT_DRAM_ADDR, NULL, 16);

	if(pSecureInfo==NULL)
	{
		release_tftp_buffer(strtoul(argv[2], NULL, 16));
		UBOOT_ERROR("malloc fail\n");
		return -1;
	}
    
	memset((void *)pSecureInfo,0,sizeof(SECURITY_INFO_ANDROID));
    
	ret = SignatureLoad_Android(pSecureInfo);
	if(EN_ERROR_OF_CMD == ret)
	{
		free(pSecureInfo);
		release_tftp_buffer(strtoul(argv[2], NULL, 16));
		UBOOT_ERROR("SignatureLoad fail\n");
		return -1;
	}
	else if(EN_ERROR_OF_CRC == ret)
	{
		UBOOT_INFO("\x1b[37;46m ===== [%s:%d] SECURITY_INFO might first upgrade !!! ===== \x1b[0m\n",__FUNCTION__,__LINE__);
	}

	if(strcmp(OBJECT_NAME,"keySet")==0){
		UBOOT_DEBUG("do_save_secure_info: keySet\n");
		memcpy((void *)&pSecureInfo->data.Key,(void *)strtoul(OBJECT_DRAM_ADDR, NULL, 16),sizeof(SECURE_KEY_SET));
		memcpy((void *)&pSecureInfo->data_interleave.Key,(void *)strtoul(OBJECT_DRAM_ADDR, NULL, 16),sizeof(SECURE_KEY_SET));        
	}
	else{
		if(0==strcmp(OBJECT_NAME,"bootSign"))
		{
			UBOOT_DEBUG("do_save_secure_info: bootSign\n");
			memcpy((void *)&pSecureInfo->data.Boot,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.Boot,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));
	        }
	        else if(0==strcmp(OBJECT_NAME,"recoverySign"))
	        {
			UBOOT_DEBUG("do_save_secure_info: recoverySign\n");
			memcpy((void *)&pSecureInfo->data.Recovery,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.Recovery,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));
		}
		else if(strcmp(OBJECT_NAME,"teeSign")==0){
			UBOOT_DEBUG("do_save_secure_info: teeSign\n");
			memcpy((void *)&pSecureInfo->data.tee,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.tee,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));            
		}
		else if(strcmp(OBJECT_NAME,"keySetSign")==0){
			UBOOT_DEBUG("do_save_secure_info: keySetSign\n");
		}
		else if(strcmp(OBJECT_NAME,"seckernelSign")==0){
			UBOOT_DEBUG("do_save_secure_info: seckernelSign\n");
			memcpy((void *)&pSecureInfo->data.secKernel,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.secKernel,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));            
		}
		else
		{
			free(pSecureInfo);
			release_tftp_buffer(strtoul(argv[2], NULL, 16));
			UBOOT_ERROR("Invalid symbol %s\n",OBJECT_NAME);
			return -1;
		}
	}


	ret = SignatureSave_Android(pSecureInfo);
	if(EN_SUCCESS != ret)
	{
		free(pSecureInfo);
		release_tftp_buffer(strtoul(argv[2], NULL, 16));
		UBOOT_ERROR("SignatureSave fail\n");
		return -1;
	}
    
	free(pSecureInfo);
	release_tftp_buffer(strtoul(argv[2], NULL, 16));
	UBOOT_TRACE("OK\n");
	return 0;
}

static int do_save_secure_info_android_store(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int ret=0;
	char s_addr[16];
	unsigned long l_addr; 

	SUB_SECURE_INFO *pSubInfo = NULL;
	SECURITY_INFO_ANDROID_SECURESTORE *pSecureInfo = (SECURITY_INFO_ANDROID_SECURESTORE *)malloc(sizeof(SECURITY_INFO_ANDROID_SECURESTORE));

	if(argc<3)
	{
		cmd_usage(cmdtp);
		return -1;
	}

	memset(s_addr, 0, sizeof(s_addr));
	l_addr = strtoul(argv[2], NULL, 16);
	ret = search_tftp_buffer(&l_addr);
	sprintf(s_addr, "%lx", l_addr);
	
	pSubInfo=(SUB_SECURE_INFO *)strtoul(OBJECT_DRAM_ADDR, NULL, 16);

	if(pSecureInfo==NULL)
	{
		release_tftp_buffer(strtoul(argv[2], NULL, 16));
		UBOOT_ERROR("malloc fail\n");
		return -1;
	}
    
	memset((void *)pSecureInfo,0,sizeof(SECURITY_INFO_ANDROID_SECURESTORE));
    
	ret = SignatureLoad_Android_Store(pSecureInfo);
	if(EN_ERROR_OF_CMD == ret)
	{
		free(pSecureInfo);
		release_tftp_buffer(strtoul(argv[2], NULL, 16));
		UBOOT_ERROR("SignatureLoad fail\n");
		return -1;
	}
	else if(EN_ERROR_OF_CRC == ret)
	{
		UBOOT_INFO("\x1b[37;46m ===== [%s:%d] SECURITY_INFO might first upgrade !!! ===== \x1b[0m\n",__FUNCTION__,__LINE__);
	}

	if(strcmp(OBJECT_NAME,"keySet")==0){
		UBOOT_DEBUG("do_save_secure_info: keySet\n");
		memcpy((void *)&pSecureInfo->data.Key,(void *)strtoul(OBJECT_DRAM_ADDR, NULL, 16),sizeof(SECURE_KEY_SET));
		memcpy((void *)&pSecureInfo->data_interleave.Key,(void *)strtoul(OBJECT_DRAM_ADDR, NULL, 16),sizeof(SECURE_KEY_SET));        
	}
	else{
		if(0==strcmp(OBJECT_NAME,"bootSign"))
		{
			UBOOT_DEBUG("do_save_secure_info: bootSign\n");
			memcpy((void *)&pSecureInfo->data.Boot,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.Boot,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));
	        }
	        else if(0==strcmp(OBJECT_NAME,"recoverySign"))
	        {
			UBOOT_DEBUG("do_save_secure_info: recoverySign\n");
			memcpy((void *)&pSecureInfo->data.Recovery,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.Recovery,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));
		}
		else if(strcmp(OBJECT_NAME,"teeSign")==0){
			UBOOT_DEBUG("do_save_secure_info: teeSign\n");
			memcpy((void *)&pSecureInfo->data.tee,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.tee,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));            
		}
		else if(strcmp(OBJECT_NAME,"keySetSign")==0){
			UBOOT_DEBUG("do_save_secure_info: keySetSign\n");
		}
		else if(strcmp(OBJECT_NAME,"ChunkSign")==0){
			UBOOT_DEBUG("do_save_secure_info: ChunkSign\n");
			memcpy((void *)&pSecureInfo->data.Chunk,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.Chunk,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));            
		}
		else if(strcmp(OBJECT_NAME,"ChunkBackupSign")==0){
			UBOOT_DEBUG("do_save_secure_info: ChunkbackupSign\n");
			memcpy((void *)&pSecureInfo->data.Chunk_backup,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.Chunk_backup,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));            
		}
		else if(strcmp(OBJECT_NAME,"keySetSign")==0){
			UBOOT_DEBUG("do_save_secure_info: keySetSign\n");
		}
		else
		{
			free(pSecureInfo);
			release_tftp_buffer(strtoul(argv[2], NULL, 16));
			UBOOT_ERROR("Invalid symbol %s\n",OBJECT_NAME);
			return -1;
		}
	}


	ret = SignatureSave_Android_Store(pSecureInfo);
	if(EN_SUCCESS != ret)
	{
		free(pSecureInfo);
		release_tftp_buffer(strtoul(argv[2], NULL, 16));
		UBOOT_ERROR("SignatureSave fail\n");
		return -1;
	}

	free(pSecureInfo);
	release_tftp_buffer(strtoul(argv[2], NULL, 16));
	UBOOT_TRACE("OK\n");
	return 0;
}

static int do_save_secure_info_none_android(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int ret=0;
	char s_addr[16];
	unsigned long l_addr; 

	SUB_SECURE_INFO *pSubInfo = NULL;
	SUB_SECURE_INFO_FOR_PARTIAL_AUTH *pSubInfoForParAuth = NULL;
	SECURITY_INFO_NONE_ANDROID *pSecureInfo = (SECURITY_INFO_NONE_ANDROID *)malloc(sizeof(SECURITY_INFO_NONE_ANDROID));

	if(argc<3)
	{
		cmd_usage(cmdtp);
		return -1;
	}

	memset(s_addr, 0, sizeof(s_addr));
	l_addr = strtoul(argv[2], NULL, 16);
	ret = search_tftp_buffer(&l_addr);
	sprintf(s_addr, "%lx", l_addr);

	pSubInfo=(SUB_SECURE_INFO *)strtoul(OBJECT_DRAM_ADDR, NULL, 16);
	pSubInfoForParAuth=(SUB_SECURE_INFO_FOR_PARTIAL_AUTH *)strtoul(OBJECT_DRAM_ADDR, NULL, 16);

	if(pSecureInfo==NULL)
	{
		release_tftp_buffer(strtoul(argv[2], NULL, 16));
		UBOOT_ERROR("malloc fail\n");
		return -1;
	}
    
	memset((void *)pSecureInfo,0,sizeof(SECURITY_INFO_NONE_ANDROID));
    
	ret = SignatureLoad_None_Android(pSecureInfo);
	if(EN_ERROR_OF_CMD == ret)
	{
		free(pSecureInfo);
		release_tftp_buffer(strtoul(argv[2], NULL, 16));
		UBOOT_ERROR("SignatureLoad fail\n");
		return -1;
	}
	else if(EN_ERROR_OF_CRC == ret)
	{
		UBOOT_INFO("\x1b[37;46m ===== [%s:%d] SECURITY_INFO might first upgrade !!! ===== \x1b[0m\n",__FUNCTION__,__LINE__);
	}

	if(strcmp(OBJECT_NAME,"keySet")==0){
		UBOOT_DEBUG("do_save_secure_info: keySet\n");
		memcpy((void *)&pSecureInfo->data.Key,(void *)strtoul(OBJECT_DRAM_ADDR, NULL, 16),sizeof(SECURE_KEY_SET));
		memcpy((void *)&pSecureInfo->data_interleave.Key,(void *)strtoul(OBJECT_DRAM_ADDR, NULL, 16),sizeof(SECURE_KEY_SET));        
	}
	else{
		if(strcmp(OBJECT_NAME,"kernelSign")==0){
			UBOOT_DEBUG("do_save_secure_info: kernelSign\n");
			memcpy((void *)&pSecureInfo->data.Kernel,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.Kernel,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));            
		}
		else if(strcmp(OBJECT_NAME,"ROOTFSSign")==0){
			UBOOT_DEBUG("do_save_secure_info: ROOTFSSign\n");
			memcpy((void *)pSecureInfo->data.RootFs,(void *)(&pSubInfoForParAuth->sInfo),sizeof(_SUB_SECURE_INFO)*FRAGMENT_NUM);
			memcpy((void *)pSecureInfo->data_interleave.RootFs,(void *)(&pSubInfoForParAuth->sInfo_Interleave),sizeof(_SUB_SECURE_INFO)*FRAGMENT_NUM);
		}
		else if(strcmp(OBJECT_NAME,"mslibSign")==0){
			UBOOT_DEBUG("do_save_secure_info mslibSign:\n");
			memcpy((void *)pSecureInfo->data.MsLib,(void *)(&pSubInfoForParAuth->sInfo),sizeof(_SUB_SECURE_INFO)*FRAGMENT_NUM);
			memcpy((void *)pSecureInfo->data_interleave.MsLib,(void *)(&pSubInfoForParAuth->sInfo_Interleave),sizeof(_SUB_SECURE_INFO)*FRAGMENT_NUM);
		}
		else if(strcmp(OBJECT_NAME,"configSign")==0){
			UBOOT_DEBUG("do_save_secure_info configSign:\n");
			memcpy((void *)pSecureInfo->data.Config,(void *)(&pSubInfoForParAuth->sInfo),sizeof(_SUB_SECURE_INFO)*FRAGMENT_NUM);
			memcpy((void *)pSecureInfo->data_interleave.Config,(void *)(&pSubInfoForParAuth->sInfo_Interleave),sizeof(_SUB_SECURE_INFO)*FRAGMENT_NUM);
		}
		else if(strcmp(OBJECT_NAME,"applicationsSign")==0){
			UBOOT_DEBUG("do_save_secure_info: applicationsSign\n");
			memcpy((void *)pSecureInfo->data.App,(void *)(&pSubInfoForParAuth->sInfo),sizeof(_SUB_SECURE_INFO)*FRAGMENT_NUM);
			memcpy((void *)pSecureInfo->data_interleave.App,(void *)(&pSubInfoForParAuth->sInfo_Interleave),sizeof(_SUB_SECURE_INFO)*FRAGMENT_NUM);
		}
		else if(strcmp(OBJECT_NAME,"teeSign")==0){
			UBOOT_DEBUG("do_save_secure_info: teeSign\n");
			memcpy((void *)&pSecureInfo->data.tee,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.tee,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));            
		}
		else if(strcmp(OBJECT_NAME,"keySetSign")==0){
			UBOOT_DEBUG("do_save_secure_info: keySetSign\n");
		}
		else if(0==strcmp(OBJECT_NAME,"recoverySign")){
			UBOOT_DEBUG("do_save_secure_info: recoverySign\n");
			memcpy((void *)&pSecureInfo->data.Recovery,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.Recovery,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));
		}
		else if(strcmp(OBJECT_NAME,"seckernelSign")==0){
			UBOOT_DEBUG("do_save_secure_info: seckernelSign\n");
			memcpy((void *)&pSecureInfo->data.secKernel,(void *)(&pSubInfo->sInfo),sizeof(_SUB_SECURE_INFO));
			memcpy((void *)&pSecureInfo->data_interleave.secKernel,(void *)(&pSubInfo->sInfo_Interleave),sizeof(_SUB_SECURE_INFO));            
		}
		else
		{
			free(pSecureInfo);
			release_tftp_buffer(strtoul(argv[2], NULL, 16));
			UBOOT_ERROR("Invalid symbol %s\n",OBJECT_NAME);
			return -1;
		}
	}

	ret = SignatureSave_None_Android(pSecureInfo);
	if(EN_SUCCESS != ret)
	{
		free(pSecureInfo);
		release_tftp_buffer(strtoul(argv[2], NULL, 16));
		UBOOT_ERROR("SignatureSave fail\n");
		return -1;
	}
    
	free(pSecureInfo);
	release_tftp_buffer(strtoul(argv[2], NULL, 16));
	UBOOT_TRACE("OK\n");
	return 0;
}

int do_save_secure_info(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int ret;

	//when making emmb bin, MPOOL.bin should be create before,
	//maybe support emmc raw data read/write later
	//so just skip emmc raw data rw now, 
	//---------xiaohui.zhu, 2014.4.10
	if(info.rawdata != TOOL_RAWDATA_NAND){
		UBOOT_DEBUG("Raw data Not in nand, skip save\n");
		return 0;
	}

	if(info.securetype == S_ANDROID)
		return do_save_secure_info_android(cmdtp, flag, argc, argv);
	else if(info.securetype == S_ANDROID_STORE)
		return do_save_secure_info_android_store(cmdtp, flag, argc, argv);
	else if(info.securetype == S_NONE_ANDROID)
		return do_save_secure_info_none_android(cmdtp, flag, argc, argv);
	else{
		printf("[%s]: Unknown secure type\n", __func__);
		return -1;
	}
}

int do_init_secure_info(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	char *val;

	val = iniparser_getstring(dict, "secure:type", "Not Set");
	if(strcmp(val, "android") == 0){
		info.securetype = S_ANDROID;
	}
	else if(strcmp(val, "android_s") == 0){
		info.securetype = S_ANDROID_STORE;
	}
	else
		info.securetype = S_NONE_ANDROID;

	if(info.securetype == S_ANDROID)
		val = "android";
	if(info.securetype == S_ANDROID_STORE)
		val = "android store";
	if(info.securetype == S_NONE_ANDROID)
		val = "none android";	

	UBOOT_DEBUG("Secure type: %s\n", val);
	return 0;
}
