/***********************************************************
*
* Copyright (C), 2010-2014, TP-LINK TECHNOLOGIES CO., LTD.
* All rights reserved.
*
* File name  : tar-handler.c
* Version	 : 1.0
* Description:
*
* Author	 : Wang FuYu <wangfuyu@tp-link.net>
* Create Date: 2014-04-09
*
* History	 :
*------------------------------------------------------------
*
*01, 09Apr14, Wang FuYu, create file.
************************************************************/

#include "make-plugin.h"
#include "tar-handler.h"
#include "common.h"
#include <inttypes.h>
#include "compress-handler.h"

HEADER_TAG gImgHeader;
EFS_HEADER gEfsHeader;
EFS_FOLDER_LEVEL gEfsFolderLevel;
extern char gDirPath[MAX_FILENAME_LEN];

/******************************************************************************
* FUNCTION		: writeFilesToBuf()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: read compressed file data to buf
* INPUT			: @filePath:full path of file
				  @buf:buffer for saving file data
				  @compress_len: data len after compressing
* OUTPUT		:
* RETURN		:
******************************************************************************/
int compressFilesToBuf(char* filePath, unsigned char* buf, int *compress_len)
{
	FILE* fp = NULL;
	int readBytes = 0;
	uint32_t file_len = 0;
	unsigned char *file_buffer = NULL;

	if (NULL == filePath || NULL == buf)
	{
		MSG_ERR("input paras invalid!\n");
		return -1;
	}

	fp = fopen(filePath, "rb");
	if(!fp)
	{
		return -1;
	}

	fseek(fp, 0L, SEEK_END);
	file_len = ftell(fp);
	fseek(fp, 0L, SEEK_SET);

	file_buffer = (unsigned char *)malloc(file_len);
	if (file_buffer == NULL)
	{
		fclose(fp);
		return -1;
	}

	memset(file_buffer, 0, file_len);

	readBytes = fread(file_buffer, sizeof(unsigned char), file_len, fp);
	fclose(fp);

	if (readBytes != file_len)
	{
		return -1;
	}

	compressFile(file_buffer, file_len, buf, compress_len);

	return 0;
}

/******************************************************************************
* FUNCTION		: getFolderEntryPath()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	:
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
int getFolderEntryPath(EFS_HEADER* efsHeader, unsigned int level,
                       unsigned int entryNum, char *parentPath)
{
	int ret = MK_PLUGIN_ERR;
	unsigned int idx = 0;
	unsigned int parentLevel = 0;
	char path[MAX_FILENAME_LEN] = {0};
	char fullPath[MAX_FILENAME_LEN] = {0};

	if (NULL == efsHeader || NULL == parentPath)
	{
		MSG_ERR("input paras invalid!\n");
		return MK_PLUGIN_ERR;
	}

	memset(path, 0, MAX_FILENAME_LEN);
	memset(fullPath, 0, MAX_FILENAME_LEN);
	if (efsHeader->efsEntry[entryNum].type == EFS_ENTRY_TYPE_FILE)
	{
		strcpy(parentPath, path);
		return MK_PLUGIN_OK;
	}

	if (level == EFS_ROOT_LEVEL)
	{
		strcpy(parentPath, path);
		return MK_PLUGIN_OK;
	}
	else
	{
		idx = efsHeader->efsEntry[entryNum].parentIndex;
		if (idx < efsHeader->entryNum)
		{
			parentLevel = efsHeader->efsEntry[idx].folderPro.level;
			ret = getFolderEntryPath(efsHeader, parentLevel, idx, path);
			if (0 != ret)
			{
				MSG_ERR("getFolderEntryPath failed\n");
				return MK_PLUGIN_ERR;
			}
		}

		if (0 != strlen(path))
		{
			snprintf(fullPath, MAX_FILENAME_LEN, "%s%s%s",
				path, "/", efsHeader->efsEntry[entryNum].name);
		}
		else
		{
			strcpy(fullPath, efsHeader->efsEntry[entryNum].name);
			MSG_LOG("fullPath is %s\n", fullPath);
		}
	}

	strcpy(parentPath, fullPath);
	MSG_LOG("parentPath is %s\n", parentPath);
	return MK_PLUGIN_OK;
}

/******************************************************************************
* FUNCTION		: getFileEntryPath()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: get file's full path
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
int getFileEntryPath(EFS_HEADER* efsHeader, unsigned int entryNum, char *filePath)
{
	int ret = MK_PLUGIN_ERR;
	unsigned int i = 0;
	unsigned int parentIndex = 0;
	unsigned int parentLevel = 0;
	char parentPath[MAX_FILENAME_LEN] = {0};
	char fullPath[MAX_FILENAME_LEN] = {0};

	if (NULL == efsHeader || NULL == filePath)
	{
		MSG_ERR("Invalid paras: NULL == efsHeader || NULL == filePath!\n");
		return MK_PLUGIN_ERR;
	}

	memset(fullPath, 0, MAX_FILENAME_LEN);
	memset(parentPath, 0, MAX_FILENAME_LEN);
	if (entryNum >= efsHeader->entryNum)
	{
		strcpy(filePath, parentPath);
		return MK_PLUGIN_OK;
	}

	if (efsHeader->efsEntry[entryNum].type == EFS_ENTRY_TYPE_FILE)
	{
		parentIndex = efsHeader->efsEntry[entryNum].parentIndex;
		if (parentIndex < efsHeader->entryNum)
		{
			parentLevel = efsHeader->efsEntry[parentIndex].folderPro.level;
			ret = getFolderEntryPath(efsHeader, parentLevel, parentIndex, parentPath);
			if (0 != ret)
			{
				MSG_ERR("getFolderEntryPath failed!\n");
				return MK_PLUGIN_ERR;
			}
			MSG_LOG("parentIndex is %d, level is %d, parentPath is %s\n",
				parentIndex, parentLevel, parentPath);
		}

		if (0 != strlen(parentPath))
		{
			snprintf(fullPath, MAX_FILENAME_LEN, "%s%s%s",
				parentPath,
				"/",
				efsHeader->efsEntry[entryNum].name);
		}
		else
		{
			strcpy(fullPath, efsHeader->efsEntry[entryNum].name);
		}
	}

	strcpy(filePath, fullPath);
	return MK_PLUGIN_OK;
}

/******************************************************************************
* FUNCTION		: writeEfsEntriesToBuf()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: according to dir tree msg, read file data to buf
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
int writeEfsEntriesToBuf(unsigned char *buf, char *filePathPrefix, int *compressLen)
{
	int ret = 0;
	unsigned int i = 0;
	unsigned char* bufStart = NULL;
	char filePath[MAX_FILENAME_LEN]= {0};
	char fullPath[MAX_FILENAME_LEN] = {0};
	EFS_HEADER tmpHeader;
	uLong cLen = 0;

	if (NULL == buf || NULL == filePathPrefix)
	{
		MSG_ERR("buf or srcDirPath is NULL!\n");
		return MK_PLUGIN_ERR;
	}

	memset(filePath, 0, MAX_FILENAME_LEN);
	memset(fullPath, 0, MAX_FILENAME_LEN);

	bufStart = buf;
	bufStart += sizeof(EFS_HEADER);

	for (i = 0; i < gEfsHeader.entryNum; i++)
	{
		if (gEfsHeader.efsEntry[i].type == EFS_ENTRY_TYPE_FILE) //file
		{
			/* ȡļ·(Ŀ¼У׽ڵ洢) */
			ret = getFileEntryPath(&gEfsHeader, i, filePath);
			if (0 != ret)
			{
				MSG_ERR("getFileEntryPath failed!\n");
				return ret;
			}
			snprintf(fullPath, MAX_FILENAME_LEN, "%s%s", filePathPrefix, filePath);

			/* ļѹд뵽 */
			ret = compressFilesToBuf(fullPath, bufStart, &cLen);

			if (0 != ret)
			{
				MSG_ERR("write file data to buf failed!\n");
				return ret;
			}
			gEfsHeader.efsEntry[i].filePro.length = (unsigned int)cLen;

			/* ƶ洢ָ */
			bufStart += gEfsHeader.efsEntry[i].filePro.length;
		}
		else
		{
			getFolderEntryPath(&gEfsHeader, gEfsHeader.efsEntry[i].folderPro.level, i, filePath);
			strncpy(fullPath, filePath, MAX_FILENAME_LEN);
		}
	}

	memset(&tmpHeader, 0, sizeof(EFS_HEADER));
	memcpy(&tmpHeader, &gEfsHeader, sizeof(EFS_HEADER));
	for (i = 0; i < tmpHeader.entryNum; i++)
	{
		if (tmpHeader.efsEntry[i].type == EFS_ENTRY_TYPE_FILE)
		{
			tmpHeader.efsEntry[i].parentIndex = htonl(gEfsHeader.efsEntry[i].parentIndex);
			tmpHeader.efsEntry[i].filePro.length = htonl(gEfsHeader.efsEntry[i].filePro.length);
			tmpHeader.efsEntry[i].filePro.start = htonl(gEfsHeader.efsEntry[i].filePro.start);
		}
		else
		{
			tmpHeader.efsEntry[i].parentIndex = htonl(gEfsHeader.efsEntry[i].parentIndex);
			tmpHeader.efsEntry[i].folderPro.level = htonl(gEfsHeader.efsEntry[i].folderPro.level);
		}
	}
	tmpHeader.entryNum = htonl(gEfsHeader.entryNum);

	/* efsHeaderд뵽 */
	memcpy(buf, (unsigned char*)&tmpHeader, sizeof(EFS_HEADER));

	*compressLen = bufStart - buf - sizeof(EFS_HEADER);

	return 0;
}

/******************************************************************************
* FUNCTION		: mkEfsEntry()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: create directory tree message entries
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
int mkEfsEntry(char* container, unsigned int *imgFileDataSize)
{
	int ret = 0;
	unsigned int i = 0;
	unsigned int num = 0;
	unsigned int level = 0;
	unsigned int levelNum = 0;
	unsigned int parentIndex = 0;
	char fullPath[MAX_FILENAME_LEN] = {0};
	char *currentFolder = NULL;
	struct dirent *ptr = NULL;
	struct stat info;
	DIR *dir = NULL;

	if (NULL == container)
	{
		MSG_ERR("container is NULL!\n");
		return MK_PLUGIN_ERR;
	}

	dir = opendir(container);
	if (NULL == dir)
	{
		MSG_ERR("open dir failed!\n");
		return MK_PLUGIN_ERR;
	}

	currentFolder = container;
	if (!strcmp(currentFolder, gDirPath))
	{
		level = EFS_ROOT_LEVEL;
		parentIndex = MAX_EFS_ENTRY_NUM; /* 512ڱʶڵ㣬ϰ߲-1ش */
	}
	else
	{
		for(i = 0; i < gEfsFolderLevel.num; i++)
		{
			if(!strcmp(currentFolder, gEfsFolderLevel.entryInfo[i].path))
			{
				level = gEfsFolderLevel.entryInfo[i].level;
				parentIndex = gEfsFolderLevel.entryInfo[i].indexInImage;
				break;
			}
		}

		if (i == gEfsFolderLevel.num)
		{
			MSG_ERR("Not found parent path!\n");
			closedir(dir);
			return MK_PLUGIN_ERR;
		}
	}

	while (ptr = readdir(dir))
	{
		num = gEfsHeader.entryNum;
		if (num > (MAX_EFS_ENTRY_NUM - 1))
		{
			MSG_ERR("too many entry!\n");
			closedir(dir);
			return MK_PLUGIN_ERR;
		}

		if (!strcmp(".", ptr->d_name) || !strcmp("..", ptr->d_name))
		{
			continue;
		}

		snprintf(fullPath, MAX_FILENAME_LEN, "%s%s", container, ptr->d_name);
		ret = stat(fullPath, &info);
		if(0 != ret)
		{
			MSG_ERR("stat %s error!\n", fullPath);
			closedir(dir);
			return MK_PLUGIN_ERR;
		}

		if(S_ISDIR(info.st_mode))
		{
			snprintf(fullPath, MAX_FILENAME_LEN, "%s%s", fullPath, "/");
			gEfsHeader.efsEntry[num].type = EFS_ENTRY_TYPE_FOLDER;
			/* numڵϢ */
			sprintf(gEfsHeader.efsEntry[num].name, "%s", ptr->d_name);
			gEfsHeader.efsEntry[num].folderPro.level = level + 1;
			gEfsHeader.efsEntry[num].parentIndex = parentIndex;

			/* Ŀ¼ṹУlevelnumļϢ */
			levelNum = gEfsFolderLevel.num;
			strncpy(gEfsFolderLevel.entryInfo[levelNum].path, fullPath, MAX_FILENAME_LEN);
			gEfsFolderLevel.entryInfo[levelNum].level = level + 1;
			/* Ŀ¼ڵĿ¼е */
			gEfsFolderLevel.entryInfo[levelNum].indexInImage = num;
			gEfsFolderLevel.num++;
			gEfsHeader.entryNum++;

			ret = mkEfsEntry(fullPath, imgFileDataSize);
			if (ret != 0)
			{
				return ret;
			}
		}
		else if (S_ISREG(info.st_mode))
		{
			/* numڵϢ */
			gEfsHeader.efsEntry[num].type = EFS_ENTRY_TYPE_FILE;
			sprintf(gEfsHeader.efsEntry[num].name, "%s", ptr->d_name);

			gEfsHeader.efsEntry[num].parentIndex = parentIndex;
			gEfsHeader.efsEntry[num].filePro.start = *imgFileDataSize;
			gEfsHeader.efsEntry[num].filePro.length = compressBound(info.st_size) + 2 * sizeof(uint32_t);
			/* imgFileDataSizeֻļƫƣ൱ļĴС */
			*imgFileDataSize += gEfsHeader.efsEntry[num].filePro.length;
			gEfsHeader.entryNum++;
		}
	}

	closedir(dir);
	return 0;
}

/******************************************************************************
* FUNCTION		: packPluginIpk()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: pack uc file
* INPUT			: @srcDirPath:directory path of ipk files
* OUTPUT		:
* RETURN		:
******************************************************************************/
unsigned char *packPluginIpk(char *srcDirPath, int *srcCompressBufLen)
{
	int ret = 0;
	int imgHeaderLen = 0;
	int imgBufLen = 0;
	unsigned int imgFileDataSize = 0;
	unsigned int compressLen = 0;
	unsigned char *imgStart = NULL;
	unsigned char *imgBufPtr = NULL;

	if (NULL == srcDirPath || NULL == srcCompressBufLen)
	{
		MSG_ERR("srcDirPath or srcCompressBufLen is NULL!\n");
		return NULL;
	}

	imgHeaderLen = sizeof(HEADER_TAG);
	memset((char*)&gImgHeader, 0, imgHeaderLen);

	/* efs img header. */
	memset((char*)&gEfsHeader, 0, sizeof(EFS_HEADER));

	/* ʼļвΪ0 */
	gEfsFolderLevel.num = 0;

	/* make efs entry info. Ŀ¼ṹϢ */
	ret = mkEfsEntry(srcDirPath, &imgFileDataSize);
	if (ret != 0)
	{
		MSG_ERR("create dir tree failed!\n");
		return NULL;
	}

	imgBufLen = imgHeaderLen;
	imgBufLen +=  imgFileDataSize + sizeof(EFS_HEADER);

	/* ׼뾵ļ */
	imgBufPtr = (unsigned char*)malloc(imgBufLen);
	if (NULL == imgBufPtr)
	{
		MSG_ERR("malloc imgBufPtr failed!\n");
		return NULL;
	}
	memset(imgBufPtr, 0, imgBufLen);

	imgStart = imgBufPtr + sizeof(gImgHeader);

	/* Ŀ¼ļϢд뵽 */
	ret = writeEfsEntriesToBuf(imgStart, srcDirPath, &compressLen);
	if (ret != 0)
	{
		MSG_ERR("write EFS ERROR!\n");
		FREE_P(imgBufPtr);
		return NULL;
	}

	/* set efs file info */
	gImgHeader.efsFiles.type = 0;
	gImgHeader.efsFiles.start = imgHeaderLen;
	gImgHeader.efsFiles.length = compressLen + sizeof(EFS_HEADER);

	/* ĳ */
	imgBufLen = imgHeaderLen + gImgHeader.efsFiles.length;
	gImgHeader.imgLen = imgBufLen;

	/* ȽͷϢд뵽 */
	gImgHeader.imgLen = htonl(gImgHeader.imgLen);
	memcpy(imgBufPtr, (unsigned char*)&gImgHeader, sizeof(gImgHeader));

	*srcCompressBufLen = imgBufLen;

	return imgBufPtr;
}

/******************************************************************************
* FUNCTION		: writeBufToFile()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: write buf data to file
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
int writeBufToFile(char* filePath, unsigned char* buf, int len)
{
	FILE* fp = NULL;
	int writeBytes = 0;

	if (NULL == filePath || NULL == buf)
	{
		MSG_ERR("Invalid paras:NULL == filePath || NULL == buf\n");
		return -1;
	}

	fp = fopen(filePath, "wb");
	if(!fp)
	{
		MSG_ERR("open %s failed!\n", filePath);
		return -1;
	}

	writeBytes = fwrite(buf, sizeof(unsigned char), len, fp);
	fclose(fp);

	if (writeBytes != len)
	{
		MSG_ERR("write files failed! len = %d\n", len);
		return -1;
	}

	return 0;
}

