/***********************************************************
*
* Copyright (C), 2010-2015, TP-LINK TECHNOLOGIES CO., LTD.
* All rights reserved.
*
* File name  : pack-plugin.c
* Version	 : 1.0
* Description: pack plugin resources
*
* Author	 : Wang FuYu <wangfuyu@tp-link.net>
* Create Date: 2015-01-23
*
* History	 :
*------------------------------------------------------------
*
*01, 23Jan15, Wang FuYu, create file.
************************************************************/

#include "pack-plugin.h"

PLUGIN_PARTS_HAEDER gPluginHeader;
char gPluginPartsPath[PLUGIN_PARTS_MAX][MAX_FILENAME_LEN] = {0};
char gPluginBinFile[MAX_FILENAME_LEN] = {0};
UINT8 gPluginMD5Salt[MAX_MD5_LEN + 1] = {0xCC, 0x96, 0x28, 0xEE, 0x8D, 0xFB, 0x21, 0xBB, \
						 				0x3D, 0xEF, 0x6C, 0xB5, 0x9F, 0x77, 0x4C, 0x7C};

/******************************************************************************
* FUNCTION		: showUsage()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: show the usage of this plugin.
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
static void showUsage(char* name)
{
	SHOW_USAGE("%s -i <plugin_id> -v <version> -n <name> -s <size> -t <tag> \
				-c <components> -a <author> -o <os> -b <bin_file> -x <webicon> \
				-y <appzip> -z <appicon> -p <plugin_file>\n", name);
	SHOW_USAGE("\t-i <plugin_id>\t plugin's id.\n");
	SHOW_USAGE("\t-v <version>\t plugin's version.\n");
	SHOW_USAGE("\t-n <name>\t plugin's name.\n");
	SHOW_USAGE("\t-s <size>\t size.\n");
	SHOW_USAGE("\t-t <tag>\t tag.\n");
	SHOW_USAGE("\t-c <components>\t packages.\n");
	SHOW_USAGE("\t-a <author>\t author.\n");
	SHOW_USAGE("\t-o <os>\t OS version.\n");
	SHOW_USAGE("\t-d <data>\t ipks file.\n");
	SHOW_USAGE("\t-x <webicon>\t web icon file.\n");
	SHOW_USAGE("\t-y <appzip>\t app icon file.\n");
	SHOW_USAGE("\t-z <appicon>\t app webzip file.\n");
	SHOW_USAGE("\t-p <plugin_bin>\t plugin bin for OA.\n");
}

/************************************************************
 Function   : getFilename
 Author		: Wang FuYu <wangfuyu@tp-link.net>
 Description: get Filename from URL
 Input      : url
 Output     : filename
 Return     : pointer of filename
************************************************************/
char *getFilename(char *url)
{
	int idx = 0;
	int len = 0;
	char tmp = '/';
	char *pageFilename = NULL;

	if (!url)
	{
		return NULL;
	}

	len = strlen(url);
	for (idx = len; idx >= 0; idx--)
	{
		tmp = *(url + idx);
		if ('/' == tmp)
		{
			break;
		}
	}

	if ((0 == idx) || ((len - 1) == idx))
	{
		PLUGIN_LOG("Get filename failed because URL is invalid!\n");
	}
	else
	{
		pageFilename = url + idx + 1;
	}

	return pageFilename;
}

/******************************************************************************
* FUNCTION		: writePartsToBuf()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: read file data to buf
* INPUT			: @filePath:full path of file
				  @buf:buffer for saving file data
* OUTPUT		:
* RETURN		: file's size
******************************************************************************/
int writePartsToBuf(char* filePath, unsigned char* buf)
{
	FILE* fp = NULL;
	int fileSize = 0;
	int readBytes = 0;

	if (NULL == filePath || NULL == buf)
	{
		PLUGIN_LOG("input paras invalid!\n");
		return -1;
	}

	fp = fopen(filePath, "rb");
	if(!fp)
	{
		return -1;
	}

	fseek(fp, 0L, SEEK_END);
	fileSize = ftell(fp);
	fseek(fp, 0L, SEEK_SET);

	readBytes = fread(buf, sizeof(unsigned char), fileSize, fp);
	fclose(fp);

	PLUGIN_LOG("file path is %s, read bytes is %d, flie size is %d.\n", filePath, readBytes, fileSize);
	if (readBytes != fileSize)
	{
		return -1;
	}

	return fileSize;
}

/******************************************************************************
* FUNCTION		: writeBufToPluginFile()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: write buf data to file
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
int writeBufToPluginFile(char* filePath, unsigned char* buf, int len)
{
	FILE* fp = NULL;
	int writeBytes = 0;

	if (NULL == filePath || NULL == buf)
	{
		PLUGIN_LOG("Invalid paras:NULL == filePath || NULL == buf\n");
		return -1;
	}

	fp = fopen(filePath, "wb");
	if(!fp)
	{
		PLUGIN_LOG("open %s failed!\n", filePath);
		return -1;
	}

	writeBytes = fwrite(buf, sizeof(unsigned char), len, fp);
	fclose(fp);

	if (writeBytes != len)
	{
		PLUGIN_LOG("write files failed! len = %d\n", len);
		return -1;
	}

	return 0;
}

/******************************************************************************
* FUNCTION		: packPlugin()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	:
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
static int packPlugin(void)
{
	int i = 0;
	int ret = 0;
	int pluginSize = 0;
	int partsSize = 0;
	char *pluginPartsName = NULL;
	char *buf = NULL;
	char md5Digest[MAX_MD5_LEN + 1] = {0};

	if (NULL == (buf = (char*)calloc(1, PLUGIN_FILE_MAX_SIZE)))
	{
		return -1;
	}
	memset(buf, 0, PLUGIN_FILE_MAX_SIZE);
	pluginSize = sizeof(PLUGIN_PARTS_HAEDER);
	for (i = 0; i < PLUGIN_PARTS_MAX; i++)
	{
		pluginPartsName = getFilename(gPluginPartsPath[i]);
		if (NULL == pluginPartsName)
		{
			free(buf);
			return -1;
		}

		PLUGIN_LOG("plugin parts name is %s.\n", pluginPartsName);
		memcpy(gPluginHeader.pluginFile[i].fileName, pluginPartsName, MAX_PARTS_FILENAME_LEN);
		gPluginHeader.pluginFile[i].fileOffset = htonl(pluginSize);

		partsSize = writePartsToBuf(gPluginPartsPath[i], buf + pluginSize);
		if (0 > partsSize)
		{
			PLUGIN_LOG("write parts to buf failed.\n");
			free(buf);
			return -1;
		}

		gPluginHeader.pluginFile[i].fileSize = htonl(partsSize);
		pluginSize = pluginSize + partsSize;
	}

	memcpy(buf, &gPluginHeader, sizeof(PLUGIN_PARTS_HAEDER));
	memset(md5Digest, 0, MAX_MD5_LEN + 1);
	md5_make_digest(md5Digest, buf, pluginSize);

#if PLUGIN_DEBUG
	PLUGIN_LOG("md5:\n");
	for (i = 0; i < 16; i++)
	{
		PLUGIN_LOG("%02x.", (unsigned char)gPluginHeader.md5[i]);
	}
	PLUGIN_LOG("\n");

	PLUGIN_LOG("md5:\n");
	for (i = 0; i < 16; i++)
	{
		PLUGIN_LOG("%02x.", (unsigned char)md5Digest[i]);
	}
	PLUGIN_LOG("\n");
#endif

	memcpy(&gPluginHeader.md5, md5Digest, MAX_MD5_LEN);
	memcpy(buf, &gPluginHeader, sizeof(PLUGIN_PARTS_HAEDER));

	PLUGIN_LOG("plugin size is %d, salt is %08x, md5 is %08x\n", pluginSize, gPluginMD5Salt, md5Digest);
	ret = writeBufToPluginFile(gPluginBinFile, buf, pluginSize);
	if (0 != ret)
	{
		PLUGIN_LOG("write buf to plugin file failed.\n");
		free(buf);
		return -1;
	}

	free(buf);
	return 0;
}

/******************************************************************************
* FUNCTION		: initPluginParas()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	:
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
static int initPluginParas(int argc, char** argv)
{
	int ch = 0;
	extern char *optarg;

	if (argc < MIN_ARGC_NUM)
	{
		showUsage(argv[0]);
		return -1;
	}

	memset(gPluginPartsPath, 0, sizeof(gPluginPartsPath));
	memset(gPluginBinFile, 0, sizeof(gPluginBinFile));
	memset(&gPluginHeader, 0, sizeof(PLUGIN_PARTS_HAEDER));
	memcpy(gPluginHeader.md5, gPluginMD5Salt, MAX_MD5_LEN);
	gPluginHeader.headerVer = htonl(PLUGIN_TOOLS_VERSION);
	gPluginHeader.pluginType = htonl(PLUGIN_TYPE);

	while (-1 != (ch = getopt(argc, argv, "i:v:n:s:t:c:a:o:d:x:y:z:p:")))
	{
		switch (ch)
		{
		case 'i':
			PLUGIN_LOG("plugin_id: %s\n", optarg);
			memcpy(gPluginHeader.pluginId, optarg, sizeof(gPluginHeader.pluginId));
			break;
		case 'v':
			PLUGIN_LOG("plugin_version: %s\n", optarg);
			memcpy(gPluginHeader.pluginVer, optarg, sizeof(gPluginHeader.pluginVer));
			break;
		case 'n':
			PLUGIN_LOG("plugin_name: %s\n", optarg);
			memcpy(gPluginHeader.pluginName, optarg, sizeof(gPluginHeader.pluginName));
			break;
		case 's':
			PLUGIN_LOG("plugin_size: %s\n", optarg);
			gPluginHeader.pluginSize= htonl(atoi(optarg));
			break;
		case 't':
			PLUGIN_LOG("plugin_tag: %s\n", optarg);
			memcpy(gPluginHeader.pluginTag, optarg, sizeof(gPluginHeader.pluginTag));
			break;
		case 'c':
			PLUGIN_LOG("plugin_components: %s\n", optarg);
			memcpy(gPluginHeader.pluginComp, optarg, sizeof(gPluginHeader.pluginComp));
			break;
		case 'a':
			PLUGIN_LOG("plugin_author: %s\n", optarg);
			memcpy(gPluginHeader.pluginAuthor, optarg, sizeof(gPluginHeader.pluginAuthor));
			break;
		case 'o':
			PLUGIN_LOG("plugin_os: %s\n", optarg);
			memcpy(gPluginHeader.osVer, optarg, sizeof(gPluginHeader.osVer));
			break;
		case 'd':
			PLUGIN_LOG("plugin_data: %s\n", optarg);
			memcpy(gPluginPartsPath[PLUGIN_BIN_FILE], optarg, MAX_FILENAME_LEN);
			break;
		case 'x':
			PLUGIN_LOG("plugin_webicon: %s\n", optarg);
			memcpy(gPluginPartsPath[PLUGIN_WEB_ICON], optarg, MAX_FILENAME_LEN);
			break;
		case 'y':
			PLUGIN_LOG("plugin_appicon: %s\n", optarg);
			memcpy(gPluginPartsPath[PLUGIN_APP_ICON], optarg, MAX_FILENAME_LEN);
			break;
		case 'z':
			PLUGIN_LOG("plugin_appzip: %s\n", optarg);
			memcpy(gPluginPartsPath[PLUGIN_APP_WEB], optarg, MAX_FILENAME_LEN);
			break;
		case 'p':
			PLUGIN_LOG("plugin_bin: %s\n", optarg);
			memcpy(gPluginBinFile, optarg, MAX_FILENAME_LEN);
			break;
		default:
			showUsage(argv[0]);
			return -1;
		}
	}

	PLUGIN_LOG("gPluginHeader, id: %s, ver: %s, name: %s, size: %d, tag: %s, comp: %s, \
				author: %s, os: %s, data: %s, webicon: %s, appicon: %s, appzip: %s, plugin: %s\n",
				gPluginHeader.pluginId, gPluginHeader.pluginVer, gPluginHeader.pluginName,
				gPluginHeader.pluginSize, gPluginHeader.pluginTag, gPluginHeader.pluginComp,
				gPluginHeader.pluginAuthor, gPluginHeader.osVer, gPluginPartsPath[PLUGIN_BIN_FILE],
				gPluginPartsPath[PLUGIN_WEB_ICON], gPluginPartsPath[PLUGIN_APP_ICON],
				gPluginPartsPath[PLUGIN_APP_WEB], gPluginBinFile);
	return 0;
}

/******************************************************************************
* FUNCTION		: main()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: entry
* INPUT			:
* OUTPUT		:
* RETURN		: 0 means success, -1 means failed
******************************************************************************/
int main(int argc, char** argv)
{
	int ret = -1;

	ret = initPluginParas(argc, argv);
	if (0 != ret)
	{
		PLUGIN_LOG("parseOpt failed!\n");
		return -1;
	}

	ret = packPlugin();
	if (0 != ret)
	{
		PLUGIN_LOG("pack plugin failed!\n");
		return -1;
	}

	return 0;
}

