#include <getopt.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <inttypes.h>

#define DEBUG_OUT 1
#if DEBUG_OUT

#define ERR_PRINT(...) do { \
	fprintf(stderr, "%10s %5d:", __FUNCTION__, __LINE__); \
	fprintf(stderr, __VA_ARGS__); \
}while(0)

#else

#define ERR_PRINT(...)

#endif

#define RET_VAR_IF(VAR, COND) do { \
	if (COND) \
	{ \
		ERR_PRINT("error occurred\n"); \
		return VAR; \
	} \
}while(0)

#define H_MAGIC_NUMBER			"5A87AA52D998D0167C5D"		/* equal with tp header v2, compatible for upgrade*/
#define H_MAGIC_LEN				24
#define H_RSA_SIGN_LEN			128
#define MD5SUM_LEN				16

#define IPK_HEADER_LEN			(256)

#define H_HEAD_VERSION			(0x00000001)
#define H_PRODUCT_ID			(0x75000002)
#define H_VENDOR_ID				(0x00000001)
typedef struct _IPKG_HEADER
{
	uint32_t headerVersion;
	unsigned char magicNumber[H_MAGIC_LEN];
	uint32_t productId;
	
	uint32_t vendorId;
	uint32_t appId;
	uint32_t appVersion;
	uint32_t appReleaseDate;	/* eg: 0x20140627 对应2014年6月27日 */

	unsigned char rsaSignature[H_RSA_SIGN_LEN];	/*整个文件的签名(头部MD5和签名本身为salt)*/
#if 0
	unsigned char md5sum[MD5SUM_LEN];			/*头部MD5(MD5本身为salt，签名为有效值)*/
#endif
}IPKG_HEADER;

typedef struct _OPTIONS
{
	//char *md5_hex;
	char *sig_hex;
	int append;

	char *filename;
}OPTIONS;

OPTIONS g_options;

const unsigned char md5salt[MD5SUM_LEN] = 
{
	0xBF, 0xD8, 0x35, 0x1F, 0x65, 0x83, 0x54, 0xCC, 
	0x0B, 0x6E, 0xCF, 0x04, 0x01, 0x81, 0xAF, 0x81, 
};

const unsigned char signsalt[H_RSA_SIGN_LEN] =
{
	0x81, 0xCA, 0xFD, 0x2C, 0x55, 0x25, 0xD7, 0x64, 
	0x9B, 0x33, 0xAF, 0x9B, 0x7B, 0x41, 0x3A, 0x4F, 
	0x41, 0x26, 0xA5, 0x73, 0xBB, 0x14, 0xE1, 0x51, 
	0xF8, 0xDF, 0xCF, 0x8E, 0xC8, 0x38, 0xEC, 0x38, 
	
	0x09, 0xB4, 0x58, 0x98, 0x7B, 0xF3, 0x2B, 0xD5, 
	0x6F, 0x6B, 0x16, 0xF5, 0x4D, 0xFB, 0xBD, 0x2F, 
	0xB3, 0x00, 0xD1, 0x80, 0x8D, 0xAA, 0xAD, 0xB7, 
	0x5C, 0xE7, 0x69, 0x25, 0x34, 0x1E, 0xA6, 0x21, 
	
	0xAC, 0xCD, 0xB0, 0x49, 0x77, 0x16, 0x8A, 0x40, 
	0xEA, 0xE6, 0xA8, 0x79, 0x2E, 0x37, 0x89, 0x8E, 
	0x76, 0x7C, 0x69, 0xA6, 0xC3, 0x95, 0xD4, 0xDB, 
	0x50, 0x87, 0xBC, 0x92, 0xE7, 0xD9, 0x8A, 0x27,
	
	0xB2, 0x8C, 0x18, 0xFF, 0x69, 0xBB, 0x60, 0x0A, 
	0x21, 0x53, 0x64, 0xE0, 0xA1, 0x4B, 0x00, 0x36, 
	0x52, 0xBD, 0x01, 0x83, 0x1F, 0x4F, 0xEA, 0xC0, 
	0xEB, 0xD7, 0x1D, 0x2F, 0x16, 0xA5, 0xF5, 0xDD, 
};

static void init(void)
{
	memset(&g_options, 0, sizeof(g_options));
}

static void usage(char *progName)
{
	fprintf(stdout, "Usage: %s [OPTIONS...] filename\n", progName);
	fprintf(stdout, 
		"\n"
		"Options:\n"
		"-s sig		set signature value with sig(hex type).\n"
		//"-m md5sum	set md5 value with md5sum(hex type).\n"
		"-a 		if add a header to file.\n"
		);
}

static void init_header(IPKG_HEADER *header)
{
	memset(header, 0, sizeof(IPKG_HEADER));

	header->headerVersion = H_HEAD_VERSION;
	strncpy(header->magicNumber, H_MAGIC_NUMBER, strlen(H_MAGIC_NUMBER));
	
	header->productId = H_PRODUCT_ID;
	header->vendorId = H_VENDOR_ID;
	
	memcpy(header->rsaSignature, signsalt, sizeof(header->rsaSignature));
	//memcpy(header->md5sum, md5salt, sizeof(header->md5sum));
}

#define CONVERT_HEX_TO_BIN(HEX, BIN) do { \
	if (((HEX) >= '0') && ((HEX) <= '9')) \
	{ \
		(BIN) = (HEX) - '0'; \
	} \
	else if (((HEX) >= 'a') && ((HEX) <= 'f')) \
	{ \
		(BIN) = (HEX) - 'a' + 10; \
	} \
	else if (((HEX) >= 'A') && ((HEX) <= 'F')) \
	{ \
		(BIN) = (HEX) - 'A' + 10; \
	} \
	else \
	{ \
		ERR_PRINT("invalid character in hex: %c\n", HEX); \
		return -1; \
	} \
}while(0)
static int hex2bin(unsigned char *hex, int hexlen, unsigned char *bin, int binlen)
{
	int idx;
	int outlen;
	unsigned int high, low;

	/* hexlen必须为偶数，且binlen应不小于其一半 */
	if (  ((hexlen & 0x1) != 0)
	    ||(binlen < (hexlen >> 1)))
	{
		ERR_PRINT("hexlen: %d, binlen: %d\n", hexlen, binlen);
		return -1;
	}

	outlen = (hexlen >> 1);
	for (idx = 0; idx < outlen; idx++)
	{
		CONVERT_HEX_TO_BIN(hex[2 * idx], high);
		CONVERT_HEX_TO_BIN(hex[2 * idx + 1], low);
		bin[idx] = ((high << 4) | low);
	}

	return outlen;
}

static int write_header_to_file(FILE *fp, IPKG_HEADER *header)
{
	uint32_t tmp;
	
	tmp = htonl(header->headerVersion);
	fwrite(&tmp, sizeof(uint32_t), 1, fp);
	
	fwrite(header->magicNumber, sizeof(header->magicNumber), 1, fp);
	
	tmp = htonl(header->productId);
	fwrite(&tmp, sizeof(uint32_t), 1, fp);
	
	tmp = htonl(header->vendorId);
	fwrite(&tmp, sizeof(uint32_t), 1, fp);
	
	tmp = htonl(header->appId);
	fwrite(&tmp, sizeof(uint32_t), 1, fp);
	
	tmp = htonl(header->appVersion);
	fwrite(&tmp, sizeof(uint32_t), 1, fp);
	
	tmp = htonl(header->appReleaseDate);
	fwrite(&tmp, sizeof(uint32_t), 1, fp);
	
	//fwrite(header->md5sum, sizeof(header->md5sum), 1, fp);
	
	fwrite(header->rsaSignature, sizeof(header->rsaSignature), 1, fp);
	
	return 0;
}

static int add_header_to_file(char *filename, IPKG_HEADER *header)
{
	const int BUF_SIZE = 512;
	char buf[BUF_SIZE];
	FILE *fpnew = NULL;
	FILE *fpold = NULL;
	char *newfilename = NULL;
	int ret;
	int readlen;

	fpold = fopen(filename, "rb");
	if (NULL == fpold)
	{
		ret = -1;
		goto errout;
	}
	
	ret = sprintf(buf, "%s_new", filename);
	if (ret >= BUF_SIZE)
	{
		ret = -1;
		goto errout;
	}

	newfilename = (char *)malloc(strlen(buf) + 1);
	strcpy(newfilename, buf);
	
	fpnew = fopen(newfilename, "wb");
	if (NULL == fpnew)
	{
		ret = -1;
		goto errout;
	}
	
	memset(buf, 0, IPK_HEADER_LEN);
	fwrite(buf, IPK_HEADER_LEN, 1, fpnew);

	rewind(fpnew);
	write_header_to_file(fpnew, header);

	fseek(fpnew, IPK_HEADER_LEN, SEEK_SET);
	rewind(fpold);
	while(!feof(fpold))
	{
		readlen = fread(buf, 1, BUF_SIZE, fpold);
		fwrite(buf, 1, readlen, fpnew);
	}
	
	fclose(fpnew);
	fclose(fpold);

	unlink(filename);
	rename(newfilename, filename);
	
	free(newfilename);

	return 0;
	
errout:
	if (fpnew)
	{
		fclose(fpnew);
	}
	if (fpold)
	{
		fclose(fpold);
	}
	if (newfilename)
	{
		free(newfilename);
	}

	return ret;
}

static int replace_file_header(char *filename, IPKG_HEADER *header)
{
	FILE *fp = NULL;

	fp = fopen(filename, "rb+");
	RET_VAR_IF(-1, (NULL == fp));
	
	fseek(fp, 0, SEEK_SET);
	fwrite(header, sizeof(IPKG_HEADER), 1, fp);
	
	fclose(fp);

	return 0;
}

static int get_file_header(char *filename, IPKG_HEADER *header)
{
	FILE *fp = NULL;
	
	fp = fopen(filename, "rb+");
	RET_VAR_IF(-1, (NULL == fp));
	
	fseek(fp, 0, SEEK_SET);
	
	fread(header, sizeof(IPKG_HEADER), 1, fp);
	
	fclose(fp);

	return 0;
}

int main(int argc, char *argv[])
{
	char ch;
	int ret;
	IPKG_HEADER header;
	
	init();
	while ((ch = getopt(argc, argv, "s:a")) != -1) 
	{
		switch(ch) 
		{
		#if 0
		case 'm':
			g_options.md5_hex = optarg;
			break;
		#endif
		case 's':
			g_options.sig_hex = optarg;
			break;
		case 'a':
			g_options.append = 1;
			break;
		default:
			usage(argv[0]);
			return -1;
		}
	}
	
	if (optind != argc - 1)					/* no ipk filename or more than one */
	{
		ERR_PRINT("can just accept one and only one filename!\n");
		return -1;
	}
	
	g_options.filename = argv[optind];
	
	/* 如果不是新增文件头，那么保持原文件头中的所有不需修改的信息 */
	if (!g_options.append)
	{
		get_file_header(g_options.filename, &header);
	}
	/* 新增文件头，则对基本信息进行初始化 */
	else
	{
		init_header(&header);
	}

#if 0
	if (g_options.md5_hex)
	{
		ret = hex2bin(g_options.md5_hex, strlen(g_options.md5_hex), header.md5sum, sizeof(header.md5sum));
		if (ret < 0)
		{
			ERR_PRINT("parse md5 string failed!\n");
			return -1;
		}
	}
#endif
	
	if (g_options.sig_hex)
	{
		ret = hex2bin(g_options.sig_hex, strlen(g_options.sig_hex), header.rsaSignature, sizeof(header.rsaSignature));
		if (ret < 0)
		{
			ERR_PRINT("parse sigature string failed!\n");
			return -1;
		}
	}
	
	if (g_options.append)
	{
		ret = add_header_to_file(g_options.filename, &header);
	}
	else
	{
		ret = replace_file_header(g_options.filename, &header);
	}
	
	if (ret < 0)
	{
		ERR_PRINT("add/replace file header failed!\n");
		return -1;
	}
	
	return 0;
}
