#ifndef __JZ_DMAC_H__
#define __JZ_DMAC_H__

/*************************************************************************
 * DMAC (DMA Controller)
 *************************************************************************/
#define	DMAC_BASE	(0XB3420000)
#define MAX_DMA_NUM	6  /* max 6 channels */

#define DMAC_DSAR(n)	(DMAC_BASE + (0x00 + (n) * 0x20)) /* DMA source address */
#define DMAC_DTAR(n)	(DMAC_BASE + (0x04 + (n) * 0x20)) /* DMA target address */
#define DMAC_DTCR(n)	(DMAC_BASE + (0x08 + (n) * 0x20)) /* DMA transfer count */
#define DMAC_DRSR(n)	(DMAC_BASE + (0x0c + (n) * 0x20)) /* DMA request source */
#define DMAC_DCCSR(n)	(DMAC_BASE + (0x10 + (n) * 0x20)) /* DMA control/status */
#define DMAC_DCMD(n)	(DMAC_BASE + (0x14 + (n) * 0x20)) /* DMA command */
#define DMAC_DDA(n)	(DMAC_BASE + (0x18 + (n) * 0x20)) /* DMA descriptor address */
#define DMAC_DMACR	(DMAC_BASE + 0x1000)              /* DMA control register */
#define DMAC_DMAIPR	(DMAC_BASE + 0x0304)              /* DMA interrupt pending */
#define DMAC_DMADBR	(DMAC_BASE + 0x0308)              /* DMA doorbell */
#define DMAC_DMADBSR	(DMAC_BASE + 0x030C)              /* DMA doorbell set */

// channel 0
#define DMAC_DSAR0      DMAC_DSAR(0)
#define DMAC_DTAR0      DMAC_DTAR(0)
#define DMAC_DTCR0      DMAC_DTCR(0)
#define DMAC_DRSR0      DMAC_DRSR(0)
#define DMAC_DCCSR0     DMAC_DCCSR(0)
#define DMAC_DCMD0	DMAC_DCMD(0)
#define DMAC_DDA0	DMAC_DDA(0)

// channel 1
#define DMAC_DSAR1      DMAC_DSAR(1)
#define DMAC_DTAR1      DMAC_DTAR(1)
#define DMAC_DTCR1      DMAC_DTCR(1)
#define DMAC_DRSR1      DMAC_DRSR(1)
#define DMAC_DCCSR1     DMAC_DCCSR(1)
#define DMAC_DCMD1	DMAC_DCMD(1)
#define DMAC_DDA1	DMAC_DDA(1)

// channel 2
#define DMAC_DSAR2      DMAC_DSAR(2)
#define DMAC_DTAR2      DMAC_DTAR(2)
#define DMAC_DTCR2      DMAC_DTCR(2)
#define DMAC_DRSR2      DMAC_DRSR(2)
#define DMAC_DCCSR2     DMAC_DCCSR(2)
#define DMAC_DCMD2	DMAC_DCMD(2)
#define DMAC_DDA2	DMAC_DDA(2)

// channel 3
#define DMAC_DSAR3      DMAC_DSAR(3)
#define DMAC_DTAR3      DMAC_DTAR(3)
#define DMAC_DTCR3      DMAC_DTCR(3)
#define DMAC_DRSR3      DMAC_DRSR(3)
#define DMAC_DCCSR3     DMAC_DCCSR(3)
#define DMAC_DCMD3	DMAC_DCMD(3)
#define DMAC_DDA3	DMAC_DDA(3)

// channel 4
#define DMAC_DSAR4      DMAC_DSAR(4)
#define DMAC_DTAR4      DMAC_DTAR(4)
#define DMAC_DTCR4      DMAC_DTCR(4)
#define DMAC_DRSR4      DMAC_DRSR(4)
#define DMAC_DCCSR4     DMAC_DCCSR(4)
#define DMAC_DCMD4	DMAC_DCMD(4)
#define DMAC_DDA4	DMAC_DDA(4)

// channel 5
#define DMAC_DSAR5      DMAC_DSAR(5)
#define DMAC_DTAR5      DMAC_DTAR(5)
#define DMAC_DTCR5      DMAC_DTCR(5)
#define DMAC_DRSR5      DMAC_DRSR(5)
#define DMAC_DCCSR5     DMAC_DCCSR(5)
#define DMAC_DCMD5	DMAC_DCMD(5)
#define DMAC_DDA5	DMAC_DDA(5)

#define REG_DMAC_DSAR(n)	REG32(DMAC_DSAR((n)))
#define REG_DMAC_DTAR(n)	REG32(DMAC_DTAR((n)))
#define REG_DMAC_DTCR(n)	REG32(DMAC_DTCR((n)))
#define REG_DMAC_DRSR(n)	REG32(DMAC_DRSR((n)))
#define REG_DMAC_DCCSR(n)	REG32(DMAC_DCCSR((n)))
#define REG_DMAC_DCMD(n)	REG32(DMAC_DCMD((n)))
#define REG_DMAC_DDA(n)		REG32(DMAC_DDA((n)))
#define REG_DMAC_DMACR		REG32(DMAC_DMACR)
#define REG_DMAC_DMAIPR		REG32(DMAC_DMAIPR)
#define REG_DMAC_DMADBR		REG32(DMAC_DMADBR)
#define REG_DMAC_DMADBSR	REG32(DMAC_DMADBSR)

// DMA request source register
#define DMAC_DRSR_RS_BIT	0
#define DMAC_DRSR_RS_MASK	(0x1f << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_AUTO	(8 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_UART0OUT	(20 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_UART0IN	(21 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_SSIOUT	(22 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_SSIIN	(23 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_AICOUT	(24 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_AICIN	(25 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_MSCOUT	(26 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_MSCIN	(27 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_TCU	(28 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_SADC	(29 << DMAC_DRSR_RS_BIT)
  #define DMAC_DRSR_RS_SLCD	(30 << DMAC_DRSR_RS_BIT)

// DMA channel control/status register
#define DMAC_DCCSR_NDES		(1 << 31) /* descriptor (0) or not (1) ? */
#define DMAC_DCCSR_CDOA_BIT	16        /* copy of DMA offset address */
#define DMAC_DCCSR_CDOA_MASK	(0xff << DMAC_DCCSR_CDOA_BIT)
#define DMAC_DCCSR_INV		(1 << 6)  /* descriptor invalid */
#define DMAC_DCCSR_AR		(1 << 4)  /* address error */
#define DMAC_DCCSR_TT		(1 << 3)  /* transfer terminated */
#define DMAC_DCCSR_HLT		(1 << 2)  /* DMA halted */
#define DMAC_DCCSR_CT		(1 << 1)  /* count terminated */
#define DMAC_DCCSR_EN		(1 << 0)  /* channel enable bit */

// DMA channel command register
#define DMAC_DCMD_SAI		(1 << 23) /* source address increment */
#define DMAC_DCMD_DAI		(1 << 22) /* dest address increment */
#define DMAC_DCMD_RDIL_BIT	16        /* request detection interval length */
#define DMAC_DCMD_RDIL_MASK	(0x0f << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_IGN	(0 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_2	(1 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_4	(2 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_8	(3 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_12	(4 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_16	(5 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_20	(6 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_24	(7 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_28	(8 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_32	(9 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_48	(10 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_60	(11 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_64	(12 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_124	(13 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_128	(14 << DMAC_DCMD_RDIL_BIT)
  #define DMAC_DCMD_RDIL_200	(15 << DMAC_DCMD_RDIL_BIT)
#define DMAC_DCMD_SWDH_BIT	14  /* source port width */
#define DMAC_DCMD_SWDH_MASK	(0x03 << DMAC_DCMD_SWDH_BIT)
  #define DMAC_DCMD_SWDH_32	(0 << DMAC_DCMD_SWDH_BIT)
  #define DMAC_DCMD_SWDH_8	(1 << DMAC_DCMD_SWDH_BIT)
  #define DMAC_DCMD_SWDH_16	(2 << DMAC_DCMD_SWDH_BIT)
#define DMAC_DCMD_DWDH_BIT	12  /* dest port width */
#define DMAC_DCMD_DWDH_MASK	(0x03 << DMAC_DCMD_DWDH_BIT)
  #define DMAC_DCMD_DWDH_32	(0 << DMAC_DCMD_DWDH_BIT)
  #define DMAC_DCMD_DWDH_8	(1 << DMAC_DCMD_DWDH_BIT)
  #define DMAC_DCMD_DWDH_16	(2 << DMAC_DCMD_DWDH_BIT)
#define DMAC_DCMD_DS_BIT	8  /* transfer data size of a data unit */
#define DMAC_DCMD_DS_MASK	(0x07 << DMAC_DCMD_DS_BIT)
  #define DMAC_DCMD_DS_32BIT	(0 << DMAC_DCMD_DS_BIT)
  #define DMAC_DCMD_DS_8BIT	(1 << DMAC_DCMD_DS_BIT)
  #define DMAC_DCMD_DS_16BIT	(2 << DMAC_DCMD_DS_BIT)
  #define DMAC_DCMD_DS_16BYTE	(3 << DMAC_DCMD_DS_BIT)
  #define DMAC_DCMD_DS_32BYTE	(4 << DMAC_DCMD_DS_BIT)
#define DMAC_DCMD_TM		(1 << 7)  /* transfer mode: 0-single 1-block */
#define DMAC_DCMD_DES_V		(1 << 4)  /* descriptor valid flag */
#define DMAC_DCMD_DES_VM	(1 << 3)  /* descriptor valid mask: 1:support V-bit */
#define DMAC_DCMD_DES_VIE	(1 << 2)  /* DMA valid error interrupt enable */
#define DMAC_DCMD_TIE		(1 << 1)  /* DMA transfer interrupt enable */
#define DMAC_DCMD_LINK		(1 << 0)  /* descriptor link enable */

// DMA descriptor address register
#define DMAC_DDA_BASE_BIT	12  /* descriptor base address */
#define DMAC_DDA_BASE_MASK	(0x0fffff << DMAC_DDA_BASE_BIT)
#define DMAC_DDA_OFFSET_BIT	4  /* descriptor offset address */
#define DMAC_DDA_OFFSET_MASK	(0x0ff << DMAC_DDA_OFFSET_BIT)

// DMA control register
#define DMAC_DMACR_PR_BIT	8  /* channel priority mode */
#define DMAC_DMACR_PR_MASK	(0x03 << DMAC_DMACR_PR_BIT)
  #define DMAC_DMACR_PR_012345	(0 << DMAC_DMACR_PR_BIT)
  #define DMAC_DMACR_PR_023145	(1 << DMAC_DMACR_PR_BIT)
  #define DMAC_DMACR_PR_201345	(2 << DMAC_DMACR_PR_BIT)
  #define DMAC_DMACR_PR_RR	(3 << DMAC_DMACR_PR_BIT) /* round robin */
#define DMAC_DMACR_HLT		(1 << 3)  /* DMA halt flag */
#define DMAC_DMACR_AR		(1 << 2)  /* address error flag */
#define DMAC_DMACR_DMAE		(1 << 0)  /* DMA enable bit */

// DMA doorbell register
#define DMAC_DMADBR_DB5		(1 << 5)  /* doorbell for channel 5 */
#define DMAC_DMADBR_DB4		(1 << 5)  /* doorbell for channel 4 */
#define DMAC_DMADBR_DB3		(1 << 5)  /* doorbell for channel 3 */
#define DMAC_DMADBR_DB2		(1 << 5)  /* doorbell for channel 2 */
#define DMAC_DMADBR_DB1		(1 << 5)  /* doorbell for channel 1 */
#define DMAC_DMADBR_DB0		(1 << 5)  /* doorbell for channel 0 */

// DMA doorbell set register
#define DMAC_DMADBSR_DBS5	(1 << 5)  /* enable doorbell for channel 5 */
#define DMAC_DMADBSR_DBS4	(1 << 5)  /* enable doorbell for channel 4 */
#define DMAC_DMADBSR_DBS3	(1 << 5)  /* enable doorbell for channel 3 */
#define DMAC_DMADBSR_DBS2	(1 << 5)  /* enable doorbell for channel 2 */
#define DMAC_DMADBSR_DBS1	(1 << 5)  /* enable doorbell for channel 1 */
#define DMAC_DMADBSR_DBS0	(1 << 5)  /* enable doorbell for channel 0 */

// DMA interrupt pending register
#define DMAC_DMAIPR_CIRQ5	(1 << 5)  /* irq pending status for channel 5 */
#define DMAC_DMAIPR_CIRQ4	(1 << 4)  /* irq pending status for channel 4 */
#define DMAC_DMAIPR_CIRQ3	(1 << 3)  /* irq pending status for channel 3 */
#define DMAC_DMAIPR_CIRQ2	(1 << 2)  /* irq pending status for channel 2 */
#define DMAC_DMAIPR_CIRQ1	(1 << 1)  /* irq pending status for channel 1 */
#define DMAC_DMAIPR_CIRQ0	(1 << 0)  /* irq pending status for channel 0 */


/* n is the DMA channel (0 - 5) */

#define __dmac_enable_module() \
  ( REG_DMAC_DMACR |= DMAC_DMACR_DMAE | DMAC_DMACR_PR_RR )
#define __dmac_disable_module() \
  ( REG_DMAC_DMACR &= ~DMAC_DMACR_DMAE )

/* p=0,1,2,3 */
#define __dmac_set_priority(p) 				\
do {							\
	REG_DMAC_DMACR &= ~DMAC_DMACR_PR_MASK;		\
	REG_DMAC_DMACR |= ((p) << DMAC_DMACR_PR_BIT);	\
} while (0)

#define __dmac_test_halt_error() ( REG_DMAC_DMACR & DMAC_DMACR_HLT )
#define __dmac_test_addr_error() ( REG_DMAC_DMACR & DMAC_DMACR_AR )

#define __dmac_enable_descriptor(n) \
  ( REG_DMAC_DCCSR((n)) &= ~DMAC_DCCSR_NDES )
#define __dmac_disable_descriptor(n) \
  ( REG_DMAC_DCCSR((n)) |= DMAC_DCCSR_NDES )

#define __dmac_enable_channel(n) \
  ( REG_DMAC_DCCSR((n)) |= DMAC_DCCSR_EN )
#define __dmac_disable_channel(n) \
  ( REG_DMAC_DCCSR((n)) &= ~DMAC_DCCSR_EN )
#define __dmac_channel_enabled(n) \
  ( REG_DMAC_DCCSR((n)) & DMAC_DCCSR_EN )

#define __dmac_channel_enable_irq(n) \
  ( REG_DMAC_DCMD((n)) |= DMAC_DCMD_TIE )
#define __dmac_channel_disable_irq(n) \
  ( REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_TIE )

#define __dmac_channel_transmit_halt_detected(n) \
  (  REG_DMAC_DCCSR((n)) & DMAC_DCCSR_HLT )
#define __dmac_channel_transmit_end_detected(n) \
  (  REG_DMAC_DCCSR((n)) & DMAC_DCCSR_TT )
#define __dmac_channel_address_error_detected(n) \
  (  REG_DMAC_DCCSR((n)) & DMAC_DCCSR_AR )
#define __dmac_channel_count_terminated_detected(n) \
  (  REG_DMAC_DCCSR((n)) & DMAC_DCCSR_CT )
#define __dmac_channel_descriptor_invalid_detected(n) \
  (  REG_DMAC_DCCSR((n)) & DMAC_DCCSR_INV )

#define __dmac_channel_clear_transmit_halt(n) \
  (  REG_DMAC_DCCSR(n) &= ~DMAC_DCCSR_HLT )
#define __dmac_channel_clear_transmit_end(n) \
  (  REG_DMAC_DCCSR(n) &= ~DMAC_DCCSR_TT )
#define __dmac_channel_clear_address_error(n) \
  (  REG_DMAC_DCCSR(n) &= ~DMAC_DCCSR_AR )
#define __dmac_channel_clear_count_terminated(n) \
  (  REG_DMAC_DCCSR((n)) &= ~DMAC_DCCSR_CT )
#define __dmac_channel_clear_descriptor_invalid(n) \
  (  REG_DMAC_DCCSR((n)) &= ~DMAC_DCCSR_INV )

#define __dmac_channel_set_single_mode(n) \
  (  REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_TM )
#define __dmac_channel_set_block_mode(n) \
  (  REG_DMAC_DCMD((n)) |= DMAC_DCMD_TM )

#define __dmac_channel_set_transfer_unit_32bit(n)	\
do {							\
	REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_DS_MASK;	\
	REG_DMAC_DCMD((n)) |= DMAC_DCMD_DS_32BIT;	\
} while (0)

#define __dmac_channel_set_transfer_unit_16bit(n)	\
do {							\
	REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_DS_MASK;	\
	REG_DMAC_DCMD((n)) |= DMAC_DCMD_DS_16BIT;	\
} while (0)

#define __dmac_channel_set_transfer_unit_8bit(n)	\
do {							\
	REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_DS_MASK;	\
	REG_DMAC_DCMD((n)) |= DMAC_DCMD_DS_8BIT;	\
} while (0)

#define __dmac_channel_set_transfer_unit_16byte(n)	\
do {							\
	REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_DS_MASK;	\
	REG_DMAC_DCMD((n)) |= DMAC_DCMD_DS_16BYTE;	\
} while (0)

#define __dmac_channel_set_transfer_unit_32byte(n)	\
do {							\
	REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_DS_MASK;	\
	REG_DMAC_DCMD((n)) |= DMAC_DCMD_DS_32BYTE;	\
} while (0)

/* w=8,16,32 */
#define __dmac_channel_set_dest_port_width(n,w)		\
do {							\
	REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_DWDH_MASK;	\
	REG_DMAC_DCMD((n)) |= DMAC_DCMD_DWDH_##w;	\
} while (0)

/* w=8,16,32 */
#define __dmac_channel_set_src_port_width(n,w)		\
do {							\
	REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_SWDH_MASK;	\
	REG_DMAC_DCMD((n)) |= DMAC_DCMD_SWDH_##w;	\
} while (0)

/* v=0-15 */
#define __dmac_channel_set_rdil(n,v)				\
do {								\
	REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_RDIL_MASK;		\
	REG_DMAC_DCMD((n) |= ((v) << DMAC_DCMD_RDIL_BIT);	\
} while (0)

#define __dmac_channel_dest_addr_fixed(n) \
  (  REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_DAI )
#define __dmac_channel_dest_addr_increment(n) \
  (  REG_DMAC_DCMD((n)) |= DMAC_DCMD_DAI )

#define __dmac_channel_src_addr_fixed(n) \
  (  REG_DMAC_DCMD((n)) &= ~DMAC_DCMD_SAI )
#define __dmac_channel_src_addr_increment(n) \
  (  REG_DMAC_DCMD((n)) |= DMAC_DCMD_SAI )

#define __dmac_channel_set_doorbell(n) \
  (  REG_DMAC_DMADBSR = (1 << (n)) )

#define __dmac_channel_irq_detected(n)  ( REG_DMAC_DMAIPR & (1 << (n)) )
#define __dmac_channel_ack_irq(n)       ( REG_DMAC_DMAIPR &= ~(1 << (n)) )

void dma_init(unsigned int ch);
void dma_write(unsigned int ch,unsigned int *src_buf, unsigned int dst_buf, unsigned int len);
void dma_read(unsigned int ch, unsigned int src_buf, unsigned int *dst_buf, unsigned int len);
/*
static __inline__ int __dmac_get_irq(void)
{
	int i;
	for (i = 0; i < MAX_DMA_NUM; i++)
		if (__dmac_channel_irq_detected(i))
			return i;
	return -1;
}
*/
#endif

