/*
 * (C) Copyright 2000-2010
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * (C) Copyright 2001 Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Andreas Heppel <aheppel@sysgo.de>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "tool-util.h"
#include "types.h"
#include "common.h"
#include "command.h"
#include "global_data.h"
#include "environment.h"
#include "linux/stddef.h"
#include "search.h"

DECLARE_GLOBAL_DATA_PTR;

extern struct platform_info info;

uchar environment[ENV_SIZE];

env_t *env_ptr =  (env_t *)(&environment[0]);

/************************************************************************
 * Default settings to be used when no valid environment is found
 */
#define XMK_STR(x)	#x
#define MK_STR(x)	XMK_STR(x)


uchar default_environment[] = {
	"Mstarbin=20150709"		"\0"
	"\0"
};

unsigned int cfg_env_offset = 0;
int default_environment_size = sizeof(default_environment);
struct hsearch_data env_htab;


static uchar env_get_char_init (int index)
{
	uchar c;

	/* if crc was bad, use the default environment */
	if (gd->env_valid){
		if(info.envtype == UBIENV)
			c = env_get_char_spec_ubi(index);
		else if(info.envtype == NANDENV)
			printf("[%s]: nand env, to be continue\n");
		else
			printf("[%s]: unknown env type, to be continue\n");
	}
	else
		c = default_environment[index];

	return (c);
}

uchar env_get_char_memory (int index)
{
	return *env_get_addr(index);
}

uchar env_get_char (int index)
{
	uchar c;

	/* if relocated to RAM */
	if (gd->flags & GD_FLG_RELOC)
		c = env_get_char_memory(index);
	else
		c = env_get_char_init(index);

	return (c);
}

uchar *env_get_addr (int index)
{
	if (gd->env_valid)
		return (uchar *)(gd->env_addr + index);
	else
		return &default_environment[index];
}

void set_default_env(const char *s)
{
	if (sizeof(default_environment) > ENV_SIZE) {
		puts("*** Error - default environment is too large\n\n");
		return;
	}

	if (s) {
		if (*s == '!') {
			printf("*** Warning - %s, "
				"using default environment\n\n",
				s+1);
		} else {
			puts(s);
		}
	} else {
		puts("Using default environment\n\n");
	}

	if (himport_r(&env_htab, (char *)default_environment,
		    sizeof(default_environment), '\0', 0) == 0) {
		error("Environment import failed: errno = %d\n", errno);
	}
	gd->flags |= GD_FLG_ENV_READY;
}

/*
 * Check if CRC is valid and (if yes) import the environment.
 * Note that "buf" may or may not be aligned.
 */
int env_import(const char *buf, int check)
{
	env_t *ep = (env_t *)buf;

	if (check) {
		uint32_t crc;

		memcpy(&crc, &ep->crc, sizeof(crc));

		if (crc32(0, ep->data, ENV_SIZE) != crc) {
			set_default_env("!bad CRC");
			return 0;
		}
	}

	if (himport_r(&env_htab, (char *)ep->data, ENV_SIZE, '\0', 0)) {
		gd->flags |= GD_FLG_ENV_READY;
		return 1;
	}

	error("Cannot import environment: errno = %d\n", errno);

	set_default_env("!import failed");

	return 0;
}

int env_relocate (void)
{
	// MSTAR start
	env_ptr = (env_t *)malloc (CONFIG_ENV_SIZE);
	if(env_ptr == NULL){
		printf("%s: malloc failed, at %d\n", __func__, __LINE__);
		return -1;
	}
	memset (env_ptr, 0, sizeof(env_t));
	// MSTAR end
	if (gd->env_valid == 0) {
		set_default_env("!bad CRC");
		memcpy (env_ptr->data, default_environment, sizeof(default_environment));
		gd->env_valid = 1;
	} else {
		printf("TT--------------------%d----------------------------------------\n", info.envtype);
		if(info.envtype == UBIENV)
			env_relocate_spec_ubi();
		else if(info.envtype == NANDENV)
			env_relocate_spec_nand();
		else
			printf("Unknown env type, to be continue\n");
	}
	return 0;
}

int env_complete(char *var, int maxv, char *cmdv[], int bufsz, char *buf)
{
	ENTRY *match;
	int found, idx;

	idx = 0;
	found = 0;
	cmdv[0] = NULL;

	while ((idx = hmatch_r(var, idx, &match, &env_htab))) {
		int vallen = strlen(match->key) + 1;

		if (found >= maxv - 2 || bufsz < vallen)
			break;

		cmdv[found++] = buf;
		memcpy(buf, match->key, vallen);
		buf += vallen;
		bufsz -= vallen;
	}

	qsort(cmdv, found, sizeof(cmdv[0]), strcmp_compar);

	if (idx)
		cmdv[found++] = "...";
	cmdv[found] = NULL;
	return found;
}
