/***********************************************************
*
* Copyright (C), 2010-2014, TP-LINK TECHNOLOGIES CO., LTD.
* All rights reserved.
*
* File name  : plugin_util.c
* Version	 : 1.0
* Description:
*
* Author	 : Wang FuYu <wangfuyu@tp-link.net>
* Create Date: 2014-04-14
*
* History	 :
*------------------------------------------------------------
*
*01, 14Apr14, Wang FuYu, create file.
************************************************************/

#include <inttypes.h>
#include "plugin_util.h"
#include "common.h"

HEADER_TAG gImgHeader;
EFS_HEADER gEfsHeader;

/******************************************************************************
* FUNCTION		: getFolderEntryPath()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	:
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
static int getFolderEntryPath(EFS_HEADER* efsHeader, unsigned int level,
                       unsigned int entryNum, char *parentPath)
{
	int ret = -1;
	unsigned int idx = 0;
	unsigned int parentLevel = 0;
	char path[MAX_FILENAME_LEN] = {0};
	char fullPath[MAX_FILENAME_LEN] = {0};

	if (NULL == efsHeader || NULL == parentPath)
	{
		OPKG_LOG("input paras invalid!\n");
		return -1;
	}

	memset(path, 0, MAX_FILENAME_LEN);
	memset(fullPath, 0, MAX_FILENAME_LEN);
	if (efsHeader->efsEntry[entryNum].type == EFS_ENTRY_TYPE_FILE)
	{
		strcpy(parentPath, path);
		return 0;
	}

	if (level == EFS_ROOT_LEVEL)
	{
		strcpy(parentPath, path);
		return 0;
	}
	else
	{
		idx = efsHeader->efsEntry[entryNum].parentIndex;
		if (idx < efsHeader->entryNum)
		{
			parentLevel = efsHeader->efsEntry[idx].folderPro.level;
			ret = getFolderEntryPath(efsHeader, parentLevel, idx, path);
			if (0 != ret)
			{
				OPKG_LOG("getFolderEntryPath failed\n");
				return -1;
			}
		}

		if (0 != strlen(path))
		{
			snprintf(fullPath, MAX_FILENAME_LEN, "%s%s%s",
				path, "/", efsHeader->efsEntry[entryNum].name);
		}
		else
		{
			strcpy(fullPath, efsHeader->efsEntry[entryNum].name);
			OPKG_LOG("fullPath is %s\n", fullPath);
		}
	}

	strcpy(parentPath, fullPath);
	OPKG_LOG("parentPath is %s\n", parentPath);
	return 0;
}

/******************************************************************************
* FUNCTION		: getFileEntryPath()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: get file's full path
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
static int getFileEntryPath(EFS_HEADER* efsHeader, unsigned int entryNum, char *filePath)
{
	int ret = -1;
	unsigned int i = 0;
	unsigned int parentIndex = 0;
	unsigned int parentLevel = 0;
	char parentPath[MAX_FILENAME_LEN] = {0};
	char fullPath[MAX_FILENAME_LEN] = {0};

	if (NULL == efsHeader || NULL == filePath)
	{
		OPKG_LOG("Invalid paras: NULL == efsHeader || NULL == filePath!\n");
		return -1;
	}

	memset(fullPath, 0, MAX_FILENAME_LEN);
	memset(parentPath, 0, MAX_FILENAME_LEN);
	if (entryNum >= efsHeader->entryNum)
	{
		strcpy(filePath, parentPath);
		return 0;
	}

	if (efsHeader->efsEntry[entryNum].type == EFS_ENTRY_TYPE_FILE)
	{
		parentIndex = efsHeader->efsEntry[entryNum].parentIndex;
		if (parentIndex < efsHeader->entryNum)
		{
			parentLevel = efsHeader->efsEntry[parentIndex].folderPro.level;
			ret = getFolderEntryPath(efsHeader, parentLevel, parentIndex, parentPath);
			if (0 != ret)
			{
				OPKG_LOG("getFolderEntryPath failed!\n");
				return -1;
			}
			OPKG_LOG("parentIndex is %d, level is %d, parentPath is %s\n",
									parentIndex, parentLevel, parentPath);
		}

		if (0 != strlen(parentPath))
		{
			snprintf(fullPath, MAX_FILENAME_LEN, "%s%s%s",
				parentPath,
				"/",
				efsHeader->efsEntry[entryNum].name);
		}
		else
		{
			strcpy(fullPath, efsHeader->efsEntry[entryNum].name);
		}
	}

	strcpy(filePath, fullPath);
	return 0;
}

/******************************************************************************
* FUNCTION		: writeBufToFile()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: write buf data to file
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
static int writeBufToFile(char* filePath, unsigned char* buf, int len)
{
	FILE* fp = NULL;
	int writeBytes = 0;

	if (NULL == filePath || NULL == buf)
	{
		OPKG_LOG("Invalid paras:NULL == filePath || NULL == buf\n");
		return -1;
	}

	fp = fopen(filePath, "wb");
	if(!fp)
	{
		OPKG_LOG("open %s failed!\n", filePath);
		return -1;
	}

	writeBytes = fwrite(buf, sizeof(unsigned char), len, fp);
	fclose(fp);

	if (writeBytes != len)
	{
		OPKG_LOG("write files failed! len = %d\n", len);
		return -1;
	}

	return 0;
}

/******************************************************************************
* FUNCTION		: writeBufToEfsEntries()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: according to dir tree msg, write buf content to file
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
static int writeBufToEfsEntries(unsigned char *buf, char *dstDirPath)
{
	int ret = -1;;
	int efsHeaderLen = 0;
	unsigned int i = 0;
	unsigned char* bufStart = NULL;
	char *filePathPrefix = NULL;
	char filePath[MAX_FILENAME_LEN] = {0};
	char fullPath[MAX_FILENAME_LEN] = {0};

	if (NULL == buf || NULL == dstDirPath)
	{
		OPKG_LOG("buf or dstDirPath is NULL!\n");
		return -1;
	}

	bufStart = buf;

	filePathPrefix = dstDirPath;

	memset(filePath, 0, MAX_FILENAME_LEN);
	memset(fullPath, 0, MAX_FILENAME_LEN);

	efsHeaderLen = sizeof(EFS_HEADER);
	memset((char*)&gEfsHeader, 0, efsHeaderLen);

	memcpy(&gEfsHeader, bufStart, efsHeaderLen);
	bufStart += efsHeaderLen;

	gEfsHeader.entryNum = ntohl(gEfsHeader.entryNum);

	for (i = 0; i < gEfsHeader.entryNum; i++)
	{
		if (gEfsHeader.efsEntry[i].type == EFS_ENTRY_TYPE_FILE) //file
		{
			gEfsHeader.efsEntry[i].parentIndex = ntohl(gEfsHeader.efsEntry[i].parentIndex);
			gEfsHeader.efsEntry[i].filePro.length = ntohl(gEfsHeader.efsEntry[i].filePro.length);
			gEfsHeader.efsEntry[i].filePro.start = ntohl(gEfsHeader.efsEntry[i].filePro.start);
		}
		else
		{
			gEfsHeader.efsEntry[i].parentIndex = ntohl(gEfsHeader.efsEntry[i].parentIndex);
			gEfsHeader.efsEntry[i].folderPro.level = ntohl(gEfsHeader.efsEntry[i].folderPro.level);
		}
	}

	for (i = 0; i < gEfsHeader.entryNum; i++)
	{
		if (gEfsHeader.efsEntry[i].type == EFS_ENTRY_TYPE_FILE) //file
		{
			/* ȡļ·(Ŀ¼У׽ڵ洢) */
			ret = getFileEntryPath(&gEfsHeader, i, filePath);
			if (0 != ret)
			{
				OPKG_LOG("getFileEntryPath failed!\n");
				return ret;
			}
			snprintf(fullPath, MAX_FILENAME_LEN, "%s%s", filePathPrefix, filePath);

			/* ļд뵽 */
			ret = writeBufToFile(fullPath, bufStart, gEfsHeader.efsEntry[i].filePro.length);
			if (0 != ret)
			{
				OPKG_LOG("write file data to buf failed!\n");
				return ret;
			}

			/* ƶ洢ָ */
			bufStart += gEfsHeader.efsEntry[i].filePro.length;
		}
		else
		{
			getFolderEntryPath(&gEfsHeader, gEfsHeader.efsEntry[i].folderPro.level, i, filePath);

			snprintf(fullPath, MAX_FILENAME_LEN, "%s%s", filePathPrefix, filePath);
			ret = mkdirRecursive(fullPath, PLUGIN_DIRMODE);
			if (0 != ret)
			{
				OPKG_LOG("make directory %s failed!\n", fullPath);
				return -1;
			}
		}
	}

	return 0;
}

/******************************************************************************
* FUNCTION		: unpackUciCfg()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: unpack config data
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
static int unpackUciCfg(unsigned char *unpackSrcDataBuf, int unpackSrcDataLen, char *dstDirPath)
{
	int ret = 0;
	int imgLen = 0;
	int imgHeaderLen = 0;
	unsigned char *imgBuf = NULL;
	unsigned char *imgStart = NULL;
	DIR *dir = NULL;

	if (NULL == unpackSrcDataBuf || 0 >= unpackSrcDataLen || NULL == dstDirPath)
	{
		OPKG_LOG("NULL == unpackSrcDataBuf || 0 >= unpackSrcDataLen || NULL == dstDirPath!\n");
		return -1;
	}

	dir = opendir(dstDirPath);
	if (NULL == dir)
	{
		OPKG_LOG("%s doesn't exit, Now create it!\n", dstDirPath);
		ret = mkdirRecursive(dstDirPath, PLUGIN_DIRMODE);
		if (0 != ret)
		{
			OPKG_LOG("make directory %s failed!\n", dstDirPath);
			return -1;
		}
	}
	else
	{
		closedir(dir);
	}

	imgHeaderLen = sizeof(HEADER_TAG);
	memset((char*)&gImgHeader, 0, imgHeaderLen);

 	imgBuf = unpackSrcDataBuf;
	imgLen = unpackSrcDataLen;

	imgStart = imgBuf;
	memcpy(&gImgHeader, imgStart, imgHeaderLen);
	gImgHeader.imgLen = ntohl(gImgHeader.imgLen);
	imgStart += imgHeaderLen;

	ret = writeBufToEfsEntries(imgStart, dstDirPath);
	if (0 != ret)
	{
		OPKG_LOG("write buf to files failed!\n");
		return -1;
	}

	OPKG_LOG("read ipk msg successfully!\n");
	return 0;
}


/******************************************************************************
* FUNCTION		: delHeaderForMsg()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: delete message header from flashMsgBuf
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
static unsigned char *delHeaderForMsg(unsigned char *pluginMsgBuf, int *srcDecryptBufLen)
{
	int pluginMsgHeaderLen = 0;
	unsigned char *srcDecryptBuf = NULL;
	UC_BIN_HEADER pluginMsgHeaderStr;

	if (NULL == pluginMsgBuf || NULL == srcDecryptBufLen)
	{
		OPKG_LOG("NULL == flashMsgBuf || NULL == srcDecryptBufLen!\n");
		return NULL;
	}

	pluginMsgHeaderLen = sizeof(UC_BIN_HEADER);
	memset(&pluginMsgHeaderStr, 0, pluginMsgHeaderLen);
	memcpy(&pluginMsgHeaderStr, pluginMsgBuf, pluginMsgHeaderLen);

	OPKG_LOG("flag is %08x, magic is %08x, ver is %08x, length is %08x\n",
		pluginMsgHeaderStr.updateFlag, pluginMsgHeaderStr.magic,
		pluginMsgHeaderStr.ver, pluginMsgHeaderStr.length);

	pluginMsgHeaderStr.updateFlag = ntohl(pluginMsgHeaderStr.updateFlag);
	pluginMsgHeaderStr.length = ntohl(pluginMsgHeaderStr.length);
	pluginMsgHeaderStr.magic = ntohl(pluginMsgHeaderStr.magic);
	pluginMsgHeaderStr.ver = ntohs(pluginMsgHeaderStr.ver);

	if (UC_FLASH_MAGIC_VALUE != pluginMsgHeaderStr.magic
		|| UC_FLASH_VER != pluginMsgHeaderStr.ver)
	{
		OPKG_LOG("flash data is wrong!\n");
		return NULL;
	}

	*srcDecryptBufLen = pluginMsgHeaderStr.length - pluginMsgHeaderLen;
	srcDecryptBuf = (unsigned char*)malloc(*srcDecryptBufLen);
	if (NULL == srcDecryptBuf)
	{
		OPKG_LOG("malloc srcDecryptBuf failed!\n");
		return NULL;
	}

	memset(srcDecryptBuf, 0, *srcDecryptBufLen);
	memcpy(srcDecryptBuf, pluginMsgBuf + pluginMsgHeaderLen, *srcDecryptBufLen);

	OPKG_LOG("delHeaderForMsg successfully!\n");
	return srcDecryptBuf;
}

/******************************************************************************
* FUNCTION		: uncompressFile()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: decompress file
* INPUT			:
* OUTPUT		:
* RETURN		:
******************************************************************************/
static unsigned char *uncompressFile(unsigned char *ucCmpBuf, uLong *ucCmpLen)
{
	int ret = 0;
	int bufOff = 0;
	uLong flen = 0;
	uLong ulen = 0;
	uint32_t tmpFlen = 0;
	uint32_t tmpUlen = 0;
	FILE *file = NULL;
	unsigned char *fbuf = NULL;
	unsigned char *ubuf = NULL;

	if (NULL == ucCmpBuf || NULL == ucCmpLen)
	{
		OPKG_LOG("ucCmpBuf is NULL, or ucCmpLen is NULL!\n");
		return NULL;
	}

	bufOff = sizeof(uint32_t) * 2;
	memcpy(&tmpUlen, ucCmpBuf, sizeof(uint32_t));
	memcpy(&tmpFlen, ucCmpBuf + sizeof(uint32_t), sizeof(uint32_t));

	ulen = (uLong)tmpUlen;
	flen = (uLong)tmpFlen;
	ulen = ntohl(ulen);
	flen = ntohl(flen);
	fbuf = ucCmpBuf; 
	
	ubuf = (unsigned char*)malloc(sizeof(unsigned char) * ulen);
	if (NULL == ubuf)
	{
		OPKG_LOG("No enough memory!\n");
		return NULL;
	}
	memset(ubuf, 0, ulen);

	/* decompress file */
	ret = SLP_UNCOMPRESS(ubuf, &ulen, fbuf + bufOff, flen);
	if (Z_OK != ret)
	{
		OPKG_LOG("Uncompress buf failed!\n");
		FREE_P(ubuf);
		return NULL;
	}

	*ucCmpLen = ulen;
	return ubuf;
}

/******************************************************************************
* FUNCTION		: unpackIpkToFile()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: release ipk file from buf
* INPUT			: @flashMsgBuf:repository of config data
				  @dstDirPathName:directory, used to save config file
* OUTPUT		:
* RETURN		: 0 means success, -1 means failed
******************************************************************************/
static int unpackIpkToFile(char *pluginMsgBuf, char *dstDirPathName)
{
	int ret = -1;
	int desMd5Len = 0;
	unsigned long unpackSrcDataLen = 0;
	unsigned char *desMd5Buf = NULL;
	unsigned char *unpackSrcDataBuf = NULL;

//	OPKG_LOG("unpack_ipk_to_file, dir is %s.\n", dstDirPathName);
	if (NULL == pluginMsgBuf || NULL == dstDirPathName)
	{
		OPKG_LOG("NULL == pluginMsgBuf || NULL == dstDirPathName\n");
		return -1;
	}

//	OPKG_LOG("del header begin.......................\n");
	desMd5Buf = delHeaderForMsg(pluginMsgBuf, &desMd5Len);
	if (NULL == desMd5Buf || 0 >= desMd5Len)
	{
		OPKG_LOG("delHeaderForMsg failed!\n");
		FREE_P(pluginMsgBuf);
		return -1;
	}

	FREE_P(pluginMsgBuf);

//	OPKG_LOG("uncompress buf begin....................\n");
	unpackSrcDataBuf = uncompressFile(desMd5Buf, &unpackSrcDataLen);
	if (NULL == unpackSrcDataBuf || 0 >= unpackSrcDataLen)
	{
		OPKG_LOG("uncompress buf failed!\n");
		FREE_P(desMd5Buf);
		return -1;
	}

	FREE_P(desMd5Buf);

//	OPKG_LOG("unpack begin...........................\n");
	ret = unpackUciCfg(unpackSrcDataBuf, unpackSrcDataLen, dstDirPathName);
	if (0 != ret)
	{
		OPKG_LOG("unpack uc data failed!\n");
		FREE_P(unpackSrcDataBuf);
		return -1;
	}

	FREE_P(unpackSrcDataBuf);

	OPKG_LOG("release ipk data from plugin pkts successfully!\n");
	return 0;

}

/******************************************************************************
* FUNCTION		: unzipPluginPkg()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: unzip plugin data.
* INPUT			: 
* OUTPUT		: 
* RETURN		: sucessed 0
******************************************************************************/
int unzipPluginPkg(const char *url, char ipkDir[])
{
	int i = 0;
	int ret = -1;
	int fileIdx = 0;
	int readBytes = 0;
	int pkgSize = 0;
	int pluginPartsFileLen = 0;
	int pluginPartsFileOff = 0;
	int pluginPartsHeaderLen = 0;
	char pluginPartsPrefix[MAX_FILENAME_LEN] = {0};
	char pluginPartsPath[MAX_FILENAME_LEN] = {0};
	char pluginBuf[PLUGIN_FILE_MAX_SIZE] = {0}; 
	char md5DigestOld[MAX_MD5_LEN + 1] = {0};
	char md5DigestNew[MAX_MD5_LEN + 1] = {0};
	char pluginMD5Salt[MAX_MD5_LEN + 1] = {0xCC, 0x96, 0x28, 0xEE, 0x8D, 0xFB, 0x21, 0xBB, \
						 					0x3D, 0xEF, 0x6C, 0xB5, 0x9F, 0x77, 0x4C, 0x7C};
	PLUGIN_PARTS_HAEDER pluginPartsHeader;
	FILE *file = NULL;
	
	if (!url)
	{
		OPKG_LOG("url is NULL!\n");
		return -1;
	}

	OPKG_LOG("unzip_plugin_pkg, url is %s!\n", url);
	file = fopen(url, "rb");
	if(!file)
	{
		OPKG_LOG("fopen %s failed.\n", url);
		return -1;
	}

	fseek(file, 0L, SEEK_END);
	pkgSize = ftell(file);
	fseek(file, 0L, SEEK_SET);

	pluginPartsHeaderLen = sizeof(PLUGIN_PARTS_HAEDER);
	if (PLUGIN_FILE_MAX_SIZE < pkgSize || pluginPartsHeaderLen > pkgSize)
	{
		OPKG_LOG("zip file size invalid.\n");
		fclose(file);
		return -1;	
	}

	memset(pluginBuf, 0, PLUGIN_FILE_MAX_SIZE);
	readBytes = fread(pluginBuf, sizeof(unsigned char), pkgSize, file);
	fclose(file);
	if (readBytes != pkgSize)
	{
		OPKG_LOG("read pkg file contents failed.\n");
		return -1;
	}

	/* extract plugin parts header from zip file */
	memset(&pluginPartsHeader, 0, pluginPartsHeaderLen);
	memcpy(&pluginPartsHeader, pluginBuf, pluginPartsHeaderLen);

	memcpy(md5DigestOld, pluginPartsHeader.md5, MAX_MD5_LEN);
	memcpy(pluginPartsHeader.md5, pluginMD5Salt, MAX_MD5_LEN);
	memcpy(pluginBuf, &pluginPartsHeader, pluginPartsHeaderLen);

	SLP_MD5_DIGEST(md5DigestNew, pluginBuf, pkgSize);
	if (0 != memcmp(md5DigestOld, md5DigestNew, MAX_MD5_LEN))
	{
		OPKG_LOG("plugin bin file's md5 is wrong.\n");
		return -1;
	}

	pluginPartsHeader.pluginSize = ntohl(pluginPartsHeader.pluginSize);
	pluginPartsHeader.headerVer = ntohl(pluginPartsHeader.headerVer);
	pluginPartsHeader.pluginType = ntohl(pluginPartsHeader.pluginType);
	
#if SLP_OPKG_DEBUG
	OPKG_LOG("gPluginHeader, id: %s, ver: %s, name: %s, size: %d, tag: %s, comp: %s, \
				author: %s, os: %s, data: %s, webicon: %s, appicon: %s, appzip: %s, plugin: %s\n",
				pluginPartsHeader.pluginId, pluginPartsHeader.pluginVer, pluginPartsHeader.pluginName,
				pluginPartsHeader.pluginSize, pluginPartsHeader.pluginTag, pluginPartsHeader.pluginComp,
				pluginPartsHeader.pluginAuthor, pluginPartsHeader.osVer, pluginPartsHeader.pluginFile[0].fileName,
				pluginPartsHeader.pluginFile[1].fileName, pluginPartsHeader.pluginFile[2].fileName, 
				pluginPartsHeader.pluginFile[3].fileName, url);

	OPKG_LOG("old md5:\n");
	for (i = 0; i < 16; i++)
	{
		OPKG_LOG("%02x.", (unsigned char)md5DigestOld[i]);
	}
	OPKG_LOG("\n");

	OPKG_LOG("new md5:\n");
	for (i = 0; i < 16; i++)
	{
		OPKG_LOG("%02x.", (unsigned char)md5DigestNew[i]);
	}
	OPKG_LOG("\n");
#endif

	/* extract plugin parts from bin file, and write them to new file */
	for (fileIdx = 0; fileIdx < PLUGIN_PARTS_MAX; fileIdx++)
	{
		memset(pluginPartsPrefix, 0, MAX_FILENAME_LEN);
		memset(pluginPartsPath, 0, MAX_FILENAME_LEN);

		if (PLUGIN_BIN_FILE == fileIdx)
		{
			snprintf(pluginPartsPrefix, MAX_FILENAME_LEN, "%s%s/", 
					PLUGIN_TMP_DIR, pluginPartsHeader.pluginId);
			memcpy(ipkDir, pluginPartsPrefix, MAX_FILENAME_LEN);
		}
		else
		{
			snprintf(pluginPartsPrefix, MAX_FILENAME_LEN, "%s%s/", 
					PLUGIN_PARTS_DIR, pluginPartsHeader.pluginId);
			
		}

		ret = mkdirRecursive(pluginPartsPrefix, PLUGIN_DIRMODE);
		if (0 != ret)
		{
			OPKG_LOG("make directory %s failed!\n", pluginPartsPrefix);
			return -1;
		}

		snprintf(pluginPartsPath, MAX_FILENAME_LEN, "%s%s", pluginPartsPrefix, pluginPartsHeader.pluginFile[fileIdx].fileName);
		
		pluginPartsFileOff = ntohl(pluginPartsHeader.pluginFile[fileIdx].fileOffset);
		pluginPartsFileLen = ntohl(pluginPartsHeader.pluginFile[fileIdx].fileSize);
		if (PLUGIN_FILE_MAX_SIZE < pluginPartsFileOff + pluginPartsFileLen)
		{
			OPKG_LOG("plugin's parts is too big[%d].\n", pluginPartsFileLen);
			return -1;
		}

		if (PLUGIN_BIN_FILE == fileIdx)
		{
			ret = unpackIpkToFile(pluginBuf + pluginPartsFileOff, pluginPartsPrefix);
			if (0 != ret)
			{
				OPKG_LOG("release ipk to file failed.\n");
				return -1;
			}
		}
		else
		{
			ret = writeBufToFile(pluginPartsPath, pluginBuf + pluginPartsFileOff, pluginPartsFileLen);
			if (0 != ret)
			{
				OPKG_LOG("write buf to file failed.\n");
				return -1;
			}
		}	
	}

	OPKG_LOG("unzip plugin data successful.\n");
	return 0;	
}

