/***********************************************************
*
* Copyright (C), 2010-2014, TP-LINK TECHNOLOGIES CO., LTD.
* All rights reserved.
*
* File name  : common.c
* Version	 : 1.0
* Description:
*
* Author	 : Wang FuYu <wangfuyu@tp-link.net>
* Create Date: 2014-06-05
*
* History	 :
*------------------------------------------------------------
*
*01, 05Jun14, Wang FuYu, create file.
************************************************************/

#include "plugin_util.h"
#include "common.h"

/******************************************************************************
* FUNCTION		: mkdirRecursive()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: mkdir recursively
* INPUT 		:
* OUTPUT		:
* RETURN		: 0 means success, -1 means failed
******************************************************************************/
int mkdirRecursive(const char *dstDir, mode_t mode)
{
	int idx = 0;
	int ret = -1;
	char *tmp = NULL;
	char parentPath[MAX_FILENAME_LEN] = {0};
	char name[MAX_FILENAME_LEN] = {0};

	if (NULL == dstDir || MAX_FILENAME_LEN  < strlen(dstDir))
	{
		OPKG_LOG("NULL == dstDir || MAX_FILENAME_LEN  < strlen(dstDir)\n");
		return -1;
	}

	if (0 == directoryExist(dstDir))
	{
		return 0;
	}

	memset(name, 0, MAX_FILENAME_LEN);
	strncpy(name, dstDir, strlen(dstDir));
	if ('/' == name[strlen(name) - 1])
	{
		name[strlen(name) - 1] = '\0';
	}

	tmp = strrchr(name, '/');
	if (NULL == tmp)
	{
		OPKG_LOG("[ERROR]: invalid path with no '/': %s\n", name);
		return -1;
	}

	memset(parentPath, 0, MAX_FILENAME_LEN);
	strncpy(parentPath, name, tmp - name);
	if (0 != directoryExist(parentPath))
	{
		ret = mkdirRecursive(parentPath, mode);
		if (0 != ret)
		{
			OPKG_LOG("mkdir %s failed!\n", parentPath);
			return -1;
		}
	}

	ret = mkdir(name, mode);
	if (0 != ret)
	{
		OPKG_LOG("mkdir %s failed!\n", name);
		return -1;
	}

	return 0;
}

/******************************************************************************
* FUNCTION		: directory_exist()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: check if a directory exists
* INPUT 		:
* OUTPUT		:
* RETURN		: 0 means existed, -1 means doesn't exist
******************************************************************************/
int directoryExist(char *dname)
{
	struct stat st;

	if (NULL == dname)
	{
		OPKG_LOG("dname is NULL!\n");
		return -1;
	}

	if (0 != stat(dname, &st))
	{
		OPKG_LOG("%s doesn't exist!\n", dname);
		return -1;
	}

	if (S_ISDIR(st.st_mode));
	{
		OPKG_LOG("%s existed!\n", dname);
		return 0;
	}

	return -1;
}

/******************************************************************************
* FUNCTION		: makeSymLink()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: make sysmlink
* INPUT 		:
* OUTPUT		:
* RETURN		: 0 means success, -1 means failed
******************************************************************************/
int makeSymLink(const char *oldPath, const char *newPath)
{
	int ret = -1;
	struct stat st;

	if (NULL == oldPath || NULL == newPath)
	{
		OPKG_LOG("NULL == oldPath || NULL == newPath\n");
		return -1;
	}

	if (0 == stat(newPath, &st))
	{
		OPKG_LOG("%s existed, remove it!\n", newPath);
		removeDir(newPath);
	}

	ret = symlink(oldPath, newPath);
	if (0 != ret)
	{
		OPKG_LOG("make symlink %s --> %s failed!\n", newPath, oldPath);
		return -1;
	}
	else
	{
		OPKG_LOG("make symlink %s --> %s success!\n", newPath, oldPath);
		return 0;
	}

}

/******************************************************************************
* FUNCTION		: rmDirOrFile()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: remove dir of file
* INPUT 		:
* OUTPUT		:
* RETURN		: 0 means success, -1 means failed
******************************************************************************/
static int rmDirOrFile(const char *fpath, const struct stat *sb,
	                int typeflag, struct FTW *ftwbuf)
{
	int ret = -1;

	/* other para may be NULL */
	if (NULL == fpath)
	{
		return -1;
	}

	ret = remove(fpath);
	if (0 != ret)
	{
		OPKG_LOG("remove %s failed!\n", fpath);
		return -1;
	}

	return 0;
}

/******************************************************************************
* FUNCTION		: removeDir()
* AUTHOR		: wangfuyu <wangfuyu@tp-link.net>
* DESCRIPTION	: remove dir
* INPUT 		:
* OUTPUT		:
* RETURN		: 0 means success, -1 means failed
******************************************************************************/
int removeDir(const char *dirPath)
{
	int ret = -1;
	struct stat st;

	if (NULL == dirPath)
	{
		OPKG_LOG("NULL == dirPath\n");
		return -1;
	}

	if (0 != stat(dirPath, &st))
	{
		OPKG_LOG("%s doesn't exist, do not need remove\n", dirPath);
		return 0;
	}

	ret = nftw(dirPath, rmDirOrFile, MAX_FD_OPEN_SIMULTANEOUSLY, FTW_PHYS | FTW_DEPTH);
	if (0 != ret)
	{
		OPKG_LOG("nftw, remove dir:%s failed!\n", dirPath);
		return -1;
	}

	OPKG_LOG("remove %s success!\n", dirPath);
	return 0;
}

