/*
 * drivers/mtd/nand/nand_util.c
 *
 * Copyright (C) 2006 by Weiss-Electronic GmbH.
 * All rights reserved.
 *
 * @author:	Guido Classen <clagix@gmail.com>
 * @descr:	NAND Flash support
 * @references: borrowed heavily from Linux mtd-utils code:
 *		flash_eraseall.c by Arcom Control System Ltd
 *		nandwrite.c by Steven J. Hill (sjhill@realitydiluted.com)
 *			       and Thomas Gleixner (tglx@linutronix.de)
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 * Copyright 2010 Freescale Semiconductor
 * The portions of this file whose copyright is held by Freescale and which
 * are not considered a derived work of GPL v2-only code may be distributed
 * and/or modified under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>

#include "common.h"
#include "nand.h"
#include "tool-util.h"
#include "../../mstar/unfd/inc/common/drvNAND.h"

typedef struct erase_info erase_info_t;
typedef struct mtd_info	  mtd_info_t;
extern struct platform_info info;

/*
 * buffer array used for writing data
 */
unsigned char *data_buf;
unsigned char *oob_buf;

extern EN_NAND_FS_FORMAT nand_fs_format;


/*******************************CODE*************************************/
/**
 * nand_erase_opts: - erase NAND flash with support for various options
 *		      (jffs2 formating)
 *
 * @param meminfo	NAND device to erase
 * @param opts		options,  @see struct nand_erase_options
 * @return		0 in case of success
 *
 * This code is ported from flash_eraseall.c from Linux mtd utils by
 * Arcom Control System Ltd.
 */
int nand_erase_opts(nand_info_t *meminfo, const nand_erase_options_t *opts)
{
	erase_info_t erase;

	memset(&erase, 0, sizeof(erase_info_t));
	erase.mtd = meminfo;
	erase.addr = opts->offset;
	erase.len = opts->length;

	return meminfo->erase(meminfo, &erase);
}

/**
 * _nand_write_opts: - write image to NAND flash with support for various options
 *
 * @param meminfo        AND device to erase
 * @param opts              write options (@see nand_write_options)
 * @param nandoffset     nand offset where image writen to
 * @return                      in case of success
 *
 * This code is ported from nandwrite.c from Linux mtd utils by
 * Strawfly.
 */
int _nand_write_opts(nand_info_t *meminfo, nand_write_options_t *opts, u64 nandoffset)
{
	ulong imglen = 0;
	int readlen;
	u64 mtdoffset = opts->offset;
	ulong erasesize_blockalign, pagelen;
	u_char *buffer = opts->buffer;
	size_t written;
	int result;
	u64 nandsize_oob=0, mtdoffsetnoOob=0, nandoffsetnoOob=0;
	struct mtd_oob_ops tempOobOps;

	if (opts->pad && opts->writeoob){
		printf("[%s]:Can't pad when oob data is present.\n", __func__);
		return -1;
	}

	if(opts->writeoob)
		printf("[%s]: Write image with oob data\n", __func__);
	else
		printf("[%s]: Write image without oob data\n", __func__);

	if(opts->writeoob)
		pagelen = meminfo->oobblock + meminfo->oobsize;
	else
		pagelen = meminfo->oobblock;

	/* get image length */
	imglen = opts->length;
	/* nand chip size */
	nandsize_oob = (meminfo->size / meminfo->oobblock) * pagelen;

	if(opts->writeoob){
		if(imglen % pagelen){
			printf("[%s]Error: In case of write with oob, write length(0x%X) should align with page size with oob(0x%X)!\n", __func__, imglen, pagelen);
			return -1;
		}
	}

	if (imglen > (nandsize_oob - opts->offset - nandoffset)){
		printf("Input block does not fit into device\n");
		return -1;
	}

	/////////////////////////////////////////////////////////////added
	printf("[%s]: Write file at 0x%llX, length 0x%lX.........\n", __func__, (mtdoffset + nandoffset) * pagelen / meminfo->oobblock, imglen);
	while(imglen && (mtdoffset < (nandsize_oob - nandoffset))){
		mtdoffsetnoOob = mtdoffset / pagelen * meminfo->oobblock;
		nandoffsetnoOob = nandoffset / pagelen * meminfo->oobblock;
		readlen = pagelen;

		if(imglen < readlen)
			readlen = imglen;

		if(opts->writeoob){
			/* read page data from input memory buffer */
			memcpy(data_buf, buffer, meminfo->oobblock);
			memcpy(oob_buf, buffer + meminfo->oobblock, meminfo->oobsize);
		}
		else{
			memcpy(data_buf, buffer, readlen);
		}

		if (opts->pad && (readlen < pagelen)){
			memset(data_buf + readlen, 0xFF, pagelen - readlen);
		}

		if(mtdoffset < (nandsize_oob - nandoffset)){
			if(opts->writeoob){
				memset(&tempOobOps, 0, sizeof(struct mtd_oob_ops));
				tempOobOps.datbuf = data_buf;
				tempOobOps.len = meminfo->oobblock;
				tempOobOps.oobbuf = oob_buf;
				tempOobOps.ooblen =meminfo->oobsize;
				tempOobOps.mode = MTD_OOB_RAW;
				result = meminfo->write_oob(meminfo, (mtdoffsetnoOob + nandoffsetnoOob), &tempOobOps);
			}
			else{
				result = meminfo->write(meminfo,
					(mtdoffsetnoOob + nandoffsetnoOob),
					meminfo->oobblock,
					&written,
					(unsigned char *) data_buf);
			}

			if (result){
				printf("writing NAND page at offset 0x%llx failed, error %d\n",
					(mtdoffsetnoOob + nandoffsetnoOob), result);
				return result;;
			}
		}

		mtdoffset += readlen;
		imglen -= readlen;
		buffer += readlen;
	}

	if (imglen > 0){
		printf("Data did not fit into device, due to bad blocks, imglen= 0x%lu\n",imglen);
		return -1;
	}
	printf("[%s]: %d byte write to part\n", __func__, opts->length);

	/* return happy */
	return 0;
}

/**
 * nand_write_opts: - write image to NAND flash with support for various options
 *
 * @param meminfo       AND device to erase
 * @param opts              write options (@see nand_write_options)
 * @return                      in case of success
 *
 * This code is ported from nandwrite.c from Linux mtd utils by
 * Steven J. Hill and Thomas Gleixner.
 */
int nand_write_opts(nand_info_t *meminfo, const nand_write_options_t *opts)
{
	return _nand_write_opts(meminfo, opts, 0);
}

static u32 nand_empty_check(nand_info_t *meminfo, const void *buf, u32 len)
{
	int i;

	for (i = (len >> 2) - 1; i >= 0; i--)
		if (((const uint32_t *)buf)[i] != 0xFFFFFFFF)
			break;

	/* The resulting length must be aligned to the minimum flash I/O size */
	len = ALIGN((i + 1) << 2, meminfo->oobblock + meminfo->oobsize);
	return len;
}

/**
 * nand_read_opts: - read image from NAND flash with support for various options
 *
 * @param meminfo	NAND device to erase
 * @param opts		read options (@see struct nand_read_options)
 * @return		0 in case of success
 *
 */
int nand_read_opts(nand_info_t *meminfo, nand_read_options_t *opts)
{
	u64 imglen = opts->length;
	int pagelen;
	size_t readlen;
	u64 mtdoffset;
	u_char *buffer = opts->buffer;
	int result;
	struct mtd_oob_ops tempOobOps;

	pagelen = meminfo->oobblock;
	if(opts->readoob)
		pagelen += meminfo->oobsize;

	mtdoffset = (opts->offset / pagelen) * meminfo->oobblock;

	/* check, if length is not larger than device */
	if (((imglen / pagelen) * meminfo->oobblock) > (meminfo->size - mtdoffset)){
		printf("[%s]: Image %lld bytes\n", __func__, imglen);
		printf("[%s]: OOB area %u bytes\n", __func__, meminfo->oobsize);
		printf("[%s]: NAND page %d bytes\n", __func__, pagelen);
		printf("[%s]: Device size %llu bytes\n", __func__, meminfo->size);
		printf("[%s]: Input block is larger than device!!!\n", __func__);
		return -1;
	}

	if(!opts->readoob){
		printf("[%s]: read data without oob.........\n");
		result =  meminfo->read(meminfo, mtdoffset, imglen, &readlen, buffer);
		if(result < 0){
			printf("[%s]: read at offset 0x%llX, length 0x%llX failed\n", __func__, mtdoffset, imglen);
			return -1;
		}
		printf("[%s]: read 0x%X bytes at offset 0x%llX, length 0x%llX \n", __func__, readlen, mtdoffset, imglen);
		return 0;
	}

	/* get data from input and write to the device */
	while (imglen && (mtdoffset < meminfo->size)){
		memset(&tempOobOps, 0, sizeof(tempOobOps));
		tempOobOps.datbuf = (unsigned char *) data_buf;
		tempOobOps.oobbuf = (unsigned char *)oob_buf;
		tempOobOps.len = meminfo->oobblock;
		tempOobOps.ooblen = meminfo->oobsize;
		tempOobOps.mode = MTD_OOB_RAW;

		result = meminfo->read_oob(meminfo, mtdoffset, &tempOobOps);
   		if(result < 0){
			printf("reading NAND page at offset 0x%llx failed\n", mtdoffset);
			/*
			* for programming bin, corrupted data should be avoid
			* for debug purpose, corrupted block should also be dumped
			*/
			if (opts->readoob)
				return -1;
    		}

    		if (imglen <= readlen){
    			readlen = imglen;
    		}

		memcpy(buffer, data_buf, readlen);

		buffer += readlen;
    		imglen -= readlen;
		mtdoffset += meminfo->oobblock;
	}


	if (imglen > 0){
		printf("Could not read entire image due to read length larger than device size\n");
		return -1;
	}

	/* return happy */
	return 0;
}

/**
 * nand_write_skip_bad:
 *
 * Write image to NAND flash.
 * Blocks that are marked bad are skipped and the is written to the next
 * block instead as long as the image is short enough to fit even after
 * skipping the bad blocks.
 *
 * @param nand  	NAND device
 * @param offset	offset in flash
 * @param length	buffer length
 * @param buffer        buffer to read from
 * @param withoob	whether write with yaffs format
 * @return		0 in case of success
 */
int nand_write_skip_bad(nand_info_t *nand, loff_t offset, size_t *length,
			u_char *buffer, int withoob)
{
	nand_write_options_t opts;
	U32 u32_StartSector, u32_SectorCnt;
	char cmd[512];
	char *ftl_SectorCnt, *ftl_StartSector;

	memset(&opts, 0, sizeof(nand_write_options_t));
	opts.buffer = buffer;
	opts.writeoob = withoob;
	opts.offset = offset;
	opts.length = *length;
	opts.pad = withoob ? 0 : 1;

	if(info.set_ftl == 1)
	{

		ftl_StartSector = getenv_uboot("ftl_StartSector");
		if (u32_StartSector < 0){
		printf("[%s]: please set ftl_StartSector\n", __func__);
		return -EINVAL;
		}
		u32_StartSector = atoi(ftl_StartSector);
		ftl_SectorCnt = getenv_uboot("ftl_SectorCnt");
		if (u32_SectorCnt < 0){
		printf("[%s]: please set ftl_SectorCnt\n", __func__);
		return -EINVAL;
		}
		u32_SectorCnt = atoi(ftl_SectorCnt);
		printf("this project have set ftl ftl_StartSector=%d,ftl_SectorCnt=%d\n",u32_StartSector,u32_SectorCnt);
		#if defined(__VER_UNFD_FTL__) && __VER_UNFD_FTL__
		drvNAND_WriteFAT((U32 *)buffer, u32_StartSector, u32_SectorCnt);
		#endif //__VER_UNFD_FTL__
		return 0;
	}
	else
	{
		return _nand_write_opts(nand, &opts, 0);
	}


}

/**
 * nand_read_skip_bad:
 *
 * Read image from NAND flash.
 * Blocks that are marked bad are skipped and the next block is readen
 * instead as long as the image is short enough to fit even after skipping the
 * bad blocks.
 *
 * @param nand NAND device
 * @param offset offset in flash
 * @param length buffer length, on return holds remaining bytes to read
 * @param buffer buffer to write to
 * @return 0 in case of success
 */
int nand_read_skip_bad(nand_info_t *nand, loff_t offset, size_t *length,
		       u_char *buffer)
{
	nand_read_options_t opts;

	opts.buffer = buffer;
	opts.length = *length;
	opts.offset = offset;
	opts.readoob = 0;

	return nand_read_opts(nand, &opts);
}

U32 nand_MTDMarkBad(U32  u32_GoodBlkIdx)
{
	return 0;
}

/**
 * nand_write_slc_skip_bad:
 *
 * Write image to NAND flash.
 * Blocks that are marked bad are skipped and the is written to the next
 * block instead as long as the image is short enough to fit even after
 * skipping the bad blocks.
 *
 * @param nand  	NAND device
 * @param offset	offset in flash
 * @param length	buffer length
 * @param buffer        buffer to read from
 * @param withoob	whether write with yaffs format
 * @return		0 in case of success
 */
int nand_write_slc_skip_bad(nand_info_t *nand, loff_t offset, size_t *length,
			u_char *buffer, int withoob)
{
	int rval = 0, blocksize;
	size_t left_to_write = *length;
	u_char *p_buffer = buffer;
	NAND_DRIVER *pNandDrv = (NAND_DRIVER*)drvNAND_get_DrvContext_address();
	int BlkPageCnt = (pNandDrv->u8_CellType)? pNandDrv->u16_BlkPageCnt/2 : pNandDrv->u16_BlkPageCnt;
	int PageCnt = *length >> pNandDrv->u8_PageByteCntBits;
	u_char *p_tmpbuf = pNandDrv->PlatCtx_t.pu8_PageDataBuf;
	ulong u32_Row;

	if(*length % pNandDrv->u16_PageByteCnt){
		PageCnt += 1;
		left_to_write = PageCnt << pNandDrv->u8_PageByteCntBits;
	}

	memset(p_tmpbuf, 0xa5a5, nand->writesize);

	if(pNandDrv->u8_CellType)
		blocksize = nand->erasesize/2;
	else
		blocksize = nand->erasesize;

	if((offset & (nand->erasesize -1))!= 0){
		printf("[%s]: Attempt to write non block aligned data in SLC mode\n", __func__);
		*length = 0;
		return -1;
	}
	/*
	 * nand_write() handles unaligned, partial page writes.
	 *
	 * We allow length to be unaligned, for convenience in
	 * using the $filesize variable.
	 *
	 * However, starting at an unaligned offset makes the
	 * semantics of bad block skipping ambiguous (really,
	 * you should only start a block skipping access at a
	 * partition boundary).  So don't try to handle that.
	 */
	if ((offset & (nand->writesize - 1)) != 0) {
		printf ("Attempt to write non page aligned data\n");
		*length = 0;
		return -1;
	}

	u32_Row = offset >> pNandDrv->u8_PageByteCntBits;
	if(pNandDrv->u8_CellType){
		while(PageCnt >= BlkPageCnt){
			rval = nand_write_block_slcmode(u32_Row, p_buffer, BlkPageCnt, nand_MTDMarkBad);
			if(rval){
				printf("[%s]: nand_write_block_slcmode failed\n", __func__);
				return -1;
			}
			PageCnt -= BlkPageCnt;
			p_buffer += BlkPageCnt * pNandDrv->u16_PageByteCnt;
			u32_Row += pNandDrv->u16_BlkPageCnt;
		}
		while(PageCnt){
			rval = nand_write_block_slcmode(u32_Row, p_buffer, PageCnt, nand_MTDMarkBad);
			if(rval){
				printf("[%s]: nand_write_block_slcmode failed\n", __func__);
				return -1;
			}
			p_buffer += PageCnt * pNandDrv->u16_PageByteCnt;
			PageCnt -= PageCnt;
			u32_Row += pNandDrv->u16_BlkPageCnt;
		}
	}
	else{
		rval = nand_write (nand, offset, length, buffer);
		if (rval){
			*length = 0;
			printf ("NAND write to offset %llx failed %d\n", offset, rval);
			return rval;
		}
		return 0;
	}
}

/**
 * nand_read_slc_skip_bad:
 *
 * Read image from NAND flash.
 * Blocks that are marked bad are skipped and the next block is readen
 * instead as long as the image is short enough to fit even after skipping the
 * bad blocks.
 *
 * @param nand NAND device
 * @param offset offset in flash
 * @param length buffer length, on return holds remaining bytes to read
 * @param buffer buffer to write to
 * @return 0 in case of success
 */
int nand_read_slc_skip_bad(nand_info_t *nand, loff_t offset, size_t *length,
		       u_char *buffer, int refresh)
{
	int rval, blocksize;
	size_t left_to_read = *length;
	u_char *p_buffer = buffer;
	u_char *p_Spare;
	NAND_DRIVER *pNandDrv = (NAND_DRIVER*)drvNAND_get_DrvContext_address();
	int BlkPageCnt = (pNandDrv->u8_CellType)? pNandDrv->u16_BlkPageCnt/2 : pNandDrv->u16_BlkPageCnt;
	int PageCnt = *length >> pNandDrv->u8_PageByteCntBits;
	int aligned = 1;
	ulong u32_Row;

	if(((*length) & (nand->writesize - 1)) != 0){
		PageCnt += 1;
		aligned = 0;
	}

	p_Spare = pNandDrv->PlatCtx_t.pu8_PageSpareBuf;

	if(pNandDrv->u8_CellType)
		blocksize = nand->erasesize/2;
	else
		blocksize = nand->erasesize;

	if((offset & (nand->erasesize -1)) != 0){
		printf("[%s]: Attempt to read non block aligned data in SLC mode\n", __func__);
		*length = 0;
		return -EINVAL;
	}

	if ((offset & (nand->writesize - 1)) != 0) {
		printf ("[%s]: Attempt to read non page aligned data\n", __func__);
		*length = 0;
		return -EINVAL;
	}

	if(pNandDrv->u8_CellType)
	{
		u32_Row = offset >> pNandDrv->u8_PageByteCntBits;
		while(PageCnt >= BlkPageCnt){
			int i;
			ulong u32_TmpRow;
			for(i = 0; i < BlkPageCnt; i ++){
				u32_TmpRow = u32_Row + ga_tPairedPageMap[i].u16_LSB;
				rval = drvNAND_LFSRReadPhyPage(u32_TmpRow, p_buffer, p_Spare);
				if(rval){
					printf("Nand read page failed\n");
					return rval;
				}
				p_buffer += pNandDrv->u16_PageByteCnt;
			}
			PageCnt -= BlkPageCnt;
			u32_Row += pNandDrv->u16_BlkPageCnt;
		}
		if(PageCnt){
			int i;
			ulong u32_TmpRow;
			for(i = 0; i < PageCnt; i ++){
				u32_TmpRow = u32_Row + ga_tPairedPageMap[i].u16_LSB;
				rval = drvNAND_LFSRReadPhyPage(u32_TmpRow, p_buffer, p_Spare);
				if(rval){
					printf("%s, Nand read page failed\n", __func__);
					return rval;
				}
				p_buffer += pNandDrv->u16_PageByteCnt;
			}
			PageCnt -= PageCnt;
		}
	}
	else{
		rval = nand_read (nand, offset, length, buffer);
		if (rval){
			*length = 0;
			printf ("NAND read from offset %llx failed %d\n", offset, rval);
			return rval;
		}
	}
	return 0;
}

/**
 * nand_write_partial_skip_bad:
 *
 * Write image to NAND flash.
 * Blocks that are marked bad are skipped and the is written to the next
 * block instead as long as the image is short enough to fit even after
 * skipping the bad blocks.
 *
 * @param nand  	NAND device
 * @param poffset	physical offset in flash
 * @param length	buffer length
 * @param loffset	logical offset relative to poffset
 * @param buffer        buffer to read from
 * @param withoob	whether write with yaffs format
 * @return		0 in case of success
 */
int nand_write_partial_skip_bad(nand_info_t *nand, loff_t poffset, size_t *length, loff_t loffset,
			u_char *buffer, int withoob)
{
	printf("[%s]: Not support\n", __func__);
	return 0;
}


/**
 * nand_read_partial_skip_bad:
 *
 * read image from NAND flash.
 * Blocks that are marked bad are skipped and the is read from the next
 * block instead as long as the image is short enough to fit even after
 * skipping the bad blocks.
 *
 * @param nand  	NAND device
 * @param poffset	physical offset in flash
 * @param length	buffer length
 * @param loffset	logical offset relative to poffset
 * @param buffer        buffer to read from
 * @return		0 in case of success
 */
int nand_read_partial_skip_bad(nand_info_t *nand, loff_t poffset, size_t *length, loff_t loffset,
			u_char *buffer)
{
	printf("[%s]: Not support\n", __func__);
	return 0;
}


/*
 *support env in raw nand
 *save env in nand need blocks
 */
int nand_erase_env(nand_info_t *meminfo, const nand_erase_options_t *opts)
{
	struct erase_info e_info;

	memset(&e_info, 0, sizeof(struct erase_info));
	e_info.mtd = meminfo;
	e_info.addr = opts->offset;
	e_info.len = opts->length;

	return meminfo->erase(meminfo, &e_info);
}






////////////////////////////////////////////////
#include <fcntl.h>
extern char *root_directory;
extern char *image_directory;
extern struct mtd_info mtdinfo;
void nand_util_test2(void)
{
	int fd,ret;
	unsigned char path[128];
	unsigned char *buf;
	struct stat st;
	nand_write_options_t opts;
	int len = 1024*1024;

	buf = (unsigned char *)malloc(len);

	memset(buf, 0x5A, len);
	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s%s", root_directory,image_directory, "mboot_nand1.bin");
	fd=open(path, O_RDWR);
	if(fd<0){
		printf("open %s failed\n", path);
		return -1;
	}
	stat(path, &st);

	ret= read(fd, buf, st.st_size);
	if(ret != st.st_size){
		printf("read failed ,0x%X\n, 0x%X",ret, st.st_size);
	}
	close(fd);

	opts.offset=0;
	opts.buffer=buf;
	opts.length=st.st_size;
	opts.writeoob=1;
	opts.pad=0;

	_nand_write_opts(&mtdinfo, &opts, 0);
}
