/*
 * (C) Copyright 2000-2010
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * (C) Copyright 2001 Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Andreas Heppel <aheppel@sysgo.de>
 *
 * Copyright 2011 Freescale Semiconductor, Inc.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

/*
 * Support for persistent environment data
 *
 * The "environment" is stored on external storage as a list of '\0'
 * terminated "name=value" strings. The end of the list is marked by
 * a double '\0'. The environment is preceeded by a 32 bit CRC over
 * the data part and, in case of redundant environment, a byte of
 * flags.
 *
 * This linearized representation will also be used before
 * relocation, i. e. as long as we don't have a full C runtime
 * environment. After that, we use a hash table.
 */
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

#include "tool-util.h"
#include "types.h"
#include "common.h"
#include "command.h"
#include "global_data.h"
#include "environment.h"
#include "search.h"
#include "linux/stddef.h"

DECLARE_GLOBAL_DATA_PTR;
extern struct platform_info info;

#define XMK_STR(x)	#x
#define MK_STR(x)	XMK_STR(x)

#define CONFIG_SYS_MAXARGS 32

/*
 * Maximum expected input data size for import command
 */
#define	MAX_ENV_SIZE	(1 << 20)	/* 1 MiB */

/*
 * This variable is incremented on each do_env_set(), so it can
 * be used via get_env_id() as an indication, if the environment
 * has changed or not. So it is possible to reread an environment
 * variable only if the environment was changed ... done so for
 * example in NetInitLoop()
 */
static int env_id = 1;

int get_env_id(void)
{
	return env_id;
}

/*
 * Command interface: print one or all environment variables
 *
 * Returns 0 in case of error, or length of printed string
 */
static int env_print(char *name)
{
	char *res = NULL;
	size_t len;

	if (name) {		/* print a single name */
		ENTRY e, *ep;

		e.key = name;
		e.data = NULL;
		hsearch_r(e, FIND, &ep, &env_htab);
		if (ep == NULL)
			return 0;
		len = printf("%s=%s\n", ep->key, ep->data);
		return len;
	}

	/* print whole list */
	len = hexport_r(&env_htab, '\n', &res, 0);

	if (len > 0) {
		puts(res);
		free(res);
		return len;
	}

	/* should never happen */
	return 0;
}

int do_env_print (cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int i;
	int rcode = 0;

	if (argc == 1) {
		/* print all env vars */
		rcode = env_print(NULL);
		if (!rcode)
			return 1;
		printf("\nEnvironment size: %d/%ld bytes\n",
			rcode, (ulong)ENV_SIZE);
		return 0;
	}

	/* print selected env vars */
	for (i = 1; i < argc; ++i) {
		int rc = env_print(argv[i]);
		if (!rc) {
			printf("## Error: \"%s\" not defined\n", argv[i]);
			++rcode;
		}
	}

	return rcode;
}


/*
 * Set a new environment variable,
 * or replace or delete an existing one.
 */

int _do_env_set (int flag, int argc, char * const argv[])
{
	int   i, len;
	int   console = -1;
	char  *name, *value, *s;
	ENTRY e, *ep;

	name = argv[1];

	if (strchr(name, '=')) {
		printf("## Error: illegal character '=' in variable name \"%s\"\n", name);
		return 1;
	}
	env_id++;
	/*
	 * search if variable with this name already exists
	 */
	e.key = name;
	e.data = NULL;

	hsearch_r(e, FIND, &ep, &env_htab);
	/* Delete only ? */
	if ((argc < 3) || argv[2] == NULL) {
#if 1 // MStar: always return 0 when deleting env
		hdelete_r(name, &env_htab);
		return 0;
#else
		int rc = hdelete_r(name, &env_htab);
		return !rc;
#endif
	}

	/*
	 * Insert / replace new value
	 */
	for (i = 2, len = 0; i < argc; ++i)
		len += strlen(argv[i]) + 1;

	value = malloc(len);
	if (value == NULL) {
		printf("## Can't malloc %d bytes\n", len);
		return 1;
	}
	for (i = 2, s = value; i < argc; ++i) {
		char *v = argv[i];

		while ((*s++ = *v++) != '\0')
			;
		*(s-1) = ' ';
	}
	if (s != value)
		*--s = '\0';

	e.key  = name;
	e.data = value;
	hsearch_r(e, ENTER, &ep, &env_htab);
	free(value);
	if (!ep) {
		printf("## Error inserting \"%s\" variable, errno=%d\n",
			name, errno);
		return 1;
	}

	/*
	 * Some variables should be updated when the corresponding
	 * entry in the environment is changed
	 */
	return 0;
}

int setenv_uboot(char *varname, char *varvalue)
{
	char * const argv[4] = { "setenv", varname, varvalue, NULL };

	if ((varvalue == NULL) || (varvalue[0] == '\0'))
		return _do_env_set(0, 2, argv);
	else
		return _do_env_set(0, 3, argv);
}

int do_env_set(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	if (argc < 2)
		return cmd_usage(cmdtp);

	return _do_env_set(flag, argc, argv);
}

/*
 * Match a name / name=value pair
 *
 * s1 is either a simple 'name', or a 'name=value' pair.
 * i2 is the environment index for a 'name2=value2' pair.
 * If the names match, return the index for the value2, else NULL.
 */
int envmatch(uchar *s1, int i2)
{
	while (*s1 == env_get_char(i2++))
		if (*s1++ == '=')
			return i2;
	if (*s1 == '\0' && env_get_char(i2-1) == '=')
		return i2;
	return -1;
}

/*
 * Look up variable from environment,
 * return address of storage for that variable,
 * or NULL if not found
 */
char *getenv_uboot(char *name)
{
	if (gd->flags & GD_FLG_ENV_READY) {	/* after import into hashtable */
		ENTRY e, *ep;

		e.key  = name;
		e.data = NULL;
		hsearch_r(e, FIND, &ep, &env_htab);

		return ep ? ep->data : NULL;
	}

	/* restricted capabilities before import */

	if (getenv_f(name, (char *)(gd->env_buf), sizeof(gd->env_buf)) > 0)
		return (char *)(gd->env_buf);

	return NULL;
}

/*
 * Look up variable from environment for restricted C runtime env.
 */
int getenv_f(char *name, char *buf, unsigned len)
{
	int i, nxt;

	for (i = 0; env_get_char(i) != '\0'; i = nxt+1) {
		int val, n;

		for (nxt = i; env_get_char(nxt) != '\0'; ++nxt) {
			if (nxt >= CONFIG_ENV_SIZE)
				return -1;
		}

		val = envmatch((uchar *)name, i);
		if (val < 0)
			continue;

		/* found; copy out */
		for (n = 0; n < len; ++n, ++buf) {
			if ((*buf = env_get_char(val++)) == '\0')
				return n;
		}

		if (n)
			*--buf = '\0';

		printf("env_buf [%d bytes] too small for value of \"%s\"\n",
			len, name);

		return n;
	}
	return -1;
}

int do_env_save(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int ret = 0;
	extern char *env_name_spec_ubi;
//	extern char *env_name_spec_nand;

	if(info.envtype == UBIENV){
		printf("Saving Environment to %s...\n", env_name_spec_ubi);
		ret = saveenv_ubi();
		if(ret < 0)
			printf("[%s]: save env failed\n", __func__);
	}
	else if(info.envtype == NANDENV){
		printf("Saving Environment to nand...\n");
		ret = saveenv_nand();
		if(ret < 0)
			printf("[%s]: save env failed\n", __func__);
	}
	else{
		printf("Saving Environment to unkown media, to be continue\n");
	}
	return ret;
}

U_BOOT_CMD(
	saveenv, 1, 0,	do_env_save,
	"save environment variables to persistent storage",
	""
);

/*
 * New command line interface: "env" command with subcommands
 */
static cmd_tbl_t cmd_env_sub[] = {
	U_BOOT_CMD_MKENT(save, 1, 0, do_env_save, "", ""),
	U_BOOT_CMD_MKENT(set, CONFIG_SYS_MAXARGS, 0, do_env_set, "", ""),
};


U_BOOT_CMD_COMPLETE(
	printenv, CONFIG_SYS_MAXARGS, 1,	do_env_print,
	"print environment variables",
	"\n    - print values of all environment variables\n"
	"printenv name ...\n"
	"    - print value of environment variable 'name'",
	var_complete
);


U_BOOT_CMD_COMPLETE(
	setenv, CONFIG_SYS_MAXARGS, 0,	do_env_set,
	"set environment variables",
	"name value ...\n"
	"    - set environment variable 'name' to 'value ...'\n"
	"setenv name\n"
	"    - delete environment variable 'name'",
	var_complete
);
