/*
 * (C) Copyright 2003
 * Kyle Harris, kharris@nexus-tech.net
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

 /*Many of the following code comes from emmcnize.c & mmc.c & eMMC.h--xiaohui.zhu, 2014.2.20*/

#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <malloc.h>
#include <stdint.h>
#include <sys/stat.h>
#include <asm/byteorder.h>

#include "dictionary.h"
#include "libiniparser.h"
#include "command.h"
#include "part.h"
#include "mmc.h"
#include "part_emmc.h"

#ifndef U32
#define U32  unsigned long
#endif
#ifndef U16
#define U16  unsigned short
#endif
#ifndef U8
#define U8   unsigned char
#endif
#ifndef S32
#define S32  signed long
#endif
#ifndef S16
#define S16  signed short
#endif
#ifndef S8
#define S8   signed char
#endif

#define EMMC_BLK_SZ                  512
#define EMMC_RW_SHIFT             9
#define FAT_BUFFER_LEN            0x9AB000
#define EMMC_PARTITION_NAME "partitiontbl.bin"
#define EMMC_BIN_NAME             "emmc.bin"
#define EMMC_MBOOT_NAME        "MBOOT.bin"
#define EMMC_MPOOL_NAME        "MPOOL.bin"
#define EMMC_MBOOT_NAME_OUT        "MBOOT.bin.tmp"
#define EMMC_MPOOL_NAME_OUT        "MPOOL.bin.tmp"
#define EMMC_BOOT1_NAME        "boot1.bin"
#define EMMC_BOOT2_NAME        "boot2.bin"
#define EMMC_BOOT_NAME          "boot.bin"  // include boot1 and boot2
#define EMMC_CRC32_INIT          0xFFFFFFFFU
#define TRIM_START_OFFSET	   (0x700000/512)

#define EMMC_RESERVED_FOR_MAP               10
#define EMMC_RESERVED_FOR_MAP_V2         63   //0~32kb for partition table. 32kb~64kb reserved for partition table, but not used
#define EMMC_CHUNK_HEAD_START              4096        // chunk head start addr
#define EMMC_PARTITION_START_DEFAULT 14336       // ordinary partition data start addr
#define EMMC_RESERVED_FOR_KL                  8192        /* reserve 4M space for kernel */
#define EMMC_DRIVER_MAGIC                        0x1630
#define EMMC_PARTITIONTABLE_VERSION2  0x2000
#define EMMC_MBOOT_START                         EMMC_CHUNK_HEAD_START
#define EMMC_MBOOT_SIZE                            6144          /* 3M space for mboot partition */
#define EMMC_MPOOL_START                         (EMMC_MBOOT_START+EMMC_MBOOT_SIZE)
#define EMMC_MPOOL_SIZE                            4096

#define EMMC_PARTITION_START partition_start 

#define EMMC_LZO_BUF_SIZE 0x4000000
#define CONFIG_UNLZO_DST_ADDR (emmc_lzobuf)

typedef struct _eMMC_DRIVER
{
	// ExtCSD
	U32 u32_SEC_COUNT;
	U32 u32_BOOT_SEC_COUNT;
} eMMC_DRIVER, *P_eMMC_DRIVER;
eMMC_DRIVER g_eMMCDrv;

struct emmc_bintool_part
{
	int idx;
	int blocks;
	emmc_partition_t *emmc_part;
};

static int emmc_parts_num = 0;
static char *emmc_outpath;
static int bfe = 1;
static int ispad = 0;
static unsigned char padvalue = 0;
static u32 unlzo_partition_blkcnt = 0;
static int curr_device = 0;
static int cur_dev_num = 0;
static struct list_head mmc_devices;
static emmc_driver_desc_t *emmc_driver;
static struct emmc_bintool_part *emmc_partition_tbl = NULL;
static struct emmc_bintool_part *emmc_partition_mboot = NULL;
static struct emmc_bintool_part *emmc_partition_mpool = NULL;
static struct emmc_bintool_part **emmc_parts = NULL;
static unsigned char *emmc_lzobuf = NULL;
static int partition_start;

extern dictionary *dict;
extern char *outpath;
extern char *root_directory;
extern char *image_directory;

extern int lzop_decompress_part(const unsigned char *src, size_t src_len, unsigned char *dst, size_t *dst_len, size_t *src_alignlen, int part);

extern int search_tftp_buffer(unsigned long * addr);
extern void release_tftp_buffer(unsigned long addr);

int do_write_miu2emmc(U32 bfn_blk,U32 bfn_addr)
{}

int do_write_ldr2emmc(U32 bfn_blk,U32 bfn_addr)
{}

int do_write_app2emmc(U32 bfn_blk,U32 bfn_addr)
{}

void do_set_emmc_partition_start(void)
{
	char *val;
	int pstart = EMMC_PARTITION_START_DEFAULT;

	val = iniparser_getstring(dict, "boot:start", "Not Set");
	if(strcmp(val, "Not Set") == 0){
		printf("[%s]: User partition start blk is not set, using default %d\n", __func__, pstart);
	}
	else{
		pstart = iniparser_getint(dict, "boot:start", -1);
		printf("[%s]: User partition start blk is set to %d\n", __func__, pstart);
	}
	partition_start = pstart;
}

U32 eMMC_WriteBootPart(U8* pu8_DataBuf, U32 u32_DataByteCnt, U32 u32_BlkAddr, U8 u8_PartNo)
{
	int fd, ret;
	char path[128];

	memset(path, 0, sizeof(path));

	if(u8_PartNo == 1)
		sprintf(path, "%s%s", emmc_outpath, EMMC_BOOT1_NAME);
	else if(u8_PartNo == 2)
		sprintf(path, "%s%s", emmc_outpath, EMMC_BOOT2_NAME);
	else{
		printf("[%s]: Invalid boot part number %d\n", __func__, u8_PartNo);
		return 1;
	}

	fd = open(path, O_RDWR);
	if(fd < 0){
		printf("[%s]: open %s failed\n", __func__, path);
		return 1;
	}
	
	lseek(fd, u32_BlkAddr * EMMC_BLK_SZ, SEEK_SET);

	ret = write(fd, pu8_DataBuf, u32_DataByteCnt);
	if(ret < 0){
		printf("[%s]: write boot part %d failed, offset block=%d\n", __func__, u8_PartNo, u32_BlkAddr);
		close(fd);
		return 1;
	}
	close(fd);
	return 0;
}

U32 eMMC_WriteGPPart(U8* pu8_DataBuf, U32 u32_DataByteCnt, U32 u32_BlkAddr, U8 u8_PartNo)
{}

struct mmc *find_mmc_device(int dev_num)
{
	struct mmc *m;
	struct list_head *entry;

	list_for_each(entry, &mmc_devices) {
		m = list_entry(entry, struct mmc, link);

		if (m->block_dev.dev == dev_num)
			return m;
	}

	printf("MMC Device %d not found\n", dev_num);

	return NULL;
}

int mmc_init(struct mmc *mmc)
{
	mmc->has_init = 1;
	return 0;
}

int get_mmc_num(void)
{
	return cur_dev_num;
}

int get_mmc_partsize (char *puPartName,unsigned int *u32PartSize)
{
	block_dev_desc_t *mmc_dev;
	struct mmc *mmc = NULL;

	if (curr_device < 0){
		if (get_mmc_num() > 0)
			curr_device = 0;
		else{
			puts("No MMC device available\n");
			return 1;
		}
	}

	mmc = find_mmc_device(curr_device);

	if (!mmc) {
		printf("no mmc device at slot %x\n", curr_device);
		return -1;
	}
	if(!mmc->has_init){
		printf("Do mmc init first!\n");
		mmc_init(mmc);
		mmc->has_init = 1;
	}
	mmc_dev = mmc_get_dev(curr_device);
	if (mmc_dev != NULL && mmc_dev->type != DEV_TYPE_UNKNOWN) {
		int i;
		for(i = 0; i < EMMC_RESERVED_FOR_MAP; i++){
			if(emmc_parts[i]&& strcmp(puPartName, emmc_parts[i]->emmc_part->name) == 0){
				*u32PartSize = emmc_parts[i]->emmc_part->block_count * EMMC_BLK_SZ;
				return 0;
			}
		}
	}
	return -1;
}


block_dev_desc_t *mmc_get_dev(int dev)
{
	struct mmc *mmc = find_mmc_device(dev);

	return mmc ? &mmc->block_dev : NULL;
}

static u32 do_mmc_write_emptyskip(struct mmc *mmc, s32 start_blk, u32 cnt_blk,
	                                        const void *buf, u32 empty_skip)
{
	s32 i;
	int fd, ret;
	char path[128];
	
	for(i = 0; i < emmc_parts_num; i++){
		if(emmc_parts[i]){
			if(start_blk >= emmc_parts[i]->emmc_part->start_block && 
				start_blk < emmc_parts[i]->emmc_part->start_block + emmc_parts[i]->emmc_part->block_count){
				break;
			}
		}
	}
	if(i == emmc_parts_num){
		printf("[%s]: Cannot find corresponding part\n", __func__);
		return 1;
	}

	if(start_blk + cnt_blk > emmc_parts[i]->emmc_part->start_block + emmc_parts[i]->emmc_part->block_count){
		printf("[%s]: write size exceeds partition size\n", __func__);
		return 1;
	}

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s%s", emmc_outpath, emmc_parts[i]->emmc_part->name, ".bin.tmp");
	fd = open(path, O_RDWR | O_APPEND);
	if(fd < 0){
		printf("[%s]: open file %s failed\n", __func__, path);
		printf("[%s]:partition %s is not set yet\n", __func__, emmc_parts[i]->emmc_part->name);
		return -1;
	}

	lseek(fd, (start_blk - emmc_parts[i]->emmc_part->start_block) * EMMC_BLK_SZ, SEEK_SET);

	
	for(i = 0; i < cnt_blk; i++){
		ret = write(fd, buf + i * EMMC_BLK_SZ, EMMC_BLK_SZ);
		if(ret < 0){
			printf("[%s]: emmc write blk %d failed\n", __func__, start_blk + i);
			return 1;
		}
	}

	close(fd);

	return i;
}

static int do_unlzo (struct mmc *mmc, int argc, char * const argv[])
{
	int ret=0, cnt, cnt_total=0;
	unsigned char *AddrSrc=NULL, *AddrDst=NULL;
	size_t LengthSrc=0,  LengthDst=0;
	size_t LenTail=0, LenSpl=0, LenSrcAlign=0;
	disk_partition_t dpt;
	block_dev_desc_t *mmc_dev;
	s32 blk = -1, partnum = 1, n;
	u32 empty_skip = 0, partition_blkcnt = 0;
	char* cmdtail = strchr(argv[1], '.');
	unsigned long TempAddr;

	AddrSrc = (unsigned char *) strtoul(argv[2], NULL, 16);
	LengthSrc = (size_t) strtol(argv[3], NULL, 16);

	TempAddr = (unsigned long)AddrSrc;
	ret = search_tftp_buffer((unsigned long *)&AddrSrc);
	if(ret)
		printf("[%s]: find bootloader tftp buffer, add=0x%lx\n", __func__, (unsigned long)AddrSrc);
	else
		printf("[%s]: find NO bootloader tftp buffer, using original addr=0x%lx\n", __func__, (unsigned long)AddrSrc);


	mmc_dev = mmc_get_dev(curr_device);
	if ((mmc_dev == NULL) || (mmc_dev->type == DEV_TYPE_UNKNOWN)) {
		printf("no mmc device found!\n");
		return 1;
	}

	//Get the partition offset from partition name
	for(;;){
		if(get_partition_info_emmc(mmc_dev, partnum, &dpt))
			break;
		if(!strcmp(argv[4], (const char *)dpt.name)){
			blk = dpt.start;
			partition_blkcnt = dpt.size;
			break;
		}
		partnum++;
	}

	if(blk < 0){
		printf("ERR:Please check the partition name!\n");
		return 1;
	}

	if ((!cmdtail) || (strncmp(cmdtail, ".cont", 5)))
		unlzo_partition_blkcnt = 0;

	if(unlzo_partition_blkcnt >= partition_blkcnt){
		printf("ERR: The partition unlzo write has been done!\n");
		return 1;
	}

	blk += unlzo_partition_blkcnt;

	AddrDst = (unsigned char *) CONFIG_UNLZO_DST_ADDR;

	printf ("   Uncompressing ... \n");

	ret = lzop_decompress_part ((const unsigned char *)AddrSrc, LengthSrc,
		(unsigned char *)AddrDst, &LengthDst, &LenSrcAlign, 0);

	if (ret) {
		printf("LZO: uncompress, out-of-mem or overwrite error %d\n", ret);
		return 1;
	}

	if (argc == 6){
		empty_skip = strtoul(argv[5], NULL, 16);
	}

	/* We assume that the decompressed file is aligned to mmc block size
	when complete decompressing */
	cnt = LengthDst >> EMMC_RW_SHIFT;

	//n = mmc->block_dev.block_write(curr_device, blk, cnt, AddrDst);
	n = do_mmc_write_emptyskip(mmc, blk, cnt, AddrDst, empty_skip);
	if(n == cnt)
		cnt_total += cnt;
	else{
		printf("%d blocks written error at %d 1\n", cnt, blk);
		return 1;
	}

	/* If the decompressed file is not aligned to mmc block size, we should
	split the not aligned tail and write it in the next loop */
	LenTail = LengthDst & (EMMC_BLK_SZ - 1);

	if(LenTail){
		memcpy((unsigned char *) CONFIG_UNLZO_DST_ADDR,
			(const unsigned char *) (AddrDst + LengthDst - LenTail), LenTail);
		AddrDst = (unsigned char *) (CONFIG_UNLZO_DST_ADDR + LenTail);
	}else
		AddrDst = (unsigned char *) CONFIG_UNLZO_DST_ADDR;

	if(LenSrcAlign == LengthSrc)
		goto done;

	//Move the source address to the right offset
	AddrSrc += LenSrcAlign;

	printf("    Continue uncompressing and writing emmc...\n");

	for(;;){
		LengthDst = 0;
		ret = lzop_decompress_part ((const unsigned char *)AddrSrc, LengthSrc,
			(unsigned char *)AddrDst, &LengthDst, &LenSrcAlign, 1);
		if (ret) {
			printf("[%s]: LZO: uncompress, out-of-mem or overwrite error %d\n", __func__, ret);
			return 1;
		}

		LenSpl = LenTail + LengthDst;
		cnt = LenSpl >> EMMC_RW_SHIFT;

		if(cnt){
			//n = mmc->block_dev.block_write(curr_device, (blk+cnt_total), cnt, (const unsigned char *)CONFIG_UNLZO_DST_ADDR);
			n = do_mmc_write_emptyskip(mmc, (blk+cnt_total), cnt, (const unsigned char *)CONFIG_UNLZO_DST_ADDR, empty_skip);
			if(n == cnt)
				cnt_total += cnt;
			else{
				printf("[%s]: %d blocks written error at %d 2\n", __func__, cnt, (blk+cnt_total));
				return 1;
			}
		}

		LenTail = LenSpl & (EMMC_BLK_SZ - 1);
		if(LenTail){
			memcpy((unsigned char *) CONFIG_UNLZO_DST_ADDR,
				(const unsigned char *) (AddrDst + LengthDst - LenTail), LenTail);
			AddrDst = (unsigned char *) (CONFIG_UNLZO_DST_ADDR + LenTail);
		}else
			AddrDst = (unsigned char *) CONFIG_UNLZO_DST_ADDR;

		if(LenSrcAlign == LengthSrc)
			break;

		AddrSrc += LenSrcAlign;
	}

done:

	if(LenTail){
		if(1 != mmc->block_dev.block_write(0, (blk + cnt_total),
			1, (const unsigned char *)CONFIG_UNLZO_DST_ADDR))
		{
			printf("%d blocks written error at %d 3\n", cnt, blk);
			return 1;
		}
		cnt_total++;
	}

	unlzo_partition_blkcnt += cnt_total;

	/*
	 *
	 *Calc process block for unlzo partition,add by xiaohui.zhu, 2014.3.3
	 *
	 */
	emmc_parts[partnum - 1]->blocks += cnt_total;

	release_tftp_buffer(TempAddr);

	printf("    Depressed OK! Write to %s partition OK!\nTotal write size: 0x%0x\n",
		argv[4], cnt_total << EMMC_RW_SHIFT);

	return 0;
}

int get_partition_info_emmc (block_dev_desc_t *dev_desc, int part, disk_partition_t *info)
{
	if(part < 1 || part > EMMC_RESERVED_FOR_MAP_V2){
		printf("[%s]: part number %d Not in out of bound\n", __func__, part);
		return -1;
	}

	info->blksz = EMMC_BLK_SZ;
	info->size= emmc_parts[part - 1]->emmc_part->block_count;
	info->start = emmc_parts[part - 1]->emmc_part->start_block;
	memcpy(info->name, emmc_parts[part - 1]->emmc_part->name, 32);
	memcpy(info->type, emmc_parts[part - 1]->emmc_part->type, 32);

	return 0;
}

int do_mmcops(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	if (argc < 2)
		return cmd_usage(cmdtp);

	if (curr_device < 0) {
		if (get_mmc_num() > 0)
			curr_device = 0;
		else {
			puts("No MMC device available\n");
			return 1;
		}
	}

        /*
        * Syntax is:
        *   0    1   2             3          4
        *   mmc  bfe [miu/ldr/app] [blk addr] [miu address]
        */
	if (strcmp(argv[1], "bfe") == 0){
		printf("[%s]: Cmd %s %s to be continue\n", __func__, argv[0], argv[1]);
		return 0;
	} 
	else if(strncmp(argv[1], "read", 4) == 0) {
		printf("[%s]: Command emmc read has not been implented...\n", __func__);
		return 0;
	} 
	else if(strncmp(argv[1], "write", 5) == 0) {
		void *addr = (void *)strtoul(argv[2], NULL, 16);
		u32 n, cnt, partnum = 1, empty_skip = 0, cont = 0;
		char* cmdtail = strchr(argv[1], '.');
		char* cmdlasttail = strrchr(argv[1], '.');
		block_dev_desc_t *mmc_dev;
		disk_partition_t dpt;
		int ret;
		unsigned long TempAddr;

		s32 blk = -1;

		struct mmc *mmc = find_mmc_device(curr_device);
		if (!mmc) {
			printf("no mmc device at slot %x\n", curr_device);
			return 1;
		}

		cnt = ALIGN(strtoul(argv[4], NULL, 16), 512) / 512;
		if(!mmc->has_init){
			printf("Do mmc init first!\n");
			mmc_init(mmc);
			mmc->has_init = 1;
		}

		if((cmdtail)&&(!strncmp(cmdtail, ".p", 2))){
			mmc_dev = mmc_get_dev(curr_device);
			if ((mmc_dev == NULL) || (mmc_dev->type == DEV_TYPE_UNKNOWN)) {
				printf("no mmc device found!\n");
				return 1;
			}

			for(;;){
				if(get_partition_info_emmc(mmc_dev, partnum, &dpt) < 0)
					return -1;
				
				if(!strcmp(argv[3], (const char *)dpt.name)){
					blk = dpt.start;
					if(!strncmp(cmdlasttail, ".continue", 9)){
						blk += strtoul(argv[4], NULL, 16);
						cnt = ALIGN(strtoul(argv[5], NULL, 16), 512) / 512;
						cont = 1;
					}
					
					//Important, calc process block here
					emmc_parts[partnum - 1]->blocks += cnt;
					break;
				}
				partnum++;
			}
		}
		else if (((cmdtail)&&(!strncmp(cmdtail, ".boot", 5))) || ((cmdtail)&&(!strncmp(cmdtail, ".gp", 3)))){
			addr = (void *)strtoul(argv[3], NULL, 16);
			blk = strtoul(argv[4], NULL, 16);
			cnt = ALIGN(strtoul(argv[5], NULL, 16), 512) / 512;
		}
		else
			blk = strtoul(argv[3], NULL, 16);

		if(blk < 0){
			printf("[%s]: ERR:Please check the blk# or partiton name!\n", __func__);
			return 1;
		}

		TempAddr = (unsigned long)addr;
		ret = search_tftp_buffer((unsigned long *)&addr);
		if(ret)
			printf("[%s]: find bootloader tftp buffer, add=0x%lx\n", __func__, (unsigned long)addr);
		else
			printf("[%s]: find NO bootloader tftp buffer, using original addr=0x%lx\n", __func__, (unsigned long)addr);


		if ((cmdtail)&&(!strncmp(cmdtail, ".boot", 5))){
			if(cnt > g_eMMCDrv.u32_BOOT_SEC_COUNT){
				printf("\n eMMC not enough boot partition size for write!");
				return 1;
			}

			if (strncmp(argv[2], "1", 1) == 0)
				n = eMMC_WriteBootPart(addr, cnt << 9, blk, 1);
			else if (strncmp(argv[2], "2", 1) == 0)
				n = eMMC_WriteBootPart(addr, cnt << 9, blk, 2);
			else {
				printf("mmc access boot partition parameter not found!\n");
				return 1;
			}

			n = (n == 0) ? cnt : -1;
		}
		else if ((cmdtail)&&(!strncmp(cmdtail, ".gp", 3))){
			int gp_part;

			gp_part =  strtoul(argv[2], NULL, 10);
			if(gp_part >= 0 && gp_part < 4)
			n = eMMC_WriteGPPart(addr, 512, blk, gp_part);
			else{
				printf("mmc access GP partition No. %d is invalid!\n", gp_part);
				return 1;
			}
			n = (n == 0) ? cnt : -1;
		}
		else{
			if ((argc == 6) && (cont == 0)){
				empty_skip = strtoul(argv[5], NULL, 16);
			}
			if ((argc == 7) && (cont == 1)){
				empty_skip = strtoul(argv[6], NULL, 16);
			}

			n = do_mmc_write_emptyskip(mmc, blk, cnt, addr, empty_skip);
		}

		release_tftp_buffer(TempAddr);

		printf("%d blocks written: %s\n", n, (n == cnt) ? "OK" : "ERROR");
		return (n == cnt) ? 0 : 1;
	}
	else if(strncmp(argv[1], "create", 6) == 0) {
		unsigned long size;
		off_t off;
		char *endp;
		int fd, i, k, ret;
		char path[128];
		int pstart=EMMC_PARTITION_START;
		
		for(i = 0; i < EMMC_RESERVED_FOR_MAP_V2; i++){
			if(!emmc_parts[i])
				break;
			else{
				if(strcmp(emmc_parts[i]->emmc_part->name, argv[2]) == 0){
					printf("[%s]: partition %s alread exist\n", __func__, argv[2]);
					return -1;
				}
			}
		}
		if(i == EMMC_RESERVED_FOR_MAP_V2){
			printf("[%s]: No available emmc part\n", __func__);
			return -1;
		}
				
		emmc_parts[i] = (struct emmc_bintool_part *)malloc(sizeof(struct emmc_bintool_part));
		if(!emmc_parts[i]){
			printf("[%s]: malloc emmc part failed\n", __func__);
			return -1;
		}
		memset(emmc_parts[i], 0, sizeof(struct emmc_bintool_part));
		emmc_parts[i]->emmc_part = (emmc_partition_t *)malloc(sizeof(emmc_partition_t));
		if(!emmc_parts[i]->emmc_part){
			printf("[%s]: malloc emmc part failed\n", __func__);
			return -1;
		}
		memset(emmc_parts[i]->emmc_part, 0, sizeof(emmc_partition_t));
		//Do partition tbl write ops
		strcpy((char *)emmc_parts[i]->emmc_part->name, argv[2]);

		size = strtoul(argv[3], &endp, 16);
		emmc_parts[i]->emmc_part->block_count = size / 512;
		emmc_parts[i]->emmc_part->signature = EMMC_PARTITION_MAGIC;

		if(strcmp("MBOOT", (const char *)emmc_parts[i]->emmc_part->name) &&
			strcmp("mboot", (const char *)emmc_parts[i]->emmc_part->name) &&
			strcmp("MPOOL", (const char *)emmc_parts[i]->emmc_part->name) &&
			strcmp("mpool", (const char *)emmc_parts[i]->emmc_part->name)){
			emmc_parts[i]->emmc_part->start_block = pstart;
			for(k=0; k<emmc_parts_num; k++){
				if(emmc_parts[k]){
					if(strcmp("MBOOT", (const char *)emmc_parts[k]->emmc_part->name) &&
						strcmp("mboot", (const char *)emmc_parts[k]->emmc_part->name) &&
						strcmp("MPOOL", (const char *)emmc_parts[k]->emmc_part->name) &&
						strcmp("mpool", (const char *)emmc_parts[k]->emmc_part->name))
						emmc_parts[i]->emmc_part->start_block += emmc_parts[k]->emmc_part->block_count;
				}
			}
		}
		else if(!strcmp("MBOOT", (const char *)emmc_parts[i]->emmc_part->name) ||
			!strcmp("mboot", (const char *)emmc_parts[i]->emmc_part->name)){
			strcpy((char *)emmc_parts[i]->emmc_part->name, "MBOOT");
			emmc_parts[i]->emmc_part->start_block = EMMC_MBOOT_START;
		}
		else if(!strcmp("MPOOL", (const char *)emmc_parts[i]->emmc_part->name) ||
			!strcmp("mpool", (const char *)emmc_parts[i]->emmc_part->name)){
			strcpy((char *)emmc_parts[i]->emmc_part->name, "MPOOL");
			emmc_parts[i]->emmc_part->start_block = EMMC_MPOOL_START;
		}
		
		memset(path, 0, sizeof(path));
		sprintf(path, "%s%s%s", emmc_outpath, (char *)emmc_parts[i]->emmc_part->name, ".bin.tmp");
		fd = open(path, O_CREAT | O_TRUNC | O_RDWR, S_IWUSR | S_IRUSR | S_IRGRP | S_IWGRP | S_IROTH);
		if(fd < 0){
			printf("[%s]: open file %d failed\n", __func__, path);
			return -1;
		}
		close(fd);

		emmc_parts_num++;

		printf("[%s]: Create emmc partition %s, idx=%d, start block=0x%X, block num=0x%X\n", 
			__func__, 
			argv[2], i, 
			emmc_parts[i]->emmc_part->start_block, 
			emmc_parts[i]->emmc_part->block_count);
		
		return 0;
	}
	else if(strncmp(argv[1], "rmgpt", 5) == 0) {
		printf("[%s]: remove GPT, Success\n", __func__);
		return 0;
	} else if(strncmp(argv[1], "unlzo", 5) == 0) {
		struct mmc *mmc = find_mmc_device(curr_device);
		int ret=0;

		if (!mmc) {
			printf("no mmc device at slot %x\n", curr_device);
			return 1;
		}

		if(!mmc->has_init){
			printf("Do mmc init first!\n");
			mmc_init(mmc);
			mmc->has_init = 1;
		}

		if (argc < 5) {
			printf ("Usage:\n%s\n", cmdtp->usage);
			return 1;
		}

		ret = do_unlzo(mmc, argc, argv);
		return ret;
	}
	else{
		printf("[%s]: Cmd %s %s is NOT support now\n", __func__, argv[0], argv[1]);
		return 0;
	}
	return cmd_usage(cmdtp);
}

int emmc_create_partition_tbl(void)
{
	int fd, i, ret;
	char path[128];
	char buf[512];

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s", emmc_outpath, EMMC_PARTITION_NAME);
	fd = open(path, O_RDWR | O_CREAT | O_TRUNC, S_IWUSR | S_IRUSR | S_IRGRP | S_IWGRP | S_IROTH);
	if(fd < 0){
		printf("[%s]: open file %s failed\n", __func__, path);
		return -1;
	}

	memset(buf, 0xFF, EMMC_BLK_SZ);
	for(i = 0; i < EMMC_RESERVED_FOR_MAP_V2 + 1; i++){
		ret = write(fd, buf, EMMC_BLK_SZ);
		if(ret < 0){
			printf("[%s]: Create empty emmc partition table %s failed\n", __func__, path);
			return -1;
		}
	}

	lseek(fd, 0, SEEK_SET);
	
	ret = write(fd, emmc_driver, sizeof(emmc_driver_desc_t));
	if(ret < 0){
		printf("[%s]: write driver descriptor failed\n", __func__);
		return -1;
	}

	for(i = 0; i < EMMC_RESERVED_FOR_MAP_V2; i++){
		if(emmc_parts[i]){
			ret = write(fd, emmc_parts[i]->emmc_part, sizeof(emmc_partition_t));
			if(ret < 0){
				printf("[%s]: write partition table for part %s failed\n", __func__, emmc_parts[i]->emmc_part->name);
				return -1;
			}
		}
	}

	close(fd);
	return 0;
}

int emmc_create_partition_mpool(void)
{
	int ret = 0, len, size, fd_i, fd_o;
	char *buf;
	char path[128];
	struct stat st;

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s%s", root_directory, image_directory, EMMC_MPOOL_NAME);
	fd_i = open(path, O_RDWR);
	if(fd_i < 0){
		printf("[%s]: open %s failed\n", __func__, path);
		ret = -1;
		goto out1;
	}
	stat(path, &st);
	len = st.st_size;

	emmc_parts[1]->blocks = (st.st_size + EMMC_BLK_SZ - 1) / EMMC_BLK_SZ;

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s", emmc_outpath, EMMC_MPOOL_NAME_OUT);
	fd_o = open(path, O_CREAT | O_TRUNC | O_RDWR, S_IRWXU | S_IRWXG | S_IRWXO);
	if(fd_i < 0){
		printf("[%s]: open %s failed\n", __func__, path);
		ret = -1;
		goto out2;
	}

	buf = (char *)malloc(EMMC_BLK_SZ);
	if(!buf){
		printf("[%s]: malloc failed\n", __func__);
		ret = -1;
		goto out3;
	}

	while(len > 0){
		size = EMMC_BLK_SZ;
		if(size > len){
			memset(buf, padvalue, EMMC_BLK_SZ);
			size = len;
		}
		ret = read(fd_i, buf, size);
		if(ret < 0){
			printf("[%s]: Read failed\n", __func__);
			goto out4;
		}
		ret = write(fd_o, buf, EMMC_BLK_SZ);
		if(ret < 0){
			printf("[%s]: Write failed\n", __func__);
			goto out4;
		}
		len -= size;
	}

out4:
	free(buf);
out3:
	close(fd_o);	
out2:
	close(fd_i);
out1:
	return ret;
}

int emmc_init(void)
{
	int fd, ret;
	char path[128];
	unsigned char *buf;
	struct mmc *mmc;
	char *val;
	struct stat st;

	mmc = (struct mmc *)malloc(sizeof(struct mmc));
	if(!mmc){
		printf("[%s]: malloc struct mmc failed\n", __func__);
		return 1;
	}
	
	memset(mmc, 0, sizeof(struct mmc));

	INIT_LIST_HEAD(&mmc_devices);
	INIT_LIST_HEAD(&mmc->link);
	list_add_tail(&mmc->link, &mmc_devices);

	val =  iniparser_getstring(dict, "boot:type", "Not Set");
	if(strcmp(val, "Not Set") == 0){
		printf("[%s]: boot type is not set, using default ROMBOOT\n", __func__);
	}
	else if(strcmp(val, "ROMBOOT") == 0){
		bfe = 1;
	}
	else if(strcmp(val, "SPIBOOT") == 0){
		bfe = 0;
	}
	else{
		printf("[%s]: Unknown boot type\n", __func__);
		return 0;
	}

	do_set_emmc_partition_start();

	val =  iniparser_getstring(dict, "pad:enable", "Not Set");
	if(strcmp(val, "n") == 0 || strcmp(val ,"N") == 0){
		ispad = 0;
	}
	else if(strcmp(val, "y") == 0 || strcmp(val ,"Y") == 0){
		ispad = 1;
	}
	else
		ispad = 0;
	val =  iniparser_getstring(dict, "pad:value", "Not Set");
	if(strcmp(val, "0") == 0){
		padvalue = 0;
	}
	else if(strcmp(val, "1") == 0 || strcmp(val ,"1") == 0){
		padvalue = 1;
	}
	else
		ispad = 0;

	if(ispad)
		printf("[%s]: Enable emmc bin pad, pad vaule=%d\n", __func__, padvalue);
	else
		printf("[%s]: Disable emmc bin pad\n", __func__);
	
	emmc_outpath = outpath;
		
	emmc_driver = (emmc_driver_desc_t *)malloc(sizeof(emmc_driver_desc_t));
	if(!emmc_driver){
		printf("[%s]: malloc emmc driver failed\n", __func__);
		return -1;
	}
	memset(emmc_driver, 0, sizeof(emmc_driver_desc_t));
	emmc_driver->signature = EMMC_DRIVER_MAGIC;
	emmc_driver->version = EMMC_PARTITIONTABLE_VERSION2;
	emmc_driver->drvr_cnt = EMMC_RESERVED_FOR_MAP_V2;
		
	emmc_parts = (struct emmc_bintool_part **)malloc(EMMC_RESERVED_FOR_MAP_V2 * sizeof(struct emmc_bintool_part *));
	if(!emmc_parts){
		printf("[%s]: malloc emmc parts failed\n", __func__);
		return -1;
	}
	memset(emmc_parts, 0, EMMC_RESERVED_FOR_MAP_V2 * sizeof(struct emmc_bintool_part *));

	//emmc partition table
	emmc_partition_tbl = (struct emmc_bintool_part *)malloc(sizeof(struct emmc_bintool_part));
	if(!emmc_partition_tbl){
		printf("[%s]: malloc emmc partition table  failed\n", __func__);
		return -1;
	}
	memset(emmc_partition_tbl, 0, sizeof(struct emmc_bintool_part));
	emmc_partition_tbl->emmc_part = (emmc_partition_t *)malloc(sizeof(emmc_partition_t));
	if(!emmc_partition_tbl->emmc_part){
		printf("[%s]: malloc emmc part for emmc mboot partition failed\n", __func__);
		return -1;
	}
	memset(emmc_partition_tbl->emmc_part, 0, sizeof(emmc_partition_t));
	emmc_partition_tbl->blocks = EMMC_RESERVED_FOR_MAP_V2 + 1;
	emmc_partition_tbl->emmc_part->start_block = 0;
	emmc_partition_tbl->emmc_part->block_count = EMMC_RESERVED_FOR_MAP_V2 + 1;

	if(bfe){
		//emmc mboot partition
		emmc_parts[0]= (struct emmc_bintool_part *)malloc(sizeof(struct emmc_bintool_part));
		if(!emmc_parts[0]){
			printf("[%s]: malloc emmc mboot partition descriptor  failed\n", __func__);
			return -1;
		}
		memset(emmc_parts[0], 0, sizeof(struct emmc_bintool_part));
		emmc_parts[0]->emmc_part = (emmc_partition_t *)malloc(sizeof(emmc_partition_t));
		if(!emmc_parts[0]->emmc_part){
			printf("[%s]: malloc emmc part for emmc mboot partition failed\n", __func__);
			return -1;
		}
		memset(emmc_parts[0]->emmc_part, 0, sizeof(emmc_partition_t));
		emmc_parts[0]->blocks = 0;
		emmc_parts[0]->emmc_part->start_block = EMMC_MBOOT_START;
		emmc_parts[0]->emmc_part->block_count = EMMC_MBOOT_SIZE;
		emmc_parts[0]->emmc_part->signature = EMMC_PARTITION_MAGIC;
		memcpy(emmc_parts[0]->emmc_part->name, "MBOOT", 5);
		memset(path, 0, sizeof(path));
		sprintf(path, "%s%s", emmc_outpath, EMMC_MBOOT_NAME_OUT);
		fd = open(path, O_CREAT | O_TRUNC | O_RDWR, S_IRWXU | S_IRWXG | S_IRWXO);
		if(fd < 0){
			printf("[%s]: Create %s failed\n", __func__, path);
			return -1;
		}
		close(fd);

		//emmc mpool partition
		emmc_parts[1] = (struct emmc_bintool_part *)malloc(sizeof(struct emmc_bintool_part));
		if(!emmc_parts[1]){
			printf("[%s]: malloc emmc mpool partition descriptor  failed\n", __func__);
			return -1;
		}
		memset(emmc_parts[1], 0, sizeof(struct emmc_bintool_part));
		emmc_parts[1]->emmc_part = (emmc_partition_t *)malloc(sizeof(emmc_partition_t));
		if(!emmc_parts[1]->emmc_part){
			printf("[%s]: malloc emmc part for emmc mboot partition failed\n", __func__);
			return -1;
		}
		memset(emmc_parts[1]->emmc_part, 0, sizeof(emmc_partition_t));
		emmc_parts[1]->blocks = 0;
		emmc_parts[1]->emmc_part->start_block = EMMC_MPOOL_START;
		emmc_parts[1]->emmc_part->block_count = EMMC_MPOOL_SIZE;
		emmc_parts[1]->emmc_part->signature = EMMC_PARTITION_MAGIC;
		memcpy(emmc_parts[1]->emmc_part->name, "MPOOL", 5);
		memset(path, 0, sizeof(path));
		sprintf(path, "%s%s", emmc_outpath, EMMC_MPOOL_NAME_OUT);
		fd = open(path, O_CREAT | O_TRUNC | O_RDWR, S_IRWXU | S_IRWXG | S_IRWXO);
		if(fd < 0){
			printf("[%s]: Create %s failed\n", __func__, path);
			return -1;
		}
		close(fd);
		
		emmc_parts_num = 2;
	}

	emmc_lzobuf = (unsigned char *)malloc(EMMC_LZO_BUF_SIZE);
	if(!emmc_lzobuf){
		printf("[%s]: malloc emmc lzo buffer failed\n", __func__);
		return -1;
	}

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s", emmc_outpath, EMMC_BOOT1_NAME);
	fd = open(path, O_CREAT | O_TRUNC | O_RDWR, S_IRWXU | S_IRWXG | S_IRWXO);
	if(fd < 0){
		printf("[%s]: Create %s failed\n", __func__, path);
		return -1;
	}

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s", emmc_outpath, EMMC_BOOT2_NAME);
	fd = open(path, O_CREAT | O_TRUNC | O_RDWR, S_IRWXU | S_IRWXG | S_IRWXO);
	if(fd < 0){
		printf("[%s]: Create %s failed\n", __func__, path);
		return -1;
	}

	g_eMMCDrv.u32_BOOT_SEC_COUNT = 0x1000000;
	g_eMMCDrv.u32_SEC_COUNT = 0x1000000;
	
	return 0;
}

#define LEAPHEAD "GROUP DEFINE2\0\0\0"
#define LEAPNAME "leap.def"
#define FORCREATENAME "forcreat.def"
int mmc_bininfo_leap()
{
	int fd, ret, i = 0;
	char buf[16], path[128];

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s", emmc_outpath, LEAPNAME);


	fd = open(path, O_CREAT | O_RDWR | O_TRUNC, S_IRWXU | S_IRWXG | S_IRWXO);
	if(fd < 0){
		printf("[%s]: open %s failed\n", __func__, path);
		return -1;
	}

	memcpy(buf, LEAPHEAD, 0x10);
	ret = write(fd, buf, 16);
	if(ret < 0){
		printf("[%s]: write leap head failed\n", __func__);
		return -1;
	}

	*((ulong *)buf) = __cpu_to_le32(0x00000001);
	*((ulong *)(buf+ 0x4)) = __cpu_to_le32(emmc_partition_tbl->emmc_part->start_block);
	*((ulong *)(buf + 0x8)) = __cpu_to_le32((emmc_partition_tbl->emmc_part->start_block + emmc_partition_tbl->emmc_part->block_count - 1));
	*((ulong *)(buf + 0xc)) = __cpu_to_le32(emmc_partition_tbl->blocks);

	ret = write(fd, buf, 16);
	if(ret < 0){
		printf("[%s]: write leap head failed\n", __func__);
		return -1;
	}
	
	for(i =0; i < EMMC_RESERVED_FOR_MAP_V2; i++){
		if(emmc_parts[i]){
			if(emmc_parts[i]->blocks == 0)
				continue;
			*((ulong *)buf) = __cpu_to_le32(0x00000001);
			*((ulong *)(buf+ 0x4)) = __cpu_to_le32(emmc_parts[i]->emmc_part->start_block);
			*((ulong *)(buf + 0x8)) = __cpu_to_le32(emmc_parts[i]->emmc_part->start_block + emmc_parts[i]->emmc_part->block_count - 1);
			*((ulong *)(buf + 0xc)) = __cpu_to_le32(emmc_parts[i]->blocks);
			ret =  write(fd, buf, 0x10);
			if(ret < 0){
				printf("[%s]: write part %s descriptor for def file failed\n", __func__,  emmc_parts[i]->emmc_part->name);
				return -1;
			}
		}
	}

	memset(buf, 0xFF, 0x10);
	ret = write(fd, buf, 0x10);
	if(ret < 0){
		printf("[%s]: write tail for def file failed\n", __func__);
		return -1;
	}

	close(fd);
	return 0; 
}

int mmc_bininfo_forcreat()
{
	int fd, ret, i = 0;
	unsigned int end;
	char buf[16], path[128];

	memset(path, 0, sizeof(path));
	sprintf(path, "%s%s", emmc_outpath, FORCREATENAME);

	fd = open(path, O_CREAT | O_RDWR | O_TRUNC, S_IRWXU | S_IRWXG | S_IRWXO);
	if(fd < 0){
		printf("[%s]: open %s failed\n", __func__, path);
		return -1;
	}

	memcpy(buf, LEAPHEAD, 0x10);
	ret = write(fd, buf, 16);
	if(ret < 0){
		printf("[%s]: write head failed\n", __func__);
		return -1;
	}

	end = emmc_partition_tbl->emmc_part->start_block + emmc_partition_tbl->emmc_part->block_count - 1;
	*((ulong *)buf) = __cpu_to_le32(0x00000000);
	*((ulong *)(buf+ 0x4)) = __cpu_to_le32(emmc_partition_tbl->emmc_part->start_block);
	*((ulong *)(buf + 0x8)) = __cpu_to_le32(end);
	*((ulong *)(buf + 0xc)) = __cpu_to_le32(emmc_partition_tbl->blocks);

	ret = write(fd, buf, 16);
	if(ret < 0){
		printf("[%s]: write head failed\n", __func__);
		return -1;
	}

	printf("\n\n[%s]: %s\n", __func__, emmc_partition_tbl->emmc_part->name);
	
	for(i =0; i < EMMC_RESERVED_FOR_MAP_V2; i++){
		if(emmc_parts[i]){
			printf("[%s]: %s\n", __func__, emmc_parts[i]->emmc_part->name);
			if(emmc_parts[i]->blocks == 0)
				continue;

			end = emmc_parts[i]->emmc_part->start_block + emmc_parts[i]->emmc_part->block_count -1;
			*((ulong *)buf) = __cpu_to_le32(0x00000000);
			*((ulong *)(buf+ 0x4)) = __cpu_to_le32(emmc_parts[i]->emmc_part->start_block);
			*((ulong *)(buf + 0x8)) = __cpu_to_le32(end);
			*((ulong *)(buf + 0xc)) = __cpu_to_le32(emmc_parts[i]->blocks);
			ret =  write(fd, buf, 0x10);
			if(ret < 0){
				printf("[%s]: write part %s descriptor for def file failed\n", __func__, emmc_parts[i]->emmc_part->name);
				return -1;
			}
		}
	}
	
	close(fd);
	return 0; 
}

int do_mmc_bininfo(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int ret;
	ret = mmc_bininfo_leap();
	if(ret < 0){
		printf("[%s]: generate def file for leap failed\n", __func__);
		return -1;
	}
	ret = mmc_bininfo_forcreat();
	if(ret < 0){
		printf("[%s]: generate def file for forcreat failed\n", __func__);
		return -1;
	}
	return 0;
}

int check_empty(unsigned char *buf, int len)
{
	int i;
	if(len % EMMC_BLK_SZ){
		printf("[%s]: buffer not aligned\n", __func__);
		return -1;
	}
	for(i = 0; i < len; i++){
		if(buf[i] != padvalue)
			return 0;
	}
	return 1;
}

int emmc_truncate_partitions(void)
{
	int i, ret, fd;
	char path[128];
	char buf[EMMC_BLK_SZ];
	off_t offset;
	struct stat st;
	for(i = 0; i < emmc_parts_num; i++){
		memset(path, 0, sizeof(path));
		sprintf(path, "%s%s%s", emmc_outpath, emmc_parts[i]->emmc_part->name, ".bin.tmp");
		fd = open(path, O_RDWR);
		if(fd < 0){
			printf("[%s]: open partition %s file %s failed\n", __func__, emmc_parts[i]->emmc_part->name, path);
			return -1;
		}

		stat(path, &st);
		offset = st.st_size;
		if(!offset)
			continue;

		while(offset > 0){
			lseek(fd, offset - EMMC_BLK_SZ, SEEK_SET);
			ret =read(fd, buf, EMMC_BLK_SZ);
			if(ret < 0){
				printf("[%s]: read %s failed, offset=%lld\n", __func__, path, offset);
				close(fd);
				return ret;
			}
			if(!check_empty(buf, EMMC_BLK_SZ))
				break;
			offset -= EMMC_BLK_SZ;
		}
		close(fd);
		printf("[%s]: part %s truncate with size %lld\n", __func__, emmc_parts[i]->emmc_part->name, offset);
		ret = truncate(path, offset);
		if(ret < 0){
			printf("[%s]: truncate failed, size=%lld\n", __func__, offset);
			return ret;
		}
		emmc_parts[i]->blocks = offset / EMMC_BLK_SZ;
	}
	return ret;
}

int emmc_finish_update(void)
{
	int fd_out, fd_in, i, j, ret;
	long long len, size = EMMC_PARTITION_START;
	char path_in[128], path_out[128];
	char buf[EMMC_BLK_SZ];
	struct stat st;

	ret = emmc_create_partition_tbl();
	if(ret < 0){
		printf("[%s]: Create emmc partition table failed\n", __func__);
		return -1;
	}

	if(bfe){
		ret = emmc_create_partition_mpool();
		if(ret < 0){
			printf("[%s]: Move mpool failed\n", __func__);
			return -1;
		}
	}
	
	memset(path_out, 0, sizeof(path_out));
	sprintf(path_out, "%s%s", emmc_outpath, EMMC_BIN_NAME);

	fd_out = open(path_out, O_CREAT | O_RDWR | O_TRUNC, S_IRWXU | S_IRWXG | S_IRWXO);
	if(fd_out <0){
		printf("[%s]: open emmc out file %s failed\n", __func__, path_out);
		return -1;
	}

	if(ispad){
		printf("[%s]: Create empty emmc file\n", __func__);
		memset(buf, padvalue, EMMC_BLK_SZ);
		for(i = 0; i < emmc_parts_num; i++){
			if(emmc_parts[i]){
				if(strcmp(emmc_parts[i]->emmc_part->name, "MBOOT") && strcmp(emmc_parts[i]->emmc_part->name, "mboot") &&
					strcmp(emmc_parts[i]->emmc_part->name, "MPOOL") && strcmp(emmc_parts[i]->emmc_part->name, "mpool"))
					size += emmc_parts[i]->emmc_part->block_count;
			}
		}
		printf("[%s]: total blocks: %ld\n", __func__, size);
		for(i = 0; i < size; i++){
			ret = write(fd_out, buf, EMMC_BLK_SZ);
			if(ret < 0){
				printf("[%s]: write failed 1\n", __func__);
				return -1;
			}
		}
		lseek(fd_out, 0, SEEK_SET);
	}
	
	ret = emmc_truncate_partitions();
	if(ret < 0){
		printf("[%s]: truncate partitons failed\n", __func__);
		return ret;
	}

	memset(path_in, 0, sizeof(path_in));
	sprintf(path_in, "%s%s", emmc_outpath, EMMC_PARTITION_NAME);
	fd_in = open(path_in, O_RDWR);
	if(fd_in < 0){
		printf("[%s]: open partition table file %s failed\n", __func__, path_in);
		return -1;		
	}

	stat(path_in, &st);

	len = st.st_size;
	while(len > 0){
		ret = read(fd_in, buf, EMMC_BLK_SZ);
		if(ret < 0){
			printf("[%s]: read file %s failed 1\n", __func__, path_in);
			return -1;
		}
		ret = write(fd_out, buf, EMMC_BLK_SZ);
		if(ret < 0){
			printf("[%s]: write file %s failed 1\n", __func__, path_out);
			return -1;
		}	
		len -= EMMC_BLK_SZ;
	}
	close(fd_in);
//	unlink(path_in);

	printf("%s     %s     %s     %s     %s\n", __func__, "Name", "Start block", "Block cnt", "Write cnt");
	for(i = 0; i < emmc_parts_num; i++){
		printf("[%s]: %s     %x     %x     %x\n", __func__, 
			emmc_parts[i]->emmc_part->name, 
			emmc_parts[i]->emmc_part->start_block, 
			emmc_parts[i]->emmc_part->block_count,
			emmc_parts[i]->blocks);

		if(strcmp(emmc_parts[i]->emmc_part->name, "MBOOT") == 0 || 
			strcmp(emmc_parts[i]->emmc_part->name, "mboot") == 0 ||
			strcmp(emmc_parts[i]->emmc_part->name, "MPOOL") == 0 ||
			strcmp(emmc_parts[i]->emmc_part->name, "mpool") == 0 ){
			if(bfe && emmc_parts[i]->blocks == 0){
				printf("[%s]: The MBOOT or MPOOL partition is empty, please check updata package\n", __func__);
				return -1;
			}
		}

		if(ispad){
			lseek(fd_out, emmc_parts[i]->emmc_part->start_block * EMMC_BLK_SZ, SEEK_SET);
		}
		
		memset(path_in, 0, sizeof(path_in));
		sprintf(path_in, "%s%s%s", emmc_outpath, emmc_parts[i]->emmc_part->name, ".bin.tmp");
		fd_in = open(path_in, O_RDWR);
		if(fd_in < 0){
			printf("[%s]: open partition %s file %s failed\n", __func__, emmc_parts[i]->emmc_part->name, path_in);
			return -1;
		}
	
		stat(path_in, &st);

		len = st.st_size;
		while(len > 0){
			ret = read(fd_in, buf, EMMC_BLK_SZ);
			if(ret < 0){
				printf("[%s]: read file %s failed 2\n", __func__, path_in);
				return -1;
			}
			ret = write(fd_out, buf, EMMC_BLK_SZ);
			if(ret < 0){
				printf("[%s]: write file %s failed 2\n", __func__, path_out);
				return -1;
			}	
			len -= EMMC_BLK_SZ;
		}
		close(fd_in);
//		unlink(path_in);
	}

	if(!ispad){
		ret = run_command("mmcbininfo", 0);
		if(ret < 0){
			printf("[%s]: generate emmc def file failed\n", __func__);
			return -1;
		}
	}
	else{
		char defname[128];
		memset(defname, 0, sizeof(defname));
		sprintf(defname, "%s%s", emmc_outpath, LEAPNAME);
		unlink(defname);

		memset(defname, 0, sizeof(defname));
		sprintf(defname, "%s%s", emmc_outpath, FORCREATENAME);
		unlink(defname);
	}
	return 0;
}
